<?php

namespace apexl\Config;

use apexl\Io\includes\Utils;
use apexl\Utils\Arrays\Merge;
use RecursiveArrayIterator;
use RecursiveIteratorIterator;

/**
 * Class Config
 * A class to provide a single persisting store for all loaded configurations. Allows the overriding, temporary loading or locking of config
 * @package apexl\Config
 */
class Configuration
{
    /** List of loaded files, and associated options. */

    public static ?string $cacheDir = null;
    public static string $cacheFilename = 'configCache.json';
    /** @var ?Configuration */
    private static ?Configuration $instance = null;
    private array $data = [];
    private array $loadedFiles = [];
    private array $flatData = [];

    /**
     * config constructor. Set private to prevent the class being instantiated
     */
    private function __construct()
    {
    }

    public static function reset(): void
    {
        self::$instance = null;
        self::$cacheDir = null;
        self::$cacheFilename = 'configCache.json';
    }

    /**
     * @param $file
     * @param string $key
     * @param bool $loadFromCache
     * @param array $options
     * @return Singleton|bool
     * @throws \Exception
     */
    public static function load($file, string $key = '', bool $loadFromCache = true, array $options = []): \apexl\Config\Configuration|bool
    {
        self::getInstance();
        if (!is_file($file)) {
            trigger_error('Cannot load the provided configuration file: ' . $file);
            return false;
        }

        //load from an amalgamated config file. If one doesn't exist, create it.
        if ($loadFromCache) {
            $loaded = self::$instance->loadFromCache($file);
        } else {
            $loaded = self::$instance->loadFile($file, $key, $options, false);
        }
        //update our flat array
        self::$instance->setFlatArray();
        return $loaded === false ? false : self::$instance;
    }

    /**
     * Function to get the instance of this config. Singleton.
     * @return Configuration|Singleton|null
     * @throws \Exception
     */
    public static function getInstance(): Singleton|Configuration|null
    {
        if (!self::$instance) {
            self::$instance = new Configuration();
        }
        return self::$instance;
    }

    /**
     * @param $file
     * @param string $key
     * @param array $options
     * @return Configuration|null
     */
    private function loadFromCache($file, string $key = '', array $options = []): null|static
    {
        if ($this->cacheable()) {
            $success = true;
            if (!file_exists($this::$cacheDir . '/' . $this::$cacheFilename)) {
                $success = $this->writeConfigDataToCacheFile();
            }
            if ($success !== false) {
                return $this->loadFile($this::$cacheDir . '/' . $this::$cacheFilename, $key, $options, false);
            }
        }
        return $this->loadFile($file, $key, $options, false);
    }

    private function cacheable(): bool
    {
        return (bool)$this::$cacheDir;
    }

    /**
     * Function to take the current loaded config data and write it back to file
     */
    public function writeConfigDataToCacheFile(): false|int
    {
        $json = json_encode($this->data, JSON_PRETTY_PRINT);
        if (!is_dir(self::$cacheDir)) {
            if (mkdir(self::$cacheDir, 0755, true)) {
                return file_put_contents(trim((string)self::$cacheDir, '/') . '/' . self::$cacheFilename, $json);
            };
        } else {
            return file_put_contents(trim((string)self::$cacheDir, '/') . '/' . self::$cacheFilename, $json);
        }
        return false;
    }

    /**
     * @param $file
     * @param $key
     * @param $options
     * @param $temp
     * @return bool|Configuration
     */
    private function loadFile($file, $key, $options, $temp): bool|Configuration
    {
        $extension = strtolower(pathinfo((string)$file, PATHINFO_EXTENSION));

        $configData = $this->convertFileToArray($file, $extension);
        if (empty($configData)) {
            trigger_error('The provided configuration file is empty or is not readable: ' . $file);
            return false;
        }

        if ($temp) {
            if (empty($key)) {
                return $configData;
            } else {
                return $configData[$key] ?? false;
            }
        }

        //no defined Key? assume all root elements are keys.
        if (empty($key)) {
            foreach ($configData as $key => $keyData) {
                $this->storeConfigData($key, $keyData);
            }
        } else {
            $this->storeConfigData($key, $configData);
        }

        //mark the file as loaded;
        $this->loadedFiles[$key][] = ['file' => $file, 'options' => $this->setFileOptions($options)];
        //only store the data in this instance if we're not temp loading.
        return $this;
    }

    /**
     * Function to convert the provided configuration to an array of data depending on file type.
     * @param $loadFile
     * @param $loadExtension
     * @return array|mixed
     */
    private function convertFileToArray($loadFile, $loadExtension): mixed
    {
        switch ($loadExtension) {
            case 'php':
                //if this is a php file, we need to require it. Assume these files return config arrays (see Laravel)
                //@todo add options here.
                $fileData = require $loadFile;
                if(is_null($fileData)) {
                    $fileData = [];
                }
                //for php we want to return the filename as the config key.
                $file = basename($loadFile, ".php");
                return [$file => $fileData];
            case 'json':
            default:
                $fileData = file_get_contents($loadFile);
                return json_decode($fileData, true);
        }
    }

    /**
     * @param $key
     * @param $configData
     * @return void
     */
    public function storeConfigData($key, $configData): void
    {
        $exists = false;

        if (isset($this->loadedFiles[$key])) {
            $exists = true;
            //ok, so the file is already loaded, no point continuing unless we allow config to be overridden
            if ($this->loadedFiles[$key]) {
                foreach ($this->loadedFiles[$key] as $loadedFile) {
                    if (isset($loadedFile['options']['lock']) && $loadedFile['options']['lock'] === true) {
                        return;
                    }
                }
            }
        }

        //set the data if new, merge the key if it exists and is not locked.
        $this->data[$key] = $exists ? array_replace_recursive($this->data[$key], $configData) : $configData;
    }

    /**
     * Set loaded configuration file options.
     * @param $options
     * @return array
     */
    private function setFileOptions($options): array
    {
        $default = [
            'lock' => true, //Prevent a file key from being overwritten once its loaded.
        ];
        if (!empty($options)) {
            $default = array_merge($default, $options);
        }
        return $default;
    }

    /**
     * @param $dir
     */
    public static function setCacheDirectory($dir): void
    {
        self::$cacheDir = $dir;
    }

    /**
     * @param $dir
     */
    public static function setCacheFilename($dir): void
    {
        self::$cacheDir = $dir;
    }

    /**
     * @return ?string
     */
    public static function getCacheDirectory(): ?string
    {
        return self::$cacheDir;
    }

    /**
     * @param $key
     * @param array $config
     * @return bool
     * @throws \Exception
     */
    public static function updateConfigFileByKey($key, array $config): bool
    {
        //get the last loaded config file for this key and update the values to ensure we're using them
        //@todo change this to update the last file that contains these values, or the first one if none do.
        if (!isset(self::$instance->loadedFiles[$key])) {
            $files = array_values(self::$instance->loadedFiles);
            $file = array_shift($files)[0]['file'];
        } else {
            $file = end(self::$instance->loadedFiles[$key])['file'];
        }
        //temp load the file and then merge recursive if the key exists.
        $tempLoad = self::tempLoad($file);
        $tempLoad[$key] = isset($tempLoad[$key]) ? Merge::arrayMergeRecursive([$tempLoad[$key], $config]) : $config;

        $write = file_put_contents($file, json_encode($tempLoad));
        if ($write !== false) {
            self::$instance->data[$key] = $tempLoad[$key];
            //if we're cacheable, we need to write the cache out again.
            if (self::$instance->cacheable()) {
                self::$instance->writeConfigDataToCacheFile();
            }
        }
        return $write !== false; //just to make sure any checks are boolean safe.
    }

    /**
     * Function to temporarily load the provided configuration file, do not store the result in the singleton.
     * @param $file
     * @param string $key
     * @return bool|static
     * @throws \Exception
     */
    public static function tempLoad($file, string $key = ''): Configuration|bool|static
    {
        self::getInstance();
        return self::$instance->loadFile($file, $key, [], true);
    }

    /**
     * dot notation getter. (app.database.user will return ->app->database->user)
     * @param $key
     * @return mixed|null
     */
    public static function get($key){
        return self::$instance->flatData[$key] ?? null;
    }

    public static function asArray($data)
    {
        return json_decode(json_encode($data), true);
    }

    public function reloadConfig(): void
    {
        //cache the config files
        $cachedFiles = self::$instance->loadedFiles;
        //reset the loaded config
        self::$instance->loadedFiles = [];
        //loop over the file cache and reload the files.
        foreach ($cachedFiles as $key => $files) {
            foreach ($files as $file) {
                self::$instance->loadFile($file['file'], '', $files['options'] ?? [], false);
            }
        }
    }

    public static function getData(): array
    {
        return self::$instance->data;
    }

    public static function getLoadedFiles(): array
    {
        return self::$instance->loadedFiles;
    }

    private function toFlatArray(array $data): array
    {
        $iterator = new RecursiveIteratorIterator(new RecursiveArrayIterator($data));
        $result = [];
        foreach ($iterator as $leafValue) {
            $keys = [];
            foreach (range(0, $iterator->getDepth()) as $depth) {
                $keys[] = $iterator->getSubIterator($depth)->key();
            }
            $result[join('.', $keys)] = $leafValue;
        }

        return $result;
    }

    private function setFlatArray(): void
    {
        $this->flatData = $this->toFlatArray($this->data);
    }

    /**
     * Magic PHP __get function to get the requested configuration options by key. Response cast to object on return
     * @param $name
     * @return bool|mixed
     */
    public function __get($name)
    {
        return isset($this->data[$name]) ? json_decode(json_encode($this->data[$name]), false) : false;
    }

    /**
     * @param $name
     * @return bool
     */
    public function __isset($name)
    {
        return isset($this->data[$name]);
    }

    /**
     * @param $name
     */
    public function __unset($name)
    {
        unset($this->data[$name]);
    }

    /**
     *
     */
    public function __destruct()
    {
        $this->data = [];
        $this->loadedFiles = [];
    }
}