<?php

namespace apexl\Config;

use apexl\Config\Exceptions\InvalidConfigException;
use apexl\Io\exceptions\FileSystemException;
use RecursiveArrayIterator;
use RecursiveIteratorIterator;

/**
 * Class Config
 * A class to provide a single persisting store for all loaded configurations. Allows the overriding, temporary loading or locking of config
 * @package apexl\Config
 */
final class Configuration
{
    /** List of loaded files, and associated options. */

    private(set) array $data = [];
    private array $loadedFiles = [];
    private array $flatData = [];

    public function __construct(
        private readonly ?string $cacheDir = null,
        private readonly string $cacheFilename = 'configCache.json',
    ) {}

    /**
     * @throws InvalidConfigException
     */
    public function load($file, string $key = '', bool $loadFromCache = true, array $options = []): void
    {
        if (!is_file($file)) {
            throw new InvalidConfigException(sprintf('Cannot load the provided configuration file: %s', $file));
        }

        //load from an amalgamated config file. If one doesn't exist, create it.
        if ($loadFromCache && $this->cacheable()) {
            try {
                $this->loadFromCache($file);
            } catch (FileSystemException) {
                $this->loadFile($file, $key, $options);
            }
        } else {
            $this->loadFile($file, $key, $options);
        }
        //update our flat array
        $this->setFlatArray();
    }

    private function cacheable(): bool
    {
        return (bool) $this->cacheDir;
    }

    /**
     * @throws FileSystemException
     * @throws InvalidConfigException
     */
    private function loadFromCache(string $key = '', array $options = []): void
    {
        if ($this->cacheable()) {
            if (!file_exists($this->cacheFilePath())) {
                $this->writeConfigDataToCacheFile();
                $this->loadFile($this->cacheFilePath(), $key, $options, false);
            }
        }
    }

    private function cacheFilePath(): ?string
    {
        if ($this->cacheable()) {
            return sprintf('%s/%s', trim((string) $this->cacheDir, '/'), $this->cacheFilename);
        }

        return null;
    }

    /**
     * Function to take the current loaded config data and write it back to file
     * @throws FileSystemException
     */
    private function writeConfigDataToCacheFile(): void
    {
        $json = json_encode($this->data, JSON_PRETTY_PRINT);
        $this->ensureCacheDir();

        file_put_contents($this->cacheFilePath(), $json);
    }

    /**
     * @throws FileSystemException
     */
    private function ensureCacheDir(): void
    {
        if (is_dir($this->cacheDir)) {
            return;
        }

        if (!mkdir($this->cacheDir, 0755, true)) {
            throw new FileSystemException(sprintf('Cannot create cache directory: %s', $this->cacheDir));
        }
    }

    /**
     * @throws InvalidConfigException
     */
    private function loadFile(string $file, string $key, array $options): void
    {
        $dataArray = $this->convertFileToArray($file);

        if (!$dataArray) {
            throw new InvalidConfigException(
                sprintf('The provided configuration file is empty or is not readable: %s', $file)
            );
        }

        //no defined Key? assume all root elements are keys.
        if (empty($key)) {
            foreach ($dataArray as $key => $keyData) {
                $this->storeConfigData($key, $keyData);
            }
        } else {
            $this->storeConfigData($key, $dataArray);
        }

        //mark the file as loaded;
        $this->loadedFiles[$key][] = ['file' => $file, 'options' => $this->setFileOptions($options)];
        //only store the data in this instance if we're not temp loading.
    }

    /**
     * Function to convert the provided configuration to an array of data depending on file type.
     */
    private function convertFileToArray(string $file): array
    {
        $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));

        switch ($extension) {
            case 'php':
                //if this is a php file, we need to require it. Assume these files return config arrays (see Laravel)
                //@todo add options here.
                $fileData = require $file;
                if (is_null($fileData)) {
                    $fileData = [];
                }
                //for php we want to return the filename as the config key.
                $file = basename($file, ".php");
                return [$file => $fileData];
            case 'json':
            default:
                $fileData = file_get_contents($file);
                return json_decode($fileData, true);
        }
    }

    /**
     * @param $key
     * @param $configData
     * @return void
     */
    public function storeConfigData($key, $configData): void
    {
        $exists = false;

        if (isset($this->loadedFiles[$key])) {
            $exists = true;
            //ok, so the file is already loaded, no point continuing unless we allow config to be overridden
            if ($this->loadedFiles[$key]) {
                foreach ($this->loadedFiles[$key] as $loadedFile) {
                    if (isset($loadedFile['options']['lock']) && $loadedFile['options']['lock'] === true) {
                        return;
                    }
                }
            }
        }

        //set the data if new, merge the key if it exists and is not locked.
        $this->data[$key] = $exists ? array_replace_recursive($this->data[$key], $configData) : $configData;
    }

    private function setFileOptions(array $options): array
    {
        return [
            'lock' => true, //Prevent a file key from being overwritten once its loaded.
            ...$options,
        ];
    }

    private function setFlatArray(): void
    {
        $this->flatData = $this->toFlatArray($this->data);
    }

    private function toFlatArray(array $data): array
    {
        $iterator = new RecursiveIteratorIterator(new RecursiveArrayIterator($data));
        $result = [];
        foreach ($iterator as $leafValue) {
            $keys = [];
            foreach (range(0, $iterator->getDepth()) as $depth) {
                $keys[] = $iterator->getSubIterator($depth)->key();
                //foreach iteration, add the keys and value.
                $result[join('.', $keys)] = $this->getArrayValueByIteratorPosition($keys, $data);
            }
        }

        return $result;
    }

    private function getArrayValueByIteratorPosition(array $keys, array $data): mixed
    {
        $value = $data;
        foreach (range(0, (count($keys)) - 1) as $index) {
            $value = $value[$keys[$index]] ?? null;
        }
        return $value;
    }

    /**
     * dot notation getter. (app.database.user will return ->app->database->user)
     */
    public function get(string $key): mixed
    {
        return $this->flatData[$key] ?? null;
    }

    public function asArray($data): array
    {
        return json_decode(json_encode($data), true);
    }
}