<?php

declare(strict_types=1);

namespace apexl\dto;

use JsonSerializable;
use ReflectionObject;

abstract readonly class Dto implements JsonSerializable
{
    public function with(string|array $key, mixed $value = null): static
    {
        $props = $this->toArray(true);

        if (is_string($key)) {
            $key = [$key => $value];
        }

        foreach ($key as $_key => $value) {
            $props[$_key] = $value;
        }

        return new static(...$props);
    }

    public function toArray(bool $withProtected = false): array
    {
        $props = $this->getProps();

        if ($withProtected) {
            return $props;
        }

        return array_filter(
            $props,
            fn(string $key): bool => !in_array($key, $this->protect(), true),
            ARRAY_FILTER_USE_KEY
        );
    }

    private function getProps(): array
    {
        $props = [];

        foreach (new ReflectionObject($this)->getProperties() as $property) {
            $props[$property->getName()] = $property->getValue($this);
        }

        return $props;
    }

    protected function protect(): array
    {
        return [];
    }

    public function without(string|array $key): static
    {
        $props = array_diff_key($this->toArray(true), array_flip((array) $key));

        return new static(...$props);
    }

    public function jsonSerialize(): array
    {
        return $this->toArray();
    }
}
