<?php

namespace apexl\entityCore;

use apexl\Config\Singleton;
use apexl\entityCore\interfaces\EntityInterface;
use apexl\entityCore\interfaces\EntityOperatorInterface;

abstract class Entity implements EntityInterface
{
    protected $entityChanged = false; /** @var bool Flag the entity as changed. Allows us to only update when something has been changed - should reduce overhead*/
    protected $data = [];
    protected $type = '';
    protected $operator;
    protected $requiredData = [];
    protected $missingData = [];
    protected $config;
    protected $sensitiveDataFields = [];

    /**
     * entityBase constructor.
     * @param EntityOperatorInterface $operator
     * @throws \Exception
     */
    public function __construct()
    {
        //first lets see what we can get from config.
        //reference the config singleton
        $this->config = Singleton::getInstance();
        $this->initiateEntity();
    }

    /**
     * Method to set the current operator instance.
     * @param EntityOperatorInterface $operator
     * @return $this
     */
    protected function initiateEntity()
    {
        if (isset($this->config->baseEntity->operator)) {
            $operator = $this->config->baseEntity->operator;
            $this->operator = $this->setOperator(new $operator($this->config->baseEntity));
        }
        return $this;
    }

    /**
     * Method to force any provided operators to implement the EntityOperatorInterface interface.
     * @param EntityOperatorInterface $operator
     * @return $this
     */
    public function setOperator(EntityOperatorInterface $operator)
    {
        $entityClass = get_called_class();
        $entityName = substr($entityClass, strrpos($entityClass, '\\') + 1);
        $entityConfigProperty = str_ireplace("entity", "", $entityName);
        if (isset($this->config->app->accessControl->$entityConfigProperty)) { // Set accessControlType for use in Hooks
            $operator->setAccessControlType($this->config->app->accessControl->$entityConfigProperty);
        }
        $this->operator = $operator;
        return $this;
    }

    /**
     * Method to allow us to set certain data as 'sensitive'. This prevents the data being returned in the getData() method, but allows it to still be accessible directly.
     * @param $field
     * @return $this
     */
    public function setSensitiveData($field, $replace = false)
    {
        if (is_array($field)) {
            $this->sensitiveDataFields = $replace ? $field : array_merge($this->sensitiveDataFields, $field);
        } else {
            $this->sensitiveDataFields[] = $field;
        }
        return $this;
    }

    /**
     * Method to get the names of the fields marked sensitive.
     * @return array
     */
    public function getSensitiveDataNames()
    {
        return $this->sensitiveDataFields;
    }

    /**
     * Function to get the full Data array for the entity
     * @return mixed|array
     */
    public function getData()
    {
        $data = $this->data;
        if (!empty($this->sensitiveDataFields)) {
            foreach ($this->sensitiveDataFields as $field) {
                unset($data[$field]);
            }
        }
        return $data;
    }

    /**
     * Function to set the entity data array. Useful when checking against loaded entities or for a direct overwrite.
     * @param $entity
     * @return $this
     */
    public function setData($entity)
    {
        //loop over and clean utf8 data from the entity. We use this not array_map because utf8_encode converts NULL to blank string.
        $utf8Safe = [];
        if (!empty($entity)) {
            foreach ($entity as $key => $value) {
                $utf8Safe[$key] = !is_null($value) && is_string($value) ? utf8_encode($value) : $value;
            }
        }
        $this->data = empty($entity) ? array() : json_decode(json_encode($utf8Safe), true);
        return $this;
    }

    /**
     * Setter for individual entity field
     * @param string $field
     * @param mixed $value
     * @return $this
     */
    public function setField($field, $value)
    {
        $this->entityChanged = true;
        $this->data[$field] = $value;
        return $this;
    }

    /**
     * Getter for individual entity field
     * @param $field
     * @return bool|mixed
     */
    public function getField($field)
    {
        return isset($this->data[$field]) ? $this->data[$field] : null;
    }

    /**
     * Getter for entity changed state.
     * @return string
     */
    public function isChanged()
    {
        return $this->entityChanged;
    }

    /**
     * Allow us toi reset the change state of an entity (on load / save for example)
     */
    public function resetChanged()
    {
        $this->entityChanged = false;
    }

    /**
     * Magic PHP __get function to get the required field from the data object, without having to specify it to the entity.
     * @param $name
     * @return bool|mixed
     */
    public function __get($name)
    {
        return $this->getField($name);
    }

    /**
     * Magic PHP __set function to set provided field value
     * @param $name
     * @param $value
     * @return entity
     */
    public function __set($name, $value)
    {
        return $this->setField($name, $value);
    }

    /**
     * Magic PHP __call function. Used for all operations. the use of `...` allows us to pass as many args as the callable method expects.
     * @param $name
     * @param $arguments
     * @return mixed
     * @throws \Exception
     */
    public function __call($name, $arguments)
    {
        $response = $this->operator->$name(...$arguments);
        //account for the potential for different formatted responses
        if (is_array($response)) {
            if (isset($response['updateEntityData']) && $response['updateEntityData'] === true) {
                $this->setData($response['data']);
            }
        } elseif (is_object($response)) {
            if (isset($response->updateEntityData) && $response->updateEntityData === true) {
                $this->setData($response->data);
            }
        }
        return $response;
    }

    /**
     * Allow entities to have defined required fields.
     * @param $fields
     */
    public function setRequiredData($fields)
    {
        $this->requiredData = $fields;
    }

    /**
     * Allow the addition of single fields to the required array
     * @param $field
     */
    public function addRequiredData($field)
    {
        $this->requiredData[] = $field;
    }

    /**
     * If validation fails, allow us to get the data that was missing.
     * @return array
     */
    public function getMissingData()
    {
        return $this->missingData;
    }

    /**
     * Function to check if the entity is valid. I.E. it has all of its required fields.
     * @return bool
     */
    public function isValid()
    {
        $valid = true;
        $this->missingData = []; //clear this data at the start of a validation check
        if (!empty($this->requiredData)) {
            foreach ($this->requiredData as $field) {
                if (isset($this->data[$field]) && !empty($this->data[$field])) {
                    $valid = true;
                } else {
                    $valid = false;
                    $this->missingData[] = $field;
                }
            }
        }
        return $valid;
    }

    /**
     * function to determine the entity Type used in request URL generation based in the current class type.
     * @return mixed
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * Function to convert the entity to a jsonString
     * @return false|string
     */
    public function __toString()
    {
        return json_encode($this->data);
    }

    /**
     * Magic __isset function to allow us to use isset on $data values retrieved by __get
     * @param $name
     * @return bool
     */
    public function __isset($name)
    {
        return isset($this->data[$name]);
    }

    /**
     * Magic __unset function to allow us to unset $data values.
     * @param $name
     */
    public function __unset($name)
    {
        if (isset($this->data[$name])) {
            unset($this->data[$name]);
            $this->entityChanged = true;
        }
    }
}
