<?php

declare(strict_types=1);

namespace apexl\entityCore\traits;

use apexl\entityCore\enums\Casts;
use DateTimeImmutable;
use DateTimeInterface;
use Exception;

trait hasCasts
{
    public function __get($name)
    {
        return $this->getCasted($name);
    }

    public function __set($name, $value): void
    {
        $this->setCasted($name, $value);
    }

    /** @noinspection PhpUnused */

    protected function getCasted($name)
    {
        $value = parent::__get($name);

        if ($value !== null && $cast = $this->getCast($name)) {
            switch ($cast) {
                case Casts::DATETIME:
                case Casts::DATETIME_STAMP:
                    if ($cast === Casts::DATETIME_STAMP) {
                        $value = sprintf('@%d', $value);
                    }
                    try {
                        return new DateTimeImmutable($value);
                    } catch (Exception $e) {
                        user_error(
                            sprintf('Error parsing DateTime: %s', $e->getMessage()),
                            E_USER_WARNING
                        );

                        return null;
                    }
                case Casts::SERIALIZE:
                    return unserialize($value);
                default:
                    settype($value, $cast->getType());
            }
        }

        return $value;
    }

    private function getCast(string $name): ?Casts
    {
        $cast = $this->fieldConfig()[$name]['cast'] ?? null;

        return $cast instanceof Casts ? $cast : null;
    }

    protected function fieldConfig(): array
    {
        return $this->withCasts();
    }

    protected function withCasts(array $fieldConfig = []): array
    {
        return array_merge_recursive(
            $fieldConfig,
            array_map(
                fn(Casts $cast) => ['cast' => $cast],
                $this->casts()
            )
        );
    }

    /**
     * @return Casts[]
     */
    abstract public function casts(): array;

    protected function setCasted($name, $value): void
    {
        if ($value !== null && $cast = $this->getCast($name)) {
            switch ($cast) {
                case Casts::DATETIME:
                    if ($value instanceof DateTimeInterface) {
                        $value = $value->format('Y-m-d H:i:s');
                    }
                    break;
                case Casts::DATETIME_STAMP:
                    if ($value instanceof DateTimeInterface) {
                        $value = $value->getTimestamp();
                    }
                    break;
                case Casts::SERIALIZE:
                    $value = serialize($value);
                    break;
                default:
                    break;
            }
        }

        parent::__set($name, $value);
    }
}
