<?php

declare(strict_types=1);

namespace apexl\entityCore\traits;

use apexl\entityCore\enums\Casts;
use BackedEnum;
use DateTimeImmutable;
use DateTimeInterface;
use Exception;

trait hasCasts
{
    private const FIELD__CAST = 'cast';

    public function __get($name)
    {
        return $this->getCasted($name);
    }

    public function __set($name, $value): void
    {
        $this->setCasted($name, $value);
    }

    /** @noinspection PhpUnused */

    protected function getCasted($name)
    {
        /** @noinspection PhpMultipleClassDeclarationsInspection */
        $value = parent::getField($name);

        if ($value !== null && $cast = $this->getCast($name)) {
            if (is_string($cast)) {
                /** @var BackedEnum $cast */
                return $cast::from($value);
            }

            switch ($cast) {
                case Casts::DATE:
                case Casts::DATETIME:
                case Casts::DATETIME_STAMP:
                    if ($cast === Casts::DATETIME_STAMP) {
                        $value = sprintf('@%d', $value);
                    }
                    try {
                        return new DateTimeImmutable($value);
                    } catch (Exception $e) {
                        user_error(
                            sprintf('Error parsing DateTime: %s', $e->getMessage()),
                            E_USER_WARNING
                        );

                        return null;
                    }
                case Casts::SERIALIZE:
                    return unserialize($value);
                default:
                    settype($value, $cast->getType());
            }
        }

        return $value;
    }

    private function getCast(string $name): Casts|string|null
    {
        static $fieldConfig = null;
        if (!$fieldConfig) {
            $fieldConfig = $this->fieldConfig();
        }

        $cast = $fieldConfig[$name]['cast'] ?? null;

        return ($cast instanceof Casts || is_a($cast, BackedEnum::class, true)) ? $cast : null;
    }

    protected function fieldConfig(): array
    {
        return $this->withCasts();
    }

    protected function withCasts(array $fieldConfig = []): array
    {
        return array_merge_recursive(
            $fieldConfig,
            array_map(
                fn(string|Casts $cast) => ['cast' => $cast],
                $this->casts()
            )
        );
    }

    /**
     * @return (Casts|string)[]
     */
    public function casts(): array
    {
        $casts = [];
        foreach ($this->entityFields() as $field) {
            $casts[$field->field] = $field->type;
        };

        return $casts;
    }

    protected function setCasted($name, $value): void
    {
        if ($value !== null && $cast = $this->getCast($name)) {
            if (is_string($cast) && $value instanceof $cast) {
                $value = $value->value;
            }

            switch ($cast) {
                case Casts::DATE:
                    if ($value instanceof DateTimeInterface) {
                        $value = $value->format('Y-m-d');
                    }
                    break;
                case Casts::DATETIME:
                    if ($value instanceof DateTimeInterface) {
                        $value = $value->format('Y-m-d H:i:s');
                    }
                    break;
                case Casts::DATETIME_STAMP:
                    if ($value instanceof DateTimeInterface) {
                        $value = $value->getTimestamp();
                    }
                    break;
                case Casts::SERIALIZE:
                    $value = serialize($value);
                    break;
                default:
                    break;
            }
        }

        /** @noinspection PhpMultipleClassDeclarationsInspection */
        parent::setField($name, $value);
    }
}
