<?php

declare(strict_types=1);

namespace apexl\entityCore\traits;

use apexl\entityCore\enums\Casts;
use apexl\Io\includes\Entity;
use apexl\Io\modules\user\entities\userEntity;
use BackedEnum;
use DateTimeImmutable;
use DateTimeInterface;
use Exception;
use UnitEnum;

trait hasCasts
{
    private const FIELD__CAST = 'cast';

    public function __get($name)
    {
        return $this->getCasted($name);
    }

    public function __set($name, $value): void
    {
        $this->setCasted($name, $value);
    }

    protected function getCasted($name)
    {
        /** @noinspection PhpMultipleClassDeclarationsInspection */
        $value = parent::getField($name);

        if ($value !== null && $cast = $this->getCast($name)) {
            if ($cast instanceof Entity) {
                $key = $cast->getPrimaryKey();
                $cast->load($value);

                return $cast->{$key} ? $cast : null;
            }

            if (is_a($cast, BackedEnum::class, true)) {
                /** @var BackedEnum $cast */
                return $cast::from($value);
            }

            if (is_string($cast) && is_a($cast, UnitEnum::class, true)) {
                /** @var string $cast */
                return constant(sprintf('%s::%s', $cast, $value));
            }

            switch ($cast) {
                case Casts::DATE:
                case Casts::DATETIME:
                case Casts::DATETIME_STAMP:
                    if ($cast === Casts::DATETIME_STAMP) {
                        $value = sprintf('@%d', $value);
                    }
                    try {
                        return new DateTimeImmutable($value);
                    } catch (Exception $e) {
                        user_error(
                            sprintf('Error parsing DateTime: %s', $e->getMessage()),
                            E_USER_WARNING
                        );

                        return null;
                    }
                case Casts::SERIALIZE:
                    return unserialize(base64_decode($value));
                default:
                    settype($value, $cast->getType());
            }
        }

        return $value;
    }

    /** @noinspection PhpUnused */

    /**
     * @return Casts|Entity|class-string|null
     */
    private function getCast(string $name): Casts|Entity|string|null
    {
        static $fieldConfig = null;
        if (!$fieldConfig) {
            $fieldConfig = $this->fieldConfig();
        }

        $cast = $fieldConfig[$name]['cast'] ?? null;

        if (is_string($cast)) {
            if (is_a($cast, BackedEnum::class, true)) {
                return $cast;
            } elseif (is_a($cast, Entity::class, true)) {
                return new $cast();
            }
        }

        return $cast;
    }

    protected function fieldConfig(): array
    {
        return $this->withCasts();
    }

    protected function withCasts(array $fieldConfig = []): array
    {
        return array_merge_recursive(
            $fieldConfig,
            array_map(
                fn(string|Casts $cast) => ['cast' => $cast],
                $this->casts()
            )
        );
    }

    /**
     * @return (Casts|string)[]
     */
    public function casts(): array
    {
        $casts = [];

        foreach ($this->entityFields() as $field) {
            $casts[$field->field] = $field->type;
        };

        return [
            ...$casts,
            'created' => Casts::DATETIME_STAMP,
            'modified' => Casts::DATETIME_STAMP,
            'created_by' => userEntity::class,
            'modified_by' => userEntity::class,
        ];
    }

    protected function setCasted($name, $value): void
    {
        if ($value !== null && $cast = $this->getCast($name)) {
            if ($cast instanceof Entity && $value instanceof $cast) {
                $key = $cast->getPrimaryKey();
                $value = $cast->{$key};
            } elseif (is_a($cast, UnitEnum::class, true) && $value instanceof $cast) {
                $value = $value instanceof BackedEnum ? $value->value : $value->name;
            } else {
                switch ($cast) {
                    case Casts::DATE:
                        if ($value instanceof DateTimeInterface) {
                            $value = $value->format('Y-m-d');
                        }
                        break;
                    case Casts::DATETIME:
                        if ($value instanceof DateTimeInterface) {
                            $value = $value->format('Y-m-d H:i:s');
                        }
                        break;
                    case Casts::DATETIME_STAMP:
                        if ($value instanceof DateTimeInterface) {
                            $value = $value->getTimestamp();
                        }
                        break;
                    case Casts::SERIALIZE:
                        $value = base64_encode(serialize($value));
                        break;
                    default:
                        break;
                }
            }
        }

        /** @noinspection PhpMultipleClassDeclarationsInspection */
        parent::setField($name, $value);
    }
}
