<?php

namespace apexl\Io\modules\gohaul\controllers;

use apexl\Config\Singleton;
use apexl\EasyCSV;
use apexl\Io\includes\System;
use apexl\Io\includes\Utils;
use apexl\Io\modules\component\entities\componentEntity;
use apexl\Io\modules\formbuilder\entities\formEntity;
use apexl\Io\modules\formbuilder\includes\checkboxField;
use apexl\Io\modules\formbuilder\includes\inputField;
use apexl\Io\modules\formbuilder\includes\selectField;
use apexl\Io\modules\page\services\Page;
use apexl\Io\modules\user\entities\companiesEntity;
use apexl\Io\modules\user\entities\contactEntity;
use apexl\Io\modules\user\entities\userEntity;
use apexl\Io\modules\user\services\currentUser;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;

class contactsController{

    protected $page;
    /** @var userEntity */
    protected $currentUser;

    public function __construct(Page $page, currentUser $currentUser){
        $this->page = $page;
        $this->currentUser = $currentUser::getCurrentUser();
    }

    public function list(Request $request, Response $response, $args){
        //First, we create the basic page
        $this->page->setMetaTitle('View Contacts List');
        $this->page->setMetaDescription('A list of all Contacts.');

        //next add components, we assume the page already contains some components to only add to the content component.
        $contentTile = new componentEntity();
        $contentTile->name = 'GoHaulContentTile';
        $contentTile->props = (object)[
            "title" => 'Contact Manager',
            "content" => "In here you can make changes to the contacts. Remember any changes you make will update instantly on the website.",
            "help" => 'Please select the contact from the list below you wish to update or create a new one.',
            "buttons" => [
                (object)[
                    "text" => 'Create New Contact',
                    "href" => '/contacts/create',
                    "button" => 'btn-primary'
                ],
                (object)[
                    "text" => 'Download Contacts',
                    "href" => '/contacts/download',
                    "button" => 'btn-secondary'
                ],
                (object)[
                    "text" => 'Email List',
                    "href" => '/contacts/export/mailList',
                    "button" => 'btn-secondary'
                ]
            ]
        ];

        $filterForm = new componentEntity();
        $filterForm->name = 'FilterForm';
        $filterForm->addProperty('dataSrc', '/data/contacts/filterForm');


        $DataTable = new componentEntity();
        $DataTable->name = 'DataTable';
        $DataTable->addProperty('dataSrc', '/data/contacts/list');


        $contentWrapper = new componentEntity();
        $contentWrapper->name = 'Content';
        $contentWrapper->classes = 'col-lg-10';
        $contentWrapper->addComponent($contentTile);
        $contentWrapper->addComponent($filterForm);
        $contentWrapper->addComponent($DataTable);

        $globalWrapper = $this->page->getComponent('ThePageContent');
        $globalWrapper->addComponent($contentWrapper);
        $this->page->replaceComponent('ThePageContent', $globalWrapper);

        return System::asJson($response, $this->page->getPage());
    }

    public function view(Request $request, Response $response, $args){
        if(!isset($args['ref'])){
            return System::asJson($response, ['message' => '404 not found'], 404);
        }
        $contact = new contactEntity();
        $contact->load($args['ref']);

        if(!isset($contact->ref)){
            return System::asJson($response, ['message' => '404 not found'], 404);
        }

        //First, we create the basic page
        $this->page->setMetaTitle('Contact '.$contact->first_name.' '.$contact->last_name);
        $this->page->setMetaDescription('Contact '.$contact->first_name.' '.$contact->last_name);

        $company = new companiesEntity();
        $company->load($contact->company);

        //next add components, we assume the page already contains some components to only add to the content component.
        $contentTile = new componentEntity();
        $contentTile->name = 'GoHaulContentTile';
        $contentTile->props = (object)[
            "title" => $contact->first_name.' '.$contact->last_name. ' ('.$company->full_name.')',
            "content" => "",
            "buttons" => [
                (object)[
                    "text" => 'Edit This Contact',
                    "href" => '/contacts/'.$contact->ref.'/edit',
                    "button" => 'btn-primary'
                ],
                (object)[
                    "text" => 'Contacts List',
                    "href" => '/contacts',
                    "button" => 'btn-secondary'
                ],
                (object)[
                    "text" => 'Delete This Contact',
                    "href" => '/contacts/'.$contact->ref.'/delete',
                    "button" => 'btn-danger'
                ]
            ]
        ];

        $contentRows = [];
        $contentRows[] = ['Company', $company->full_name];
        $contentRows[] = ['Current', $contact->current ? 'Yes' : 'No'];

        $contentDetailsTable = new componentEntity();
        $contentDetailsTable->name = 'ContentTable';
        $contentDetailsTable->classes = 'col-md-3';
        $contentDetailsTable->title = 'Details';
        $contentDetailsTable->rows = $contentRows;

        $contentRows = [];
        $contentRows[] = ['Email Address', $contact->email_address];
        $contentRows[] = ['Office Phone', $contact->office_phone];
        $contentRows[] = ['Mobile Phone', $contact->mobile_phone];

        $contentContactTable = new componentEntity();
        $contentContactTable->name = 'ContentTable';
        $contentContactTable->classes = 'col-md-3';
        $contentContactTable->title = 'Contact';
        $contentContactTable->rows = $contentRows;

        $contentRows = [];
        $contentRows[] = ['Accountant', $contact->role_accountant ? 'Yes' : 'No'];
        $contentRows[] = ['Transport Manager', $contact->role_transport ? 'Yes' : 'No'];
        $contentRows[] = ['Workshop', $contact->role_workshop ? 'Yes' : 'No'];

        $contentRolesTable = new componentEntity();
        $contentRolesTable->name = 'ContentTable';
        $contentRolesTable->classes = 'col-md-3';
        $contentRolesTable->title = 'Roles';
        $contentRolesTable->rows = $contentRows;

        $createdBy = new userEntity();
        $createdBy->load($contact->created_user);

        $modifiedBy = new userEntity();
        $modifiedBy->load($contact->modified_user);

        $contentRows = [];
        $contentRows[] = ['Created By', $createdBy->first_name.' '.$createdBy->last_name];
        $contentRows[] = ['Created Date', $contact->created_date];
        $contentRows[] = ['Modified By', $modifiedBy->first_name.' '.$modifiedBy->last_name];
        $contentRows[] = ['Modified Date', $contact->modified_date];

        $contentHistoryTable = new componentEntity();
        $contentHistoryTable->name = 'ContentTable';
        $contentHistoryTable->classes = 'col-md-3';
        $contentHistoryTable->title = 'History';
        $contentHistoryTable->rows = $contentRows;

        $rowWrapper = new componentEntity();
        $rowWrapper->name = 'RowWrapper';
        $rowWrapper->addComponent($contentDetailsTable);
        $rowWrapper->addComponent($contentContactTable);
        $rowWrapper->addComponent($contentRolesTable);
        $rowWrapper->addComponent($contentHistoryTable);

        $contentWrapper = new componentEntity();
        $contentWrapper->name = 'Content';
        $contentWrapper->classes = 'col-lg-10';
        $contentWrapper->addComponent($contentTile);
        $contentWrapper->addComponent($rowWrapper);

        $globalWrapper = $this->page->getComponent('ThePageContent');
        $globalWrapper->addComponent($contentWrapper);
        $this->page->replaceComponent('ThePageContent', $globalWrapper);

        return System::asJson($response, $this->page->getPage());
    }

    public function create(Request $request, Response $response, $args){
        //First, we create the basic page
        $this->page->setMetaTitle('Create a PO');
        $this->page->setMetaDescription('Create a PO.');

        //next add components, we assume the page already contains some components to only add to the content component.
        $contentTile = new componentEntity();
        $contentTile->name = 'GoHaulContentTile';
        $contentTile->props = (object)[
            "title" => 'Create New Contact',
            "content" => "Please enter the details of your new contact and click \"Submit\".",
            "buttons" => [
                (object)[
                    "text" => 'Contacts List',
                    "href" => '/contacts',
                    "button" => 'btn-secondary'
                ],
            ]
        ];

        $form = new componentEntity();
        $form->name = 'Form';
        $form->addProperty('dataSrc', '/data/contacts/createForm');


        $contentWrapper = new componentEntity();
        $contentWrapper->name = 'Content';
        $contentWrapper->classes = 'col-lg-10';
        $contentWrapper->addComponent($contentTile);
        $contentWrapper->addComponent($form);

        $globalWrapper = $this->page->getComponent('ThePageContent');
        $globalWrapper->addComponent($contentWrapper);
        $this->page->replaceComponent('ThePageContent', $globalWrapper);

        return System::asJson($response, $this->page->getPage());
    }

    public function update(Request $request, Response $response, $args){
        //First, we create the basic page
        $contact = new contactEntity();
        $contact->load($args['ref']);

        $company = new companiesEntity();
        $company->load($contact->company);

        $this->page->setMetaTitle('Update '.$contact->first_name.' '.$contact->last_name. ' ('.$company->full_name.')');
        $this->page->setMetaDescription('Update '.$contact->first_name.' '.$contact->last_name. ' ('.$company->full_name.')');

        //next add components, we assume the page already contains some components to only add to the content component.
        $contentTile = new componentEntity();
        $contentTile->name = 'GoHaulContentTile';
        $contentTile->props = (object)[
            "title" => $contact->first_name.' '.$contact->last_name. ' ('.$company->full_name.')',
            "content" => "Please change the contact as required and click \"Submit\".",
            "buttons" => [
                (object)[
                    "text" => 'Create New Contact',
                    "href" => '/contacts/create',
                    "button" => 'btn-primary'
                ],
                (object)[
                    "text" => 'Contacts List',
                    "href" => '/contacts',
                    "button" => 'btn-secondary'
                ],
            ]
        ];

        $form = new componentEntity();
        $form->name = 'Form';
        $form->addProperty('dataSrc', '/data/contacts/'.$args['ref'].'/updateForm');


        $contentWrapper = new componentEntity();
        $contentWrapper->name = 'Content';
        $contentWrapper->classes = 'col-lg-10';
        $contentWrapper->addComponent($contentTile);
        $contentWrapper->addComponent($form);

        $globalWrapper = $this->page->getComponent('ThePageContent');
        $globalWrapper->addComponent($contentWrapper);
        $this->page->replaceComponent('ThePageContent', $globalWrapper);

        return System::asJson($response, $this->page->getPage());
    }

    public function filterForm(Request $request, Response $response, $args){
        $form = new formEntity();
        $form->setId('contactFilter');
        $form->setMethod('get');

        $form->addField((new checkboxField('current'))->setLabel('Current'), 'filter');
        $form->addField((new checkboxField('mailing_list'))->setLabel('Mailing List'), 'filter');
        $form->addField((new checkboxField('role_accountant'))->setLabel('Accountant'), 'filter');
        $form->addField((new checkboxField('role_transport'))->setLabel('Transport Manager'), 'filter');
        $form->addField((new checkboxField('role_workshop'))->setLabel('Workshop'), 'filter');
        $form->addField((new inputField('search'))->setLabel('Search'), 'filter');
        return System::asJson($response, $form->getBuiltFormArray());
    }

    public function manageForm(Request $request, Response $response, $args)
    {
        $contact = FALSE;
        if (isset($args['ref'])) {
            $contact = new contactEntity();
            $contact->load($args['ref']);
        }

        $form = new formEntity();
        $form->setId('manageContact');
        $form->setMethod('post');
        $form->setActionUrl('contacts');
        if($contact){
            $form->setActionUrl('contacts/'.$args['ref']);
        }

        //Details
        $first_name = (new inputField('first_names'))->setLabel('First Name')->isRequired();
        if($contact){
            $first_name->setValue($contact->first_names);
        }
        $form->addField($first_name, 'details');

        $last_name = (new inputField('last_name'))->setLabel('Last Name');
        if($contact){
            $last_name->setValue($contact->last_name);
        }
        $form->addField($last_name, 'details');

        $companies = (new companiesEntity())->loadMultiple([], ["full_name", 'ASC']);
        $company = (new selectField('company'))->buildOptionsFromEntities($companies, 'ref', 'full_name')->setLabel('Company')->isRequired();
        if($contact){
            $company->setValue($contact->company);
        }
        $form->addField($company, 'details');

        $current = (new checkboxField('current'))->setLabel('Current');
        if($contact && $contact->current){
            $current->setValue($contact->current);
        }
        $form->addField($current, 'details');

        //Contact
        $officePhone = (new inputField('office_phone'))->setLabel('Office Phone');
        if($contact){
            $officePhone->setValue($contact->office_phone);
        }
        $form->addField($officePhone, 'contact');

        $mobile = (new inputField('mobile_phone'))->setLabel('Mobile Phone');
        if($contact){
            $mobile->setValue($contact->mobile_phone);
        }
        $form->addField($mobile, 'contact');

        $email = (new inputField('email_address'))->setLabel('Email Address');
        if($contact){
            $email->setValue($contact->email_address);
        }
        $form->addField($email, 'contact');

        $maillist = (new checkboxField('maillist'))->setLabel('On Mailing List');
        if($contact && $contact->maillist){
            $maillist->setValue($contact->maillist);
        }
        $form->addField($maillist, 'contact');

        //Roles
        $accountant = (new checkboxField('role_accountant'))->setLabel('Accountant');
        if($contact && $contact->role_accountant){
            $accountant->setValue($contact->role_accountant);
        }
        $form->addField($accountant, 'roles');

        $transManager = (new checkboxField('role_transport'))->setLabel('Transport Manager');
        if($contact && $contact->role_transport){
            $transManager->setValue($contact->role_transport);
        }
        $form->addField($transManager, 'roles');

        $workshop = (new checkboxField('role_workshop'))->setLabel('Workshop');
        if($contact && $contact->role_workshop){
            $workshop->setValue($contact->role_workshop);
        }
        $form->addField($workshop, 'roles');

        return System::asJson($response, $form->getBuiltFormArray());
    }

    public function dataContactsList($request, $response, $data){
        $params = $request->getQueryParams();

        $limit = isset($params['limit']) && !empty($params['limit']) ? $params['limit'] : 0;
        $offset = isset($params['offset']) && !empty($params['offset']) ? $params['offset'] : 0;

        $filters = $this->buildFilterConditions($params);

        $contacts = (new contactEntity())->loadMultiple($filters, ['ref'], $limit, $offset);
        $totalResults = (new contactEntity())->totalEntities();
        $totalPages = ceil (($totalResults/$params['limit']));
        $dataArray = [];
        foreach($contacts as $contact){
            $entityData = (object)$contact->getData();

            $company = new companiesEntity();
            $company->load($entityData->company);

            $data = (object)[
                "Ref" => $entityData->ref,
                "Company" => $company->full_name,
                "Name" => $entityData->first_names.' '.$entityData->last_name,
                "Phone" => $entityData->office_phone,
                "Email" => $entityData->email_address,
                "Accountant" => $entityData->role_accountant,
                "Transport Manager" => $entityData->role_transport,
                "Workshop" => $entityData->role_workshop,
                "Mailing List" => isset($entityData->maillist) && !empty($entityData->maillist) ? 1 : 0
            ];
            $dataArray[] = $data;
        }
        $tableHeader = isset($dataArray[0]) ? array_keys((array)$dataArray[0]) : [];
        return System::asJson($response, ['rows' => $dataArray, 'totalData' => $totalResults, 'totalPages' => $totalPages, 'tableHeader' => $tableHeader]);
    }

    public function createUpdate($request, $response, $data){
        $contacts = new contactEntity();

        $body = $request->getParsedBody();

        if(isset($data['ref'])){
            $contacts->load($data['ref']);
        } else {
            $contacts->created_date = date('Y-m-d H:i:s');
            $contacts->created_user = $this->currentUser->id;
            $contacts->franchise = $this->currentUser->franchise;
        }

        $contacts->first_names = $body->first_names;
        $contacts->last_name = $body->last_name;
        $contacts->company = $body->company;
        $contacts->current = !empty($body->current) ? $body->current : 0;
        $contacts->email_address = $body->email_address;
        $contacts->office_phone = $body->office_phone;
        $contacts->mobile_phone = $body->mobile_phone;
        $contacts->role_accountant = !empty($body->role_accountant) ? $body->role_accountant : 0;
        $contacts->role_transport = !empty($body->role_transport) ? $body->role_transport : 0;
        $contacts->role_workshop = !empty($body->role_workshop) ? $body->role_workshop : 0;

        $contacts->maillist = !empty($body->maillist) ? $body->maillist : 0;

        $contacts->modified_date = date('Y-m-d H:i:s');
        $contacts->modified_user = $this->currentUser->id;

        if($contacts->isValid()){
            $contacts->store();
            return System::asJson($response, ['success' => 'true', 'redirectTo' => '/contacts']);
        }

        return System::asJson($response, ['success' => 'false'], 400);
    }

    public function deleteView($request, $response, $data){
        //First, we create the basic page
        $contact = new contactEntity();
        if(isset($data['ref'])){
            $contact->load($data['ref']);

            $this->page->setMetaTitle('Delete a Job');
            $this->page->setMetaDescription('Delete a Job.');

            //next add components, we assume the page already contains some components to only add to the content component.
            $contentTile = new componentEntity();
            $contentTile->name = 'GoHaulContentTile';
            $contentTile->props = (object)[
                "title" => 'Delete A Contact',
                "content" => "Are you sure you want to delete this Contact (".$contact->first_name.' '.$contact->last_name.")?",
                "buttons" => [
                    (object)[
                        "text" => 'Confirm Delete',
                        "href" => '/contacts/'.$contact->ref,
                        "button" => 'btn-danger',
                        "type" => "delete"
                    ],
                ]
            ];

            $contentWrapper = new componentEntity();
            $contentWrapper->name = 'Content';
            $contentWrapper->classes = 'col-lg-10';
            $contentWrapper->addComponent($contentTile);

            $globalWrapper = $this->page->getComponent('ThePageContent');
            $globalWrapper->addComponent($contentWrapper);
            $this->page->replaceComponent('ThePageContent', $globalWrapper);

            return System::asJson($response, $this->page->getPage());
        } else {
            return System::asJson($response, ['success' => 'false'], 404);
        }
    }

    public function exportMailList(Request $request, Response $response, $args){
        $config = Singleton::getInstance();
        //write out a data file, redirect to the file itself to trigger a download.
        $contactEntity = new contactEntity();
        $contacts = $contactEntity->loadMultiple(['maillist' => ['maillist', 1]]);
        //build data array for CSV
        $contactExport = [];
        foreach($contacts as $contact){
            $contactExport[] = ["emailAddress" => $contact->email_address];
        }

        $path = $this->exportCsv('mailingList.csv', 'exports', $contactExport);

        return System::asJson($response, ['success' => 'true', 'redirectToExternal' => $config->app->site->backend_domain.'/'.$path]);
    }

    public function exportContactList(Request $request, Response $response, $args){
        $config = Singleton::getInstance();
        //write out a data file, redirect to the file itself to trigger a download.
        $contactEntity = new contactEntity();
        $contacts = $contactEntity->loadMultiple();
        //build data array for CSV
        $contactExport = [];
        foreach($contacts as $contact){
            $contactExport[] = (array)$contact->getData();
        }

        $path = $this->exportCsv('contactList.csv', 'exports', $contactExport);

        return System::asJson($response, ['success' => 'true', 'redirectToExternal' => $config->app->site->backend_domain.'/'.$path]);
    }

    protected function exportCsv($filename, $directory, $data){
        $basePath = System::getBasePath();
        $config = Singleton::getInstance();

        $path = $config->app->fileUploadDirectory.'/'.$directory;

        $csv = new EasyCSV();
        $csv->arrayToCsv($data);
        $csv->setFileName($filename);
        Utils::createPath($basePath.'/'.$path);
        $csv->setStorePath($basePath.'/'.$path);
        $csv->store();

        return $path.'/'.$filename;
    }

    public function delete($request, $response, $data){
        (new contactEntity())->delete($data['ref']);
        return System::asJson($response, ['success' => 'true', 'redirectTo' => '/contacts']);
    }

    protected function buildFilterConditions($params){
        $filters = [];
        foreach($params as $param => $value){
            if($param == 'limit' || $param == 'offset' || $param == 'page' || empty($value)){
                continue;
            }
            switch($param){
                case 'search':
                    $filters['ref'] = ['ref', '%'.$value.'%', 'like', 'OR'];
                    $filters['first_names'] = ['first_names', '%'.$value.'%', 'like', 'OR'];
                    $filters['last_name'] = ['last_name', '%'.$value.'%', 'like', 'OR'];
                    $filters['office_phone'] = ['office_phone', '%'.$value.'%', 'like', 'OR'];
                    $filters['mobile_phone'] = ['mobile_phone', '%'.$value.'%', 'like', 'OR'];
                    $filters['email_address'] = ['email_address', '%'.$value.'%', 'like', 'OR'];
                    break;
                case 'current':
                    $filters['current'] = ['current', 1];
                    break;
                case 'company':
                    $filters['company'] = ['company', $value];
                    break;
                case 'mailing_list':
                    $filters['maillist'] = ['maillist', 1];
                    break;
                case 'role_accountant':
                    $filters['role_accountant'] = ['role_accountant', 1];
                    break;
                case 'role_transport':
                    $filters['role_transport'] = ['role_transport', 1];
                    break;
                case 'role_workshop':
                    $filters['role_workshop'] = ['role_workshop', 1];
                    break;
            }
        }
        return $filters;
    }
}