<?php

namespace apexl\Io\modules\gohaul\controllers;

use apexl\Config\Singleton;
use apexl\Io\includes\System;
use apexl\Io\includes\Utils;
use apexl\io\moduels\gohaul\services\SageOne;
use apexl\Io\modules\component\entities\componentEntity;
use apexl\Io\modules\formbuilder\entities\formEntity;
use apexl\Io\modules\formbuilder\includes\checkboxField;
use apexl\Io\modules\formbuilder\includes\datePickerField;
use apexl\Io\modules\formbuilder\includes\fileField;
use apexl\Io\modules\formbuilder\includes\inputField;
use apexl\Io\modules\formbuilder\includes\selectField;
use apexl\Io\modules\formbuilder\includes\textareaField;
use apexl\Io\modules\page\services\Page;
use apexl\Io\modules\user\entities\attachmentEntity;
use apexl\Io\modules\user\entities\companiesEntity;
use apexl\Io\modules\user\entities\franchiseEntity;
use apexl\Io\modules\user\entities\jobEntity;
use apexl\Io\modules\user\entities\purchaseOrderEntity;
use apexl\Io\modules\user\entities\userEntity;
use apexl\Io\modules\user\entities\vehicleEntity;
use apexl\Io\modules\user\entities\vehicleTypeEntity;
use apexl\Io\modules\user\services\currentUser;
use apexl\Vault\Vault;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;

class jobsController{

    protected $page;
    /** @var userEntity */
    protected $currentUser;
    /** @var Singleton */
    protected $config;
    public function __construct(Page $page, currentUser $currentUser, Singleton $config){
        $this->page = $page;
        $this->currentUser = $currentUser::getCurrentUser();
        $this->config = $config;
    }

    public function list(Request $request, Response $response, $args){
        //First, we create the basic page
        $this->page->setMetaTitle('View Jobs List');
        $this->page->setMetaDescription('A list of all created jobs.');

        //next add components, we assume the page already contains some components to only add to the content component.
        $contentTile = new componentEntity();
        $contentTile->name = 'GoHaulContentTile';
        $contentTile->props = (object)[
            "title" => 'Job Manager',
            "content" => "In here you can make changes to the jobs. \n\r Remember any changes you make will update instantly on the website.",
            "help" => 'Please select the job from the list below you wish to update or create a new one.',
            "buttons" => [
                (object)[
                    "text" => 'Create New Job',
                    "href" => '/jobs/create',
                    "button" => 'btn-primary'
                ],
                (object)[
                    "text" => 'Sync Invoices',
                    "href" => '/invoices/sync',
                    "button" => 'btn-secondary'
                ]
            ]
        ];

        $filterForm = new componentEntity();
        $filterForm->name = 'FilterForm';
        $filterForm->addProperty('dataSrc', '/data/jobs/filterForm');


        $DataTable = new componentEntity();
        $DataTable->name = 'DataTable';
        $DataTable->addProperty('dataSrc', '/data/jobs/list');


        $contentWrapper = new componentEntity();
        $contentWrapper->name = 'Content';
        $contentWrapper->classes = 'col-lg-10';
        $contentWrapper->addComponent($contentTile);
        $contentWrapper->addComponent($filterForm);
        $contentWrapper->addComponent($DataTable);

        $globalWrapper = $this->page->getComponent('ThePageContent');
        $globalWrapper->addComponent($contentWrapper);
        $this->page->replaceComponent('ThePageContent', $globalWrapper);

        return System::asJson($response, $this->page->getPage());
    }

    public function apiJobImage(Request $request, Response $response, $args){
        $uploadedFile = $request->getUploadedFiles();
        //save the record to DB and return the ID
        $document = new attachmentEntity();
        $document->notes = '';
        error_log(json_encode($uploadedFile));
        $document->file_mime_attachment = $uploadedFile['file']->getClientMediaType();
        $document->name = $uploadedFile['file']->getClientFilename();//$body->name.'.'.$document->file_mime_attachment;
        $document->job = $args['ref'];
        $document->franchise = $this->currentUser->franchise;

        $document->created_date = date('Y-m-d H:i:s');
        $document->created_user = $this->currentUser->id;
        $document->franchise = $this->currentUser->franchise;

        $document->modified_date = date('Y-m-d H:i:s');
        $document->modified_user = $this->currentUser->id;

        $document->store();

        //move it to the upload Dir, which in this case is the document id
        Utils::moveUploadedFile('job_attachments', $uploadedFile['file']);

        $document->store();

        return System::asJson($response, ['fid' => $document->ref]);
    }


    public function apiJobUpdate(Request $request, Response $response, $args){
        if(!isset($args['ref'])){
            return System::asJson($response, ['message' => '404 not found'], 404);
        }

        $job = new jobEntity();
        $job->load($args['ref']);

        $body = $request->getParsedBody();

        $job->driver_status = $body->driver_status;

        $job->store();

        return System::asJson($response, $job->getData());
    }

    public function apiJob(Request $request, Response $response, $args)
    {
        if(!isset($args['ref'])){
            return System::asJson($response, ['message' => '404 not found'], 404);
        }

        $job = new jobEntity();
        $job->load($args['ref']);

        if(!isset($job->ref)){
            return System::asJson($response, ['message' => '404 not found, job id '.$args['ref'].' is missing or malformed.'], 404);
        }

        $company = new companiesEntity();
        $company->load($job->company);

        $franchise = new franchiseEntity();
        $franchise->load($job->franchise);

        $job->grouped_location = $job->location_1."\n".$job->location_2."\n".$job->location_3."\n".$job->location_4;
        $job->po_reference = $franchise->short_name.'/'.date('ym', strtotime($job->job_date)).'/'.$job->purchase_order;
        $job->company_fullname = $company->full_name;

        return System::asJson($response, $job->getData());
    }

    public function view(Request $request, Response $response, $args){
        if(!isset($args['ref'])){
            return System::asJson($response, ['message' => '404 not found'], 404);
        }
        $job = new jobEntity();
        $job->load($args['ref']);

        if(!isset($job->ref)){
            return System::asJson($response, ['message' => '404 not found, job id '.$args['ref'].' is missing or malformed.'], 404);
        }

        $company = new companiesEntity();
        $company->load($job->company);

        //First, we create the basic page
        $this->page->setMetaTitle('Job #'.$job->ref.' for '.$company->name);
        $this->page->setMetaDescription('View a job');

        //next add components, we assume the page already contains some components to only add to the content component.
        $contentTile = new componentEntity();
        $contentTile->name = 'GoHaulContentTile';
        $contentTile->props = (object)[
            "title" => 'Job #'.$job->ref.' for '.$company->name,
            "content" => "",
            "buttons" => [
                (object)[
                    "text" => 'Edit This Job',
                    "href" => '/jobs/'.$job->ref.'/edit',
                    "button" => 'btn-primary'
                ],
                (object)[
                    "text" => 'Create Invoice',
                    "href" => '/jobs/'.$job->ref.'/invoice',
                    "button" => 'btn-primary'
                ],
                (object)[
                    "text" => 'Jobs List',
                    "href" => '/jobs',
                    "button" => 'btn-secondary'
                ],
                (object)[
                    "text" => 'Delete This job',
                    "href" => '/jobs/'.$job->ref.'/delete',
                    "button" => 'btn-danger'
                ]
            ]
        ];

        $vehicleType = new vehicleTypeEntity();
        $vehicleType->load($job->vehicle_type);

        $vehicle = new vehicleEntity();
        $vehicle->load($job->vehicle);

        $driver = new userEntity();
        $driver->load($job->driver);

        $purchaseOrder = new purchaseOrderEntity();
        $purchaseOrder->load($job->purchase_order);

        $contentRows = [];
        $contentRows[] = ['Company', (object)['href' => '/companies/'.$company->ref, 'title' => $company->full_name]];
        $contentRows[] = ['Purchase Order', (object)['href' => '/pos/'.$job->purchase_order, 'title' => $job->purchase_order]];
        $contentRows[] = ['Date', $job->job_date];
        $contentRows[] = ['Status', $job->status];

        $contentRows[] = ['Vehicle Type', $vehicleType->name];
        $contentRows[] = ['Vehicle Registration', $purchaseOrder->vehicle_registration];
        if($driver->ref) {
            $contentRows[] = ['Driver', $driver->first_name . ' ' . $driver->last_name];
        } else {
            $contentRows[] = ['Driver', ''];
        }
        $contentRows[] = ['Vehicle', $vehicle->registration ?? ''];

        $contentRows[] = ['Completed', $job->completed_date];
        $contentRows[] = ['Confirmed', $job->confirmed_date];
        $contentRows[] = ['Invoiced', $job->invoiced_date.' '.$job->invoice_no];

        $contentDetailsTable = new componentEntity();
        $contentDetailsTable->name = 'ContentTable';
        $contentDetailsTable->classes = 'col-md-12 col-lg-6 col-xl-4';
        $contentDetailsTable->title = 'Details';
        $contentDetailsTable->rows = $contentRows;

        $contentRows = [];
        $contentRows[] = [$job->location_1];
        $contentRows[] = [$job->location_2];
        $contentRows[] = [$job->location_3];
        $contentRows[] = [$job->location_4];

        $contentRows[] = ['Per Mile', $vehicleType->cost_per_mile.'p'];
        $contentRows[] = ['Actual', '£'.number_format($job->cost, 2)];

        $contentRows[] = ['Invoiced', $job->invoiced_amount];
        $contentRows[] = ['Paid', $job->amount_paid];
        $contentRows[] = ['Sync', ''];

        $contentContactTable = new componentEntity();
        $contentContactTable->name = 'ContentTable';
        $contentContactTable->classes = 'col-md-12 col-lg-6 col-xl-4';
        $contentContactTable->title = 'Financial';
        $contentContactTable->rows = $contentRows;

        $createdBy = new userEntity();
        $createdBy->load($job->created_user);

        $modifiedBy = new userEntity();
        $modifiedBy->load($job->modified_user);

        $contentRows = [];
        $contentRows[] = ['Created By', $createdBy->first_name.' '.$createdBy->last_name];
        $contentRows[] = ['Created Date', $job->created_date];
        $contentRows[] = ['Modified By', $modifiedBy->first_name.' '.$modifiedBy->last_name];
        $contentRows[] = ['Modified Date', $job->modified_date];

        $contentHistoryTable = new componentEntity();
        $contentHistoryTable->name = 'ContentTable';
        $contentHistoryTable->classes = 'col-md-12 col-lg-6 col-xl-4';
        $contentHistoryTable->title = 'History';
        $contentHistoryTable->rows = $contentRows;

        $contentRows = [];
        $contentRows[] = [$job->notes];

        $contentNotesTable = new componentEntity();
        $contentNotesTable->name = 'ContentTable';
        $contentNotesTable->classes = 'col-md-12';
        $contentNotesTable->rows = $contentRows;

        $form = new componentEntity();
        $form->name = 'Form';
        $form->classes = 'col-md-12';
        $form->addProperty('dataSrc', '/data/jobs/'.$job->ref.'/jobAttachmentsForm');

        $rowWrapper = new componentEntity();
        $rowWrapper->name = 'RowWrapper';
        $rowWrapper->addComponent($contentDetailsTable);
        $rowWrapper->addComponent($contentContactTable);
        $rowWrapper->addComponent($contentHistoryTable);

        $jobAttachment = new componentEntity();
        $jobAttachment->name = 'SimpleTable';
        $jobAttachment->classes = 'col-sm-12';
        $jobAttachment->addProperty('dataSrc', '/data/jobs/'.$job->ref.'/jobAttachmentsData');

        $attachmentTab = new componentEntity();
        $attachmentTab->name = 'Tab';
        $attachmentTab->label = 'Job Attachments';
        $attachmentTab->id = 'job_attachments';
        $attachmentTab->active = true;
        $attachmentTab->addComponent($jobAttachment);
        $attachmentTab->addComponent($form);

        $notesTab = new componentEntity();
        $notesTab->name = 'Tab';
        $notesTab->label = 'Notes';
        $notesTab->id = 'notes';
        $notesTab->active = false;
        $notesTab->addComponent($contentNotesTable);

        $tabContainer = new componentEntity();
        $tabContainer->name = 'TabsContainer';
        $tabContainer->addComponent($attachmentTab);
        $tabContainer->addComponent($notesTab);

        $contentWrapper = new componentEntity();
        $contentWrapper->name = 'Content';
        $contentWrapper->classes = 'col-lg-10';
        $contentWrapper->addComponent($contentTile);
        $contentWrapper->addComponent($rowWrapper);
        $contentWrapper->addComponent($tabContainer);

        $globalWrapper = $this->page->getComponent('ThePageContent');
        $globalWrapper->addComponent($contentWrapper);
        $this->page->replaceComponent('ThePageContent', $globalWrapper);

        return System::asJson($response, $this->page->getPage());
    }

    public function create(Request $request, Response $response, $args){
        //First, we create the basic page
        $this->page->setMetaTitle('Create a Job');
        $this->page->setMetaDescription('Create a Job.');

        //next add components, we assume the page already contains some components to only add to the content component.
        $contentTile = new componentEntity();
        $contentTile->name = 'GoHaulContentTile';
        $contentTile->props = (object)[
            "title" => 'Create A New Job',
            "content" => "Please enter the details of your new job and click \"Submit\".",
            "buttons" => [
                (object)[
                    "text" => 'Jobs List',
                    "href" => '/jobs',
                    "button" => 'btn-secondary'
                ],
            ]
        ];

        $form = new componentEntity();
        $form->name = 'Form';
        $form->addProperty('dataSrc', '/data/jobs/createForm');


        $contentWrapper = new componentEntity();
        $contentWrapper->name = 'Content';
        $contentWrapper->classes = 'col-lg-10';
        $contentWrapper->addComponent($contentTile);
        $contentWrapper->addComponent($form);

        $globalWrapper = $this->page->getComponent('ThePageContent');
        $globalWrapper->addComponent($contentWrapper);
        $this->page->replaceComponent('ThePageContent', $globalWrapper);

        return System::asJson($response, $this->page->getPage());
    }

    public function update(Request $request, Response $response, $args){
        //First, we create the basic page
        $this->page->setMetaTitle('Create a Job');
        $this->page->setMetaDescription('Create a Job.');

        //next add components, we assume the page already contains some components to only add to the content component.
        $contentTile = new componentEntity();
        $contentTile->name = 'GoHaulContentTile';
        $contentTile->props = (object)[
            "title" => 'Update this job',
            "content" => "Please edit the details of your job and click \"Submit\".",
            "buttons" => [
                (object)[
                    "text" => 'Create New Job',
                    "href" => '/jobs/create',
                    "button" => 'btn-primary'
                ],
                (object)[
                    "text" => 'Jobs List',
                    "href" => '/jobs',
                    "button" => 'btn-secondary'
                ],
            ]
        ];

        $form = new componentEntity();
        $form->name = 'Form';
        $form->addProperty('dataSrc', '/data/job/'.$args['ref'].'/updateForm');


        $contentWrapper = new componentEntity();
        $contentWrapper->name = 'Content';
        $contentWrapper->classes = 'col-lg-10';
        $contentWrapper->addComponent($contentTile);
        $contentWrapper->addComponent($form);

        $globalWrapper = $this->page->getComponent('ThePageContent');
        $globalWrapper->addComponent($contentWrapper);
        $this->page->replaceComponent('ThePageContent', $globalWrapper);

        return System::asJson($response, $this->page->getPage());
    }

    /**
     * build the jobs filter page form
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     * @throws \Exception
     */
    public function filterForm(Request $request, Response $response, $args){
        //initialise form
        $form = new formEntity();
        $form->setId('filterJobs');
        $form->setMethod('get');

        //set form Fields

        //filter fieldset
        $statusList = [
            "" => 'Any Status',
            "Pending" => "Pending",
            "Ongoing" => "Ongoing",
            "Overdue" => "Overdue",
            "Complete" => "Complete",
            "Ready to be invoiced" => "Ready to be invoiced",
            "Invoiced" => "Invoiced"
        ];

        $status = (new selectField('status'))->addOptions($statusList)->setLabel('Status');
        //Details Fieldset
        $form->addField($status, 'filter');

        $companies = (new companiesEntity())->loadMultiple(["type_customer" => ["type_customer", 1]], ["full_name", 'ASC']);
        $company = (new selectField('company'))->buildOptionsFromEntities($companies, 'ref', 'full_name')->setLabel('Company')->isRequired();
        $form->addField($company, 'filter');

        $drivers = (new userEntity())->loadMultiple([], ["first_name"]);
        $driversField = (new selectField('driver'))->buildOptionsFromEntities($drivers, 'id', 'first_name')->setLabel('Driver');
        $form->addField($driversField, 'filter');

        $types = (new vehicleTypeEntity())->loadMultiple();
        //callback returns cost and vehicle information.
        $typesField = (new selectField('vehicle_type'))->buildOptionsFromEntities($types, 'ref', 'name')->setLabel('Vehicle Type');
        $form->addField($typesField, 'filter');

        //this field is set by callback from the types field.
        $vehicles = (new vehicleEntity())->loadMultiple();
        $vehicleOptions = [];
        foreach ($vehicles as $vehicle){
            $type = new vehicleTypeEntity();
            $type->load($vehicle->vehicle_type);
            $vehicleOptions[$vehicle->ref] = $type->name .' ('.$vehicle->registration.')';
        }
        $vehicleField = (new selectField('vehicle'))->addOptions($vehicleOptions)->setLabel('Vehicle');
        $form->addField($vehicleField, 'filter');


        $form->addField((new datePickerField('from_date'))->setLabel('From Date'), 'filter');
        $form->addField((new datePickerField('to_date'))->setLabel('To Date'), 'filter');

        //costs
        $form->addField((new inputField('search'))->setLabel('Search'), 'filter');

        return System::asJson($response, $form->getBuiltFormArray());
    }

    /**
     * Build the Jobs manage form.
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     * @throws \Exception
     */
    public function manageForm(Request $request, Response $response, $args){
        $job = FALSE;
        if(isset($args['ref'])){
            $job = new jobEntity();
            $job->load($args['ref']);
        }

        //initialise form
        $form = new formEntity();
        $form->setId('manageJob');
        $form->setMethod('post');
        $form->setActionUrl('jobs');
        if($job){
            $form->setActionUrl('jobs/'.$job->ref);
        }

        //set form Fields
        //Details Fieldset
        $companies = (new companiesEntity())->loadMultiple(["type_customer" => ["type_customer", 1]], ["full_name", 'ASC']);
        $company = (new selectField('company'))->buildOptionsFromEntities($companies, 'ref', 'full_name')->setLabel('Company')->isRequired();
        if($job){
            $company->setValue($job->company);
        }
        $form->addField($company, 'details');

        $vrid = (new inputField('vrid'))->setLabel('VRID');
        if($job){
            $vrid->setValue($job->vrid);
        }
        $form->addField($vrid, 'details');

        $jobDate = (new datePickerField('job_date'))->setLabel('Job Date')->isRequired();
        if($job){
            $jobDate->setValue($job->job_date);
        }
        $form->addField($jobDate, 'details');

        $types = (new vehicleTypeEntity())->loadMultiple();
        //callback returns cost and vehicle information.
        $typeCallbackData = [];
        foreach ($types as $type){
            $typeCallbackData[$type->ref] = $type->cost_per_mile.'p per mile';
        }
        $form->addCallbackData('total', $typeCallbackData);

        //add types field
        $typesField = (new selectField('vehicle_type'))->buildOptionsFromEntities($types, 'ref', 'name')->setLabel('Vehicle Type')->isRequired();
        if($job){
            $typesField->setValue($job->vehicle_type);
        }
        $form->addField($typesField, 'details');

        //POs
        $collection = (new inputField('collection'))->setLabel('Collection');
        $delivery = (new inputField('delivery'))->setLabel('Delivery');
        $haulier = (new inputField('haulier'))->setLabel('Haulier')->setValue('GHL');
        $our_price = (new inputField('our_price'))->setLabel('Our Price');
        $vehicle_registration = (new inputField('vehicle_registration'))->setLabel('Vehicle Registration');

        $po = new purchaseOrderEntity();
        $po->load($job->purchase_order);

        if($po){
            $collection->setValue($po->collection);
            $delivery->setValue($po->delivery);
            $haulier->setValue($po->haulier);
            $our_price->setValue($po->our_price);
            $vehicle_registration->setValue($po->vehicle_registration);
        }

        $form->addField($collection, 'purchase order');
        $form->addField($delivery, 'purchase order');
        $form->addField($haulier, 'purchase order');
        $form->addField($our_price, 'purchase order');
        $form->addField($vehicle_registration, 'purchase order');

        $poTest = (new purchaseOrderEntity())->loadUnassigned($this->currentUser->franchise);

        $options = [];
        foreach($poTest['data'] as $po){
            $options[$po->ref] = $po->ref;
        }
        if($job){
            $options[$job->purchase_order] = $job->purchase_order;
        }
        $purchaseOrder = (new selectField('purchase_order'))->addOptions($options)->setLabel('--OR-- Select existing Purchase Order');
        if($job){
            $purchaseOrder->setValue($job->purchase_order);
        }
        $form->addField($purchaseOrder, 'purchase order');

        //locations
        $location1 = (new inputField('location_1'))->setLabel('Location 1')->isRequired();
        if($job){
            $location1->setValue($job->location_1);
        }
        $form->addField($location1, 'locations');

        $location2 = (new inputField('location_2'))->setLabel('Location 2')->isRequired();
        if($job){
            $location2->setValue($job->location_2);
        }
        $form->addField($location2, 'locations');

        $location3 = (new inputField('location_3'))->setLabel('Location 3');
        if($job){
            $location3->setValue($job->location_3);
        }
        $form->addField($location3, 'locations');

        $location4 = (new inputField('location_4'))->setLabel('Location 4');
        if($job){
            $location4->setValue($job->location_4);
        }
        $form->addField($location4, 'locations');

        //similar journeys (Nothing here?)

        //costs
        $total = (new inputField('total'))->isDisabled()->setLabel('Total')->setCallback('total', 'vehicle_type');
        $form->addField($total, 'total');

        $cost = (new inputField('cost'))->setLabel('Cost')->isRequired();
        if($job){
            $cost->setValue($job->cost);
        }
        $form->addField($cost, 'total');

        //booking
        $users = (new userEntity())->loadMultiple([], ["first_name"]);
        //TN Modification. We only want to show users with the driver role, not all users.
        //@todo add config to select the driver specific role(s)
        $drivers = [];
        foreach($users as $user){
            $roles = unserialize($user->roles); //TN @todo shouldnt need to unserialise here - fix in load multiple method on entity
            if(!empty($roles) && in_array(6, $roles)){
                $drivers[$user->id] = $user;
            }
        }

        $driversField = (new selectField('driver'))->buildOptionsFromEntities($drivers, 'id', 'first_name')->setLabel('Driver')->isRequired();
        if($job){
            $driversField->setValue($job->driver);
        }
        $form->addField($driversField, 'booking');

        //vehicles
        $vehicles = (new vehicleEntity())->loadMultiple();
        $vehicleOptions = [];
        foreach ($vehicles as $vehicle){
            $type = new vehicleTypeEntity();
            $type->load($vehicle->vehicle_type);
            $vehicleOptions[$vehicle->ref] = $type->name .' ('.$vehicle->registration.')';
        }
        $vehicleField = (new selectField('vehicle'))->addOptions($vehicleOptions)->setLabel('Vehicle');
        if($job){
            $vehicleField->setValue($job->vehicle);
        }
        $form->addField($vehicleField, 'booking');

        $conflicts = (new inputField('conflicts'))->isDisabled()->setLabel('Conflicts')->setValue('Please enter a date and select a vehicle')->isRequired();
        $form->addField($conflicts, 'booking');

        //Status
        $driverStatusCheck = (new checkboxField('driver_status_checkbox'))->setLabel('Job Delivered?');
        if($this->currentUser->isAllowed('MarkJobDelivered') === FALSE){
            $driverStatusCheck->isDisabled();
        }
        if($job && $job->driver_status == 5){
            $driverStatusCheck->setValue(1);
        }
        $form->addField($driverStatusCheck, 'status');

        $confirmed = (new checkboxField('confirmed_date'))->setLabel('Confirmed?');
        if($job && $job->confirmed_date){
            $confirmed->setValue(1);
        }
        $form->addField($confirmed, 'status');

        $completed = (new checkboxField('completed_date'))->setLabel('Completed?')->isDisabled();
        if($job && $job->completed_date){
            $completed->setValue(1);
        }
        $form->addField($completed, 'status');

        //Sage
        $invoiceDate = (new datePickerField('invoiced_date'))->setLabel('Invoice Date');
        if($job){
            $invoiceDate->setValue($job->invoiced_date);
        }
        $form->addField($invoiceDate, 'sage');

        $invoiceNumber = (new inputField('invoice_no'))->setLabel('Invoice Number');
        if($job){
            $invoiceNumber->setValue($job->invoice_no);
        }
        $form->addField($invoiceNumber, 'sage');

        $invoiced = (new inputField('invoiced_amount'))->setLabel('Invoiced');
        if($job){
            $invoiced->setValue($job->invoiced_amount);
        }
        $form->addField($invoiced, 'sage');

        $paid = (new inputField('amount_paid'))->setLabel('Paid');
        if($job){
            $paid->setValue($job->amount_paid);
        }
        $form->addField($paid, 'sage');

        //notes
        $notes = (new textareaField('notes'));
        if($job){
            $notes->setValue($job->notes);
        }
        $form->addField($notes, 'notes');

        return System::asJson($response, $form->getBuiltFormArray());
    }

    /**
     * Form Callback method
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     * @throws \Exception
     */
    public function typeSelectedCallback(Request $request, Response $response, $args){
        $form = new formEntity();
        $form->setId('manageJob');
        $form->setMethod('post');
        $form->setActionUrl('jobs');

        $type = new vehicleTypeEntity();
        $type->load($args['type']);

        $vehicle = (new vehicleEntity())->loadMultiple(['vehicle_type' => ['vehicle_type', $args['type']]]);
        $vehicleField = (new selectField('vehicle'))->addOptions($vehicle)->setLabel('Vehicle')->isRequired();
        $form->addField($vehicleField, 'booking');
        $form->addField((new inputField('cost'))->setLabel('Cost')->isRequired()->setValue($type->cost_per_mile), 'total');

        return System::asJson($response, $form->getBuiltFormArray());
    }

    /**
     * Function to generate the invoice for any given job.
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     * @throws \Exception
     */
    public function invoice(Request $request, Response $response, $args){
        $params = $request->getQueryParams();

        $job = FALSE;
        if(isset($args['ref'])){
            $job = new jobEntity();
            $job->load($args['ref']);
        }
        if(!$job){
            return System::asJson($response, ["success" => "false", "redirectTo" => "/jobs"]);
        }

        // Sage One
        $franchise = new franchiseEntity();
        $franchise->load($job->franchise);
        $sageOne = new SageOne($franchise);

        //check if we have params, if we do, process them.
        if(isset($params['code']) && isset($params['country'])){
            //got a code? get the auth token and store it against the franchise.
            $sageOne->getAuthToken($params['code']);
            $sageOne->invoice($job);

        } else {
            return System::asJson($response, ["redirectToExternal" => $sageOne->getAuthPath()]);
        }

        return System::asJson($response, ["success" => "true", "redirectTo" => "/jobs/".$job->ref]);
    }

    public function dataJobsList($request, $response, $data){
        $params = $request->getQueryParams();

        $limit = isset($params['limit']) && !empty($params['limit']) ? $params['limit'] : 0;
        $offset = isset($params['offset']) && !empty($params['offset']) ? $params['offset'] : 0;

        $orderBy = isset($params['order_by']) ? $params['order_by'] : 'ref';
        $orderByDir = isset($params['order_by_dir']) ? $params['order_by_dir'] : 'DESC';

        $filters = $this->buildFilterConditions($params);

        $jobs = (new jobEntity())->loadMultiple($filters, [$orderBy, $orderByDir], $limit, $offset);
        $totalResults = (new jobEntity())->totalEntities();
        $totalPages = ceil (($totalResults/$params['limit']));
        $dataArray = [];
        foreach($jobs as $job){
            $jobData = (object)$job->getData();
            $company = new companiesEntity();
            $company->load($jobData->company);

            $vehicleType = new vehicleTypeEntity();
            $vehicleType->load($jobData->vehicle_type);

            $franchise = new franchiseEntity();
            $franchise->load($jobData->franchise);

            $data = (object)[
                "Ref" => $jobData->ref,
                "Job Date" => $jobData->job_date,
                "Status" => $jobData->status,
                "Company" => $company->full_name,
                "Purchase Order" => $franchise->short_name.'/'.date('ym', strtotime($jobData->job_date)).'/'.$jobData->purchase_order,
                "Vehicle Type" => $vehicleType->name,
                "Location" => $jobData->location_1."\n".$jobData->location_2."\n".$jobData->location_3."\n".$jobData->location_4,
                "Cost" => "£".$jobData->cost,
                "Invoiced" => "£".$jobData->invoiced_amount,
                "Paid" => "£".$jobData->amount_paid,
                "Driver Status" => (int)$jobData->driver_status,
            ];
            $dataArray[] = $data;
        }
        $tableHeader = isset($dataArray[0]) ? array_keys((array)$dataArray[0]) : [];
        return System::asJson($response, ['rows' => $dataArray, 'totalData' => $totalResults, 'totalPages' => $totalPages, 'tableHeader' => $tableHeader]);
    }

    public function dataJobsListSimple($request, $response, $data){
        $params = $request->getQueryParams();

        $filters = $this->buildFilterConditions($params);

        $jobs = (new jobEntity())->loadMultiple($filters, ['ref']);
        $dataArray = [];
        foreach($jobs as $job){
            $jobData = (object)$job->getData();
            $company = new companiesEntity();
            $company->load($jobData->company);

            $franchise = new franchiseEntity();
            $franchise->load($jobData->franchise);

            $data = (object)[
                "Ref" => $jobData->ref,
                "Job Date" => $jobData->job_date,
                "Status" => $jobData->status,
                "Company" => $company->full_name,
                "Purchase Order" => $franchise->short_name.'/'.date('ym', strtotime($jobData->job_date)).'/'.$jobData->purchase_order,
                "Cost" => "£".$jobData->cost,
                "#" => (object)[
                    'href' => '/jobs/'.$jobData->ref.'/invoice', 'title' => '<button class="btn btn-success">Send Invoice <i class="fal fw fa-file-invoice"></i></button>',
                ]
            ];
            $dataArray[] = $data;
        }
        $tableHeader = isset($dataArray[0]) ? array_keys((array)$dataArray[0]) : [];
        return System::asJson($response, ['rows' => $dataArray, 'tableHeader' => $tableHeader]);
    }

    public function createUpdate($request, $response, $data){
        $job = new jobEntity();
        if(isset($data['ref'])){
            $job->load($data['ref']);
        } else {
            $job->created_date = date('Y-m-d H:i:s', time());
            $job->created_user = $this->currentUser->id;
            $job->franchise = $this->currentUser->franchise;
        }
        $body = $request->getParsedBody();

        //Check the PO fields, we need the PO number if these are filled in.
        if(($body->collection ?? FALSE) && !isset($data['ref'])) {
            $po = new purchaseOrderEntity();
            $po->collection = $body->collection ?? NULL;
            $po->delivery = $body->delivery ?? NULL;
            $po->haulier = $body->haulier ?? NULL;
            $po->our_price = $body->our_price ?? NULL;
            $po->vehicle_registration = $body->vehicle_registration ?? NULL;
            $po->created_date = date('Y-m-d H:i:s');
            $po->created_user = $this->currentUser->id;
            $po->franchise = $this->currentUser->franchise;
            if($po->isValid()) {
                $po->store();
                $body->purchase_order = $po->ref;
            } else {
                return System::asJson($response, ['success' => 'false'], 400);
            }
        }

        $job->company = $body->company;
        $job->purchase_order = $body->purchase_order;
        $job->vrid = !empty($body->vrid) ? $body->vrid : null;
        $job->job_date = $body->job_date;
        $job->vehicle_type = $body->vehicle_type;
        $job->driver_status = $body->driver_status ?? 0;


        $job->location_1 = $body->location_1;
        $job->location_2 = $body->location_2;
        $job->location_3 = $body->location_3;
        $job->location_4 = $body->location_4;

        $job->cost = $body->cost;
        $job->driver = !empty($body->driver) ? $body->driver : null;
        $job->vehicle = !empty($body->vehicle) ? $body->vehicle : null;

        $job->driver_status = !empty($body->driver_status) ? $body->driver_status : $job->driver_status;

        if(!empty($body->driver_status_checkbox)){
            $job->driver_status = 5;
            //we need to make sure we're on the admin form here, not the app.
        } elseif ($job->driver_status == 5 && empty($body->driver_status_checkbox) && empty($body->driver_status)){
            $job->driver_status = 0;
        }

        if(empty($job->confirmed_date) && !empty($body->confirmed_date)){
            $job->confirmed_date = date('Y-m-d H:i:s');
            $job->driver_status = 5; //Force driver status to be 5 on completion
        } elseif(!empty($job->confirmed_date) && empty($body->confirmed_date)) {
            $job->confirmed_date = null;
            $job->driver_status = 0; //force not complete if this is unchecked.
        }

        if(empty($job->completed_date) && !empty($body->completed_date)){
            $job->completed_date = date('Y-m-d H:i:s');
        } elseif(!empty($job->completed_date) && empty($body->completed_date)) {
            $job->completed_date = null;
        }

        $job->invoiced_date = !empty($body->invoiced_date) ? $body->invoiced_date : null;
        $job->invoice_no = !empty($body->invoice_no) ? $body->invoice_no : null;

        $job->invoiced_amount = !empty($body->invoiced_amount) ? $body->invoiced_amount : null;
        $job->amount_paid = !empty($body->amount_paid) ? $body->amount_paid : null ;

        $job->notes = $body->notes;

        $job->modified_date = date('Y-m-d H:i:s');
        $job->modified_user = $this->currentUser->id;
        if($job->isValid()){
            $job->store();
        }
        return System::asJson($response, ['success' => 'true', 'redirectTo' => '/jobs']);
    }

    public function jobAttachmentsData($request, $response, $data){
        $condition = ['job_ref' => ['job', $data['ref']]];
        $entities = (new attachmentEntity())->loadMultiple($condition, ['ref']);
        $dataArray = [];
        foreach($entities as $entity){
            $entityData = (object)$entity->getData();
            $createdUser = new userEntity();
            $createdUser->load($entityData->created_user);

            $document = new attachmentEntity();
            $document->load($entityData->ref);

            $downloadComponent = new componentEntity();
            $downloadComponent->name = 'DownloadButton';
            $downloadComponent->props = (object)[
                'ref' => $entityData->ref,
                'type' => 'attachments',
            ];

            $data = (object)[
                "Ref" => $entityData->ref,
                "Name" => $document->name,
                "Format" => $document->file_mime_attachment,
                "Created" => $document->created_date,
                "#" => (object)[
                    "component" => $downloadComponent->getData(),
                    'href' => '/documents/'.$entityData->ref.'/delete?type=attachments', 'title' => '<button class="btn btn-danger"><i class="fal fw fa-trash"></i></button>',
                ]
            ];
            $dataArray[] = $data;
        }
        $tableHeader = isset($dataArray[0]) ? array_keys((array)$dataArray[0]) : [];
        return System::asJson($response, ['rows' => $dataArray, 'tableHeader' => $tableHeader]);
    }

    public function jobAttachmentsForm($request, $response, $data){
        $form = new formEntity();
        $form->setId('managePos');
        $form->setMethod('post');
        $form->setActionUrl('jobs/attachments/'.$data['ref']);

        $form->addField((new inputField('name'))->setLabel('File Name'), 'Upload Job Attachment');
        $file = (new fileField('document'))->setLabel('File');
        $file->setFileSavePath($this->config->app->site->backend_domain.'/data/jobs/'.$data['ref'].'/saveFile?type=attachments');
        $form->addField($file, 'Upload Job Attachment');

        return System::asJson($response, $form->getBuiltFormArray());
    }

    public function jobAttachmentsSave(Request $request, $response, $data){
        //get the uploaded file
        $uploadedFile = $request->getUploadedFiles();
        //save the record to DB and return the ID
        $document = new attachmentEntity();
        $document->notes = '';
        $document->file_mime_attachment = $uploadedFile['file']->getClientMediaType();
        $document->name = $uploadedFile['file']->getClientFilename();//$body->name.'.'.$document->file_mime_attachment;
        $document->job = $data['ref'];
        $document->franchise = $this->currentUser->franchise;

        $document->created_date = date('Y-m-d H:i:s');
        $document->created_user = $this->currentUser->id;
        $document->franchise = $this->currentUser->franchise;

        $document->modified_date = date('Y-m-d H:i:s');
        $document->modified_user = $this->currentUser->id;

        $document->store();

        //move it to the upload Dir, which in this case is the document id
        Utils::moveUploadedFile('job_attachments', $uploadedFile['file']);

        $document->store();

        return System::asJson($response, ['fid' => $document->ref]); //'redirectTo' => '/jobs/'.$data['ref']]);
    }

    public function jobAttachmentsSaveComplete(Request $request, $response, $data){
        $body = $request->getParsedBody();

        $document = new attachmentEntity();
        $document->load($body->document);
        //rename the file so we can still download it.
        $newName = $body->name.'_'.$document->name;
        $renamed = Utils::renameFile('job_attachments', $document->name, $newName);
        if($renamed) {
            $document->name = $body->name . '_' . $document->name;
            $document->store();
        }

        return System::asJson($response, ['redirectTo' => '/jobs/'.$data['ref'].'?uploaded='.time()]);
    }

    public function deleteView($request, $response, $data){
        //First, we create the basic page
        $job = new jobEntity();
        if(isset($data['ref'])){
            $job->load($data['ref']);

            $this->page->setMetaTitle('Delete a Job');
            $this->page->setMetaDescription('Delete a Job.');

            //next add components, we assume the page already contains some components to only add to the content component.
            $contentTile = new componentEntity();
            $contentTile->name = 'GoHaulContentTile';
            $contentTile->props = (object)[
                "title" => 'Delete A Job',
                "content" => "Are you sure you want to delete this Job (j".$job->ref.")?",
                "buttons" => [
                    (object)[
                        "text" => 'Confirm Delete',
                        "href" => '/jobs/'.$job->ref,
                        "button" => 'btn-danger',
                        "type" => "delete"
                    ],
                ]
            ];

            $contentWrapper = new componentEntity();
            $contentWrapper->name = 'Content';
            $contentWrapper->classes = 'col-lg-10';
            $contentWrapper->addComponent($contentTile);

            $globalWrapper = $this->page->getComponent('ThePageContent');
            $globalWrapper->addComponent($contentWrapper);
            $this->page->replaceComponent('ThePageContent', $globalWrapper);

            return System::asJson($response, $this->page->getPage());
        } else {
            return System::asJson($response, ['success' => 'false'], 404);
        }
    }

    public function delete($request, $response, $data){
        (new jobEntity())->delete($data['ref']);
        return System::asJson($response, ['success' => 'true', 'redirectTo' => '/jobs']);
    }

    public function syncInvoices(Request $request, Response $response, $data){
        $params = $request->getQueryParams();

        //get the current users franchise
        $franchise = new franchiseEntity();
        $franchise->load($this->currentUser->franchise);

        // Sage One
        $sageOne = new SageOne($franchise);

        //check if we have params, if we do, process them.
        if(isset($params['code']) && isset($params['country'])){
            //got a code? get the auth token and store it against the franchise.
            $sageOne->getAuthToken($params['code'], 'Sync');
            $jobEntity = new jobEntity();
            $jobsToCheck = $jobEntity->isInvoiced();
            foreach($jobsToCheck as $jobData){
                $job = new jobEntity();
                $job->load($jobData->ref);
                $sageOne->sync($job);
            }
        } else {
            return System::asJson($response, ["redirectToExternal" => $sageOne->getAuthPath('Sync')]);
        }

        return System::asJson($response, ["success" => "true", "redirectTo" => "/jobs/".$job->ref]);
    }

    protected function buildFilterConditions($params){
        $filters = [];
        foreach($params as $param => $value){
            if($param == 'limit' || $param == 'offset' || $param == 'page' || empty($value)){
                continue;
            }
            switch($param){
                case 'search':
                    $filters['ref'] = ['ref', '%'.$value.'%', 'like', 'AND ('];
                    $filters['purchase_order'] = ['purchase_order', '%'.$value.'%', 'like', 'OR'];
                    $filters['vrid'] = ['vrid', '%'.$value.'%', 'like', 'OR'];
                    $filters['invoice_no'] = ['invoice_no', '%'.$value.'%', 'like', 'OR'];
                    $filters['location_1'] = ['location_1', '%'.$value.'%)', 'like', 'OR'];
                    $filters['location_2'] = ['location_1', '%'.$value.'%)', 'like', 'OR'];
                    $filters['location_3'] = ['location_1', '%'.$value.'%)', 'like', 'OR'];
                    $filters['location_4'] = ['location_1', '%'.$value.'%)', 'like', 'OR', ')'];
                    break;
                case 'company':
                    $filters['company'] = ['company', $value];
                    break;
                case 'driver':
                    $filters['driver'] = ['driver', $value];
                    break;
                case 'current_driver':
                    $filters['current_driver'] = ['driver', $this->currentUser->id];
                    break;
                case 'to_date':
                    $filters['to_date'] = ['created_date', $value, '<=', 'AND'];
                    break;
                case 'from_date':
                    $filters['from_date'] = ['created_date', $value, '>=', 'AND'];
                    break;
                case 'vehicle_type':
                    $filters['vehicle_type'] = ['vehicle_type', $value];
                    break;
                case 'vehicle':
                    $filters['vehicle'] = ['vehicle', $value];
                    break;
                case 'job_date':
                    $filters['job_date'] = ['job_date', $value];
                    break;
                case 'status':
                    $filters = $filters + $this->statusQueries($value);
                    break;
                case 'driver_status':
                    $filters['driver_status'] = ['driver_status', $value, '<='];
                    break;
            }
        }
        return $filters;
    }

    /**
     * "Pending" => "Pending",
     * "Ongoing" => "Ongoing",
     * "Overdue" => "Overdue",
     * "Complete" => "Complete",
     * "Ready to be invoiced" => "Ready to be invoiced",
     * "Invoiced" => "Invoiced"
     *
     * @param $status
     * @return array|string
     */
    protected function statusQueries($status){
        $query = [];
        switch($status){
            case 'Complete':
                return ['completed_date' => ['completed_date', 'NULL', 'IS NOT NULL']];
                break;
            case 'Invoiced': //!is_null($this->invoiced_amount) && (is_null($this->amount_paid) || $this->amount_paid < $this->invoiced_amount):
                $query['invoiced_1'] = ['invoiced_amount', 'NULL', 'IS NOT NULL'];
                $query['invoiced_2'] = ['amount_paid', 0.00];
                $query['completed_date'] = ['completed_date', 'NULL', 'IS NULL'];

                return $query;
                break;
            case 'Ready to be invoiced':
                $query['invoiced_1'] = ['invoiced_amount', 'NULL', 'IS NULL'];
                $query['completed_date'] = ['completed_date', 'NULL', 'IS NULL'];
                $query['confirmed_date'] = ['confirmed_date', 'NULL', 'IS NOT NULL'];
                return $query;
                break;
            case 'Pending':
                return ['job_date' => ['job_date', time(), '>']];
                break;
            case 'Ongoing':
                return ['job_date' => ['job_date', strtotime(date('Y-m-d'))]];
                break;
            case 'Overdue':
                return ['job_date' => ['job_date', strtotime(date('Y-m-d')), '<']];
                break;
        }
        return 'Unknown';
    }
}