<?php

namespace apexl\Io\modules\gohaul\controllers;

use apexl\hashing\Hash;
use apexl\Io\includes\System;
use apexl\Io\modules\component\entities\componentEntity;
use apexl\Io\modules\formbuilder\entities\formEntity;
use apexl\Io\modules\formbuilder\includes\checkboxField;
use apexl\Io\modules\formbuilder\includes\inputField;
use apexl\Io\modules\formbuilder\includes\selectField;
use apexl\Io\modules\page\services\Page;
use apexl\Io\modules\user\entities\companiesEntity;
use apexl\Io\modules\user\entities\franchiseEntity;
use apexl\Io\modules\user\entities\roleEntity;
use apexl\Io\modules\user\entities\userEntity;
use apexl\Io\modules\user\services\currentUser;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;

class usersController{

    protected $page;
    /** @var userEntity */
    protected $currentUser;
    public function __construct(Page $page, currentUser $currentUser){
        $this->page = $page;
        $this->currentUser = $currentUser::getCurrentUser();
    }

    public function __invoke(Request $request, Response $response, $args)
    {
        $method = $request->getMethod();
        return $this->$method($request, $response, $args);
    }

    public function list(Request $request, Response $response, $args){
        //First, we create the basic page
        $this->page->setMetaTitle('View Users List');
        $this->page->setMetaDescription('A list of all users');

        //next add components, we assume the page already contains some components to only add to the content component.
        $contentTile = new componentEntity();
        $contentTile->name = 'GoHaulContentTile';
        $contentTile->props = (object)[
            "title" => 'User Manager',
            "content" => "In here you can make changes to the users. Remember any changes you make will update instantly on the website.",
            "help" => 'Please select the user from the list below you wish to update or create a new one.',
            "buttons" => [
                (object)[
                    "text" => 'Create New User',
                    "href" => '/users/create',
                    "button" => 'btn-primary'
                ]
            ]
        ];

        $filterForm = new componentEntity();
        $filterForm->name = 'FilterForm';
        $filterForm->addProperty('dataSrc', '/data/users/filterForm');


        $DataTable = new componentEntity();
        $DataTable->name = 'DataTable';
        $DataTable->addProperty('dataSrc', '/data/users/list');


        $contentWrapper = new componentEntity();
        $contentWrapper->name = 'Content';
        $contentWrapper->classes = 'col-lg-10';
        $contentWrapper->addComponent($contentTile);
        $contentWrapper->addComponent($filterForm);
        $contentWrapper->addComponent($DataTable);

        $globalWrapper = $this->page->getComponent('ThePageContent');
        $globalWrapper->addComponent($contentWrapper);
        $this->page->replaceComponent('ThePageContent', $globalWrapper);

        return System::asJson($response, $this->page->getPage());
    }

    public function account(Request $request, Response $response, $args){
        return $this->update($request, $response, ['ref' => $this->currentUser->id, 'dataSrc' => '/data/users/'.$this->currentUser->id.'/accountForm']);
    }

    public function accountUpdate(Request $request, Response $response, $args){
        //bypass the update permission requirement when dealing with our own account
        $args['actionUrl'] = 'account/'.$this->currentUser->id.'/update';
        return $this->manageForm($request, $response, $args);
    }

    public function accountCreateUpdate(Request $request, Response $response, $args){
        $args['redirectTo'] = '/';
        return $this->createUpdate($request, $response, $args);
    }

    public function view(Request $request, Response $response, $args){
        if(!isset($args['ref'])){
            return System::asJson($response, ['message' => '404 not found'], 404);
        }
        $user = new userEntity();
        $user->load($args['ref']);

        if(!isset($user->id)){
            return System::asJson($response, ['message' => '404 not found'], 404);
        }

        //First, we create the basic page
        $this->page->setMetaTitle($user->first_name.' '.$user->last_name);
        $this->page->setMetaDescription($user->first_name.' '.$user->last_name);

        //next add components, we assume the page already contains some components to only add to the content component.
        $contentTile = new componentEntity();
        $contentTile->name = 'GoHaulContentTile';
        $contentTile->props = (object)[
            "title" => $user->first_name.' '.$user->last_name,
            "content" => "",
            "buttons" => [
                (object)[
                    "text" => 'Edit This User',
                    "href" => '/users/'.$user->id.'/edit',
                    "button" => 'btn-primary'
                ],
                (object)[
                    "text" => 'User List',
                    "href" => '/users',
                    "button" => 'btn-secondary'
                ],
            ]
        ];

        //make it very hard to delete the anon user.
        if($user->id != 0){
            $contentTile->props->buttons[] = (object)[
                "text" => 'Delete This User',
                "href" => '/users/'.$user->id.'/delete',
                "button" => 'btn-danger'
            ];
        }

        $contentRows = [];
        $contentRows[] = ['Email Address', $user->email];
        $contentRows[] = ['Last Login', $user->last_login];

        $contentDetailsTable = new componentEntity();
        $contentDetailsTable->name = 'ContentTable';
        $contentDetailsTable->classes = 'col-md-4';
        $contentDetailsTable->title = 'Details';
        $contentDetailsTable->rows = $contentRows;

        $contentRows = [];
        $contentRows[] = ['Can Login', $user->active ? 'Yes' : 'No'];
        $contentRows[] = ['Roles', ''];
        if(is_array($user->roles)) {
            foreach ($user->roles as $role) {
                $roleEntity = new roleEntity();
                $roleEntity->load($role);
                $contentRows[] = ['', $roleEntity->name];
            }
        }

        $contentSecurityTable = new componentEntity();
        $contentSecurityTable->name = 'ContentTable';
        $contentSecurityTable->classes = 'col-md-4';
        $contentSecurityTable->title = 'Security';
        $contentSecurityTable->rows = $contentRows;

        $createdBy = new userEntity();
        $createdBy->load($user->modified_by);

        $modifiedBy = new userEntity();
        $modifiedBy->load($user->modified_by);

        $contentRows = [];
        $contentRows[] = ['Created By', $createdBy->first_name.' '.$createdBy->last_name];
        $contentRows[] = ['Created Date', $user->created];
        $contentRows[] = ['Modified By', $modifiedBy->first_name.' '.$modifiedBy->last_name];
        $contentRows[] = ['Modified Date', $user->modified];

        $contentHistoryTable = new componentEntity();
        $contentHistoryTable->name = 'ContentTable';
        $contentHistoryTable->classes = 'col-md-4';
        $contentHistoryTable->title = 'History';
        $contentHistoryTable->rows = $contentRows;

        $rowWrapper = new componentEntity();
        $rowWrapper->name = 'RowWrapper';
        $rowWrapper->addComponent($contentDetailsTable);
        $rowWrapper->addComponent($contentSecurityTable);
        $rowWrapper->addComponent($contentHistoryTable);

        $contentWrapper = new componentEntity();
        $contentWrapper->name = 'Content';
        $contentWrapper->classes = 'col-lg-10';
        $contentWrapper->addComponent($contentTile);
        $contentWrapper->addComponent($rowWrapper);

        $globalWrapper = $this->page->getComponent('ThePageContent');
        $globalWrapper->addComponent($contentWrapper);
        $this->page->replaceComponent('ThePageContent', $globalWrapper);

        return System::asJson($response, $this->page->getPage());
    }

    public function create(Request $request, Response $response, $args){
        //First, we create the basic page
        $this->page->setMetaTitle('Create a PO');
        $this->page->setMetaDescription('Create a PO.');

        //next add components, we assume the page already contains some components to only add to the content component.
        $contentTile = new componentEntity();
        $contentTile->name = 'GoHaulContentTile';
        $contentTile->props = (object)[
            "title" => 'Create New User',
            "content" => "Please enter the details of your new user and click \"Submit\".",
            "buttons" => [
                (object)[
                    "text" => 'User List',
                    "href" => '/users',
                    "button" => 'btn-secondary'
                ],
            ]
        ];

        $form = new componentEntity();
        $form->name = 'Form';
        $form->addProperty('dataSrc', '/data/users/createForm');


        $contentWrapper = new componentEntity();
        $contentWrapper->name = 'Content';
        $contentWrapper->classes = 'col-lg-10';
        $contentWrapper->addComponent($contentTile);
        $contentWrapper->addComponent($form);

        $globalWrapper = $this->page->getComponent('ThePageContent');
        $globalWrapper->addComponent($contentWrapper);
        $this->page->replaceComponent('ThePageContent', $globalWrapper);

        return System::asJson($response, $this->page->getPage());
    }

    public function update(Request $request, Response $response, $args){
        //First, we create the basic page
        $user = new userEntity();
        $user->load($args['ref']);

        $this->page->setMetaTitle('Create a PO');
        $this->page->setMetaDescription('Create a PO.');

        //next add components, we assume the page already contains some components to only add to the content component.
        $contentTile = new componentEntity();
        $contentTile->name = 'GoHaulContentTile';
        $contentTile->props = (object)[
            "title" => 'Edit '.$user->first_name.' '.$user->last_name,
            "content" => "Please change the user as required and click \"Submit\".",
            "buttons" => [
                (object)[
                    "text" => 'Create New User',
                    "href" => '/users/create',
                    "button" => 'btn-primary'
                ],
                (object)[
                    "text" => 'Users List',
                    "href" => '/users',
                    "button" => 'btn-secondary'
                ],
            ]
        ];

        $form = new componentEntity();
        $form->name = 'Form';
        if(isset($args['dataSrc'])){
            $form->addProperty('dataSrc', $args['dataSrc']);
        } else {
            $form->addProperty('dataSrc', '/data/users/' . $args['ref'] . '/updateForm');
        }


        $contentWrapper = new componentEntity();
        $contentWrapper->name = 'Content';
        $contentWrapper->classes = 'col-lg-10';
        $contentWrapper->addComponent($contentTile);
        $contentWrapper->addComponent($form);

        $globalWrapper = $this->page->getComponent('ThePageContent');
        $globalWrapper->addComponent($contentWrapper);
        $this->page->replaceComponent('ThePageContent', $globalWrapper);

        return System::asJson($response, $this->page->getPage());
    }

    public function filterForm(Request $request, Response $response, $args){
        $form = new formEntity();
        $form->setId('userFilter');
        $form->setMethod('get');

        $form->addField((new checkboxField('active_no'))->setLabel('Disabled'), 'filter');
        $form->addField((new checkboxField('active_yes'))->setLabel('Enabled'), 'filter');

        $form->addField((new inputField('search'))->setLabel('Search'), 'filter');
        return System::asJson($response, $form->getBuiltFormArray());
    }

    public function manageForm(Request $request, Response $response, $args)
    {
        $user = FALSE;
        if (isset($args['ref'])) {
            $user = new userEntity();
            $user->load($args['ref']);
        }

        $form = new formEntity();
        $form->setId('manageUser');
        $form->setMethod('post');
        $form->setActionUrl('users');
        if($user){
            if(isset($args['actionUrl'])){
                $form->setActionUrl($args['actionUrl']);
            } else {
                $form->setActionUrl('users/' . $args['ref']);
            }
        }

        $first_name = (new inputField('first_name'))->setLabel('First Name');
        if($user){
            $first_name->setValue($user->first_name);
        }
        $form->addField($first_name, 'details');

        $last_name = (new inputField('last_name'))->setLabel('Last Name');
        if($user){
            $last_name->setValue($user->last_name);
        }
        $form->addField($last_name, 'details');

        $email = (new inputField('email'))->setLabel('Email Address');
        if($user){
            $email->setValue($user->email);
        }
        $form->addField($email, 'details');

        if($this->currentUser->isAllowed('CreateFranchises')){
            $franchiseEntity = new franchiseEntity();
            //remove the access checks for franchise on this pass so we can load them all.
            $franchiseEntity->changeAccessCheckStatus(FALSE);
            $franchises = $franchiseEntity->loadMultiple();
            $franchise = (new selectField('franchise'))->buildOptionsFromEntities($franchises, 'ref', 'name')->setLabel('Franchise')->isRequired();
            if($user){
                $franchise->setValue($user->franchise);
            }

            $form->addField($franchise, 'security');
        }
        $password = (new inputField('password'))->setLabel('Password');
        $password->setInputType('Password');
        $form->addField($password, 'security');


        $roles = (new roleEntity())->loadMultiple();
        foreach ($roles as $role) {
            //for users that can't change roles, show labels for those that they do have.
            $showLabel = FALSE;
            if($this->currentUser->isAllowed('ViewRoles')) {
                $roleField = (new checkboxField('role_' . $role->rid))->setLabel($role->name);
            } else if($this->currentUser->isAllowed('AssignDriver') && $role->rid == 6){
                //HARDCODED DRIVER RID
                $roleField = (new checkboxField('role_' . $role->rid))->setLabel($role->name);
            } else {
                $showLabel = TRUE;
                $roleField = (new inputField('role_' . $role->rid));
                $roleField->setInputType('hidden');
            }
            if($user && is_array($user->roles)) {
                if ($user && in_array($role->rid, $user->roles)) {
                    //allows us to show the label only for those without the role to change.
                    //Only shows labels for the roles they have.
                    if ($showLabel) {
                        $roleField->setLabel($role->name);
                    }
                    $roleField->setValue($role->rid);
                }
            }
            $form->addField($roleField, 'roles');
        }

        return System::asJson($response, $form->getBuiltFormArray());
    }

    public function dataUserList($request, $response, $data){
        $params = $request->getQueryParams();

        $limit = isset($params['limit']) && !empty($params['limit']) ? $params['limit'] : 0;
        $offset = isset($params['offset']) && !empty($params['offset']) ? $params['offset'] : 0;

        $filters = $this->buildFilterConditions($params);

        if(!$this->currentUser->isAllowed('ViewFranchises')){
            $filters['franchise'] = ['franchise', $this->currentUser->franchise];
        }

        $users = (new userEntity())->loadMultiple($filters, [], $limit, $offset);
        $totalResults = (new userEntity())->totalEntities();
        $totalPages = ceil (($totalResults/$params['limit']));
        $dataArray = [];
        foreach($users as $user) {
            $franchise = new franchiseEntity();
            $franchise->load($user->franchise);
            $user->franchise = $franchise->name;
            $data = [];
            $data["Ref"] = $user->id;
            $data["First Name"] = $user->first_name;
            $data["Last Name"] = $user->last_name;
            $data["Email"] = $user->email;
            $data["Created"] = $user->created;
            $data["Modified"] = $user->modified;
            $data["Last Login"] = $user->last_login;
            $userRoles = unserialize($user->roles);
            $roles = [];
            if (!empty($userRoles)) {
                foreach ($userRoles as $rid) {
                    //account for some blank entries in old data.
                    if($rid) {
                        $role = new roleEntity();
                        $role->load($rid);
                        $roles[] = $role->name;
                    }
                }
            }
            $data["Roles"] = implode(', ', $roles);
            $data["Franchise"] = $user->franchise;
            $data["Can Login"] = $user->active ? 'Yes' : 'No';
            $dataArray[] = (object)$data;
        }
        $tableHeader = isset($dataArray[0]) ? array_keys((array)$dataArray[0]) : [];
        return System::asJson($response, ['rows' => $dataArray, 'totalData' => $totalResults, 'totalPages' => $totalPages, 'tableHeader' => $tableHeader]);
    }

    public function createUpdate($request, $response, $data){
        $user = new userEntity();
        //check if this is an update or create.
        if(isset($data['ref'])){
            $user->load($data['ref']);
        } else {
            $user->created = date('Y-m-d H:i:s', time());
            $user->created_by = $this->currentUser->id;
        }
        $body = $request->getParsedBody();

        $user->modified = date('Y-m-d H:i:s', time());
        $user->modified_by = $this->currentUser->id;


        $user->first_name = $body->first_name ?? '';
        $user->last_name = $body->last_name ?? '';
        $user->email = $body->email ?? '';

        unset($body->first_name, $body->last_name, $body->email);

        //if we have the password, deal with the salt and the hash
        if($body->password && !empty($body->password)){
            $hash = new Hash();
            $hashData = $hash->hashString($body->password);
            $user->password = $hashData->hash;
            $user->salt = $hashData->salt;
        }
        unset($body->password);

        if(isset($body->franchise) && $this->currentUser->isAllowed('CreateFranchises')){
            $user->franchise = $body->franchise;
        }
        unset($body->franchise);

        $rids = [];
        foreach($body as $role => $set){
            if(!empty($set)) {
                $rids[] = str_replace('role_', '', $role);
            }
        }
        $user->roles = $rids;

        $redirectTo = isset($data['redirectTo']) ? $data['redirectTo'] : '/users';
        if($user->isValid()){
            $user->store();
            return System::asJson($response, ['success' => 'true', 'redirectTo' => $redirectTo]);
        }
        return System::asJson($response, ['success' => 'false', 'redirectTo' => $redirectTo], 400);
    }

    public function deleteView($request, $response, $data){
        //First, we create the basic page
        $user = new userEntity();
        if(isset($data['ref'])){
            $user->load($data['ref']);

            $this->page->setMetaTitle('Delete a User');
            $this->page->setMetaDescription('Delete a User.');

            //next add components, we assume the page already contains some components to only add to the content component.
            $contentTile = new componentEntity();
            $contentTile->name = 'GoHaulContentTile';
            $contentTile->props = (object)[
                "title" => 'Delete A User',
                "content" => "Are you sure you want to delete this User (".$user->first_name.' '.$user->last_name.")?",
                "buttons" => [
                    (object)[
                        "text" => 'Confirm Delete',
                        "href" => '/users/'.$user->id,
                        "button" => 'btn-danger',
                        "type" => "delete"
                    ],
                ]
            ];

            $contentWrapper = new componentEntity();
            $contentWrapper->name = 'Content';
            $contentWrapper->classes = 'col-lg-10';
            $contentWrapper->classes = 'col-lg-10';
            $contentWrapper->addComponent($contentTile);

            $globalWrapper = $this->page->getComponent('ThePageContent');
            $globalWrapper->addComponent($contentWrapper);
            $this->page->replaceComponent('ThePageContent', $globalWrapper);

            return System::asJson($response, $this->page->getPage());
        } else {
            return System::asJson($response, ['success' => 'false'], 404);
        }
    }

    public function delete($request, $response, $data){
        (new userEntity())->delete($data['ref']);
        return System::asJson($response, ['success' => 'true', 'redirectTo' => '/users']);
    }

    protected function buildFilterConditions($params){
        $filters = [];
        foreach($params as $param => $value){
            if($param == 'limit' || $param == 'offset' || $param == 'page' || empty($value)){
                continue;
            }
            switch($param){
                case 'search':
                    $filters['id'] = ['id', '%'.$value.'%', 'like', 'AND ('];
                    $filters['first_name'] = ['first_names', '%'.$value.'%', 'like', 'OR'];
                    $filters['last_name'] = ['last_name', '%'.$value.'%', 'like', 'OR'];
                    $filters['email'] = ['email', '%'.$value.'%)', 'like', 'OR', ')'];
                    break;
                case 'active_no':
                    $filters['active_no'] = ['active', 0];
                    break;
                case 'active_yes':
                    $filters['active_yes'] = ['active', 1];
                    break;
            }
        }
        return $filters;
    }
}