<?php
namespace apexl\io\moduels\gohaul\services;

use \apexl\Config\Singleton;
use apexl\Io\modules\user\entities\companiesEntity;
use apexl\Io\modules\user\entities\contactEntity;
use apexl\Io\modules\user\entities\franchiseEntity;
use apexl\Io\modules\user\entities\jobEntity;
use apexl\Io\modules\user\entities\purchaseOrderEntity;
use apexl\Io\services\pathUtility;
use apexl\io\moduels\gohaul\services\eventLogger;
use n1ghteyes\apicore\client;

class SageOne {
    // Authorisation
    private $client_id             = 'YOUR_CLIENT_ID';
    private $client_secret         = 'YOUR_CLIENT_SECRET';
    private $signing_secret        = 'YOUR_SIGNING_SECRET';
    private $subscription_id       = 'YOUR_SUBSCRIPTION_ID';
    private $access                = 'read_only';

    // URLS
    public  $default_return_url    = '';
    private $auth_endpoint         = 'https://www.sageone.com/oauth2/auth/central';
    private $token_endpoint        = 'https://app.sageone.com/oauth2/token';
    private $base_endpoint         = 'https://api.sageone.com/ENDPOINT/v3/';

    // Security
    private $access_token          = '';
    private $refresh_token         = '';
    private $resource_owner_id     = '';

    // Definitions
    private $MODES = array(
        'GET',
        'POST',
        'PUT',
        'DELETE',
    );

    public $LOCATIONS              = 4;

    /**
     * @var Singleton
     */
    protected $config;

    /**
     * @var franchiseEntity
     */
    protected $franchiseEntity;

    /**
     * @var client
     */
    protected $tokenApi;
    /**
     * @var client
     */
    protected $columbusApi;


    /**************************************************************************
     * Constructor
     */
    public function __construct(franchiseEntity $franchiseEntity)
    {
        // Set defaults
        $this->config = Singleton::getInstance();
        $this->franchiseEntity = $franchiseEntity;
        $this->tokenApi = new client();
        $this->columbusApi = new client();

        $this->default_return_url = $this->config->app->sage->secure_base_url;
        $this->defaults();
    }


    /**************************************************************************
     * Load and set the defaults
     */
    public function defaults()
    {
        // Access
        $this->client_id          = $this->franchiseEntity->sage_client_id;
        $this->client_secret      = $this->franchiseEntity->sage_client_secret;
        $this->signing_secret     = $this->franchiseEntity->sage_signing_secret;
        $this->subscription_id    = $this->franchiseEntity->sage_subscription_id;
        $this->access             = 'full_access';

        // URLS
        $this->auth_endpoint      =  $this->config->app->sage->auth_endpoint;
        $this->token_endpoint     =  $this->config->app->sage->token_endpoint;
        $this->base_endpoint      =  $this->config->app->sage->base_endpoint;
    }

    public function getAuthPath($type = 'Job'){
        //the Auth URL
        if($type == 'Job'){
            $redirectUrl = $this->cleanJobUrl();
        } else {
            $redirectUrl = $this->cleanSyncUrl();
        }
        return $this->auth_endpoint.'?filter=apiv3.1&response_type=code&client_id='.$this->client_id.'&scope='.$this->access.'&redirect_uri='.$redirectUrl;
    }

    protected function cleanJobUrl(){
        $path = pathUtility::getInstance();
        return $this->config->app->site->frontend_domain.'/jobs/'.$path->getPath(1).'/invoice';
    }

    protected function cleanSyncUrl(){
        return $this->config->app->site->frontend_domain.'/invoices/sync';
    }

    public function getAuthToken($code, $type = 'Job'){
        $path = pathUtility::getInstance();
        $this->tokenApi->setServer($this->token_endpoint);
        $this->tokenApi->setBodyFormat('form');
        $logger = new eventLogger();
        $this->tokenApi->addLogger($logger);

        if($type == 'Job'){
            $redirectUrl = $this->cleanJobUrl();
        } else {
            $redirectUrl = $this->cleanSyncUrl();
        }

        $response = $this->tokenApi->POST->oauth2->token([
            'client_id'     => $this->client_id,
            'client_secret' => $this->client_secret,
            'code'          => $code,
            'grant_type'    => 'authorization_code',
            'redirect_uri'  => $redirectUrl,
        ]);
        $logger->store($path->getPath(1));

        $this->access_token      = $response->data->access_token;
        $this->refresh_token     = $response->data->refresh_token;
        $this->resource_owner_id = $response->data->resource_owner_id;
    }


    /**************************************************************************
     * Executes a query
     */
    public function exec($mode, $scope, $command, $data = array(), $endpoint = 'Accounts')
    {

        // Construct the items we need for the query
        $this->ksort_recursive($data);

        $path = pathUtility::getInstance();
        $this->columbusApi->resetPath();
        $this->columbusApi->setServer($this->config->app->sage->base_endpoint);
        $this->columbusApi->auth('Authorization', 'Bearer '.$this->access_token, 'header');
        $this->columbusApi->auth('X-Site', $this->resource_owner_id, 'header');
        $this->columbusApi->auth('Ocp-Apim-Subscription-Key', $this->subscription_id, 'header');
        $this->columbusApi->setBodyFormat('json');
        $logger = new eventLogger();
        $this->columbusApi->addLogger($logger);
        $response = $this->columbusApi->$mode->uki->sageone->$scope->{'v3'}->$command($data);
        $logger->store($path->getPath(1));
        return $response;
    }


    /**************************************************************************
     * Recursively key sort an array
     */
    protected function ksort_recursive(&$array)
    {
        if(!is_array($array)) return $array;
        foreach($array as $key => $value)
            $this->ksort_recursive($array[$key]);
        ksort($array);
    }

    public function invoice(jobEntity $job)
    {
        // Fetch the model$job
        if(!$job->ref) {
            return FALSE;
        }

        // Check the jobs status
        if($job->status != 'Ready to be invoiced' || $job->invoice_no || !$job->cost) {
            return FALSE;
        }

        // Fetch the purchase order
        $purchaseOrder = new purchaseOrderEntity();
        $purchaseOrder->load($job->purchase_order);

        // Fetch the company
        $company = new companiesEntity();
        $company->load($job->company);
        if(!$company->ref) {
            return FALSE;
        }

        // Fetch the invoice contact
        $contact = new contactEntity();
        $contact->loadAccountant($job->company);

        // Sage One
        $franchise = new franchiseEntity();
        $franchise->load($job->franchise);

        // Get the ledger ID from the nominal code, if we dont already have one.
        if(isset($franchise->sage_nominalCode) && empty($franchise->sage_ledger_id)){
            $response = $this->exec('GET', 'accounts', 'ledger_accounts', ['items_per_page' => 100]);
            foreach($response->data->{'$items'} as $ledgerAccount){
                if(strpos($ledgerAccount->displayed_as, $franchise->sage_nominalCode) !== FALSE){
                    $franchise->sage_ledger_id = $ledgerAccount->id;
                    $franchise->store();
                }
            }
        }

        // Get the company/contact from sage
        $response = $this->exec('GET', 'accounts', 'contacts', array(
            'contact_type_id' => 'CUSTOMER',
            'search' => $company->full_name,
        ));
        if(!isset($response->data->{'$total'})) {
            return FALSE;
        }

        if($response->data->{'$total'} > 1)
        {
            // Create matches list
            $matches = [];
            if(isset($response->data->{'$items'})) {
                foreach ($response->data->{'$items'} as $item) {
                    $item = (array)$item; //type case to fix SageOne fuckery since we now get objects.. WHY!?
                    if (isset($item['displayed_as']) && isset($item['id'])) {
                        $matches[$item['id']] = $item['displayed_as'];
                    }
                }
            }

            // If we have an exact match use that
            $match = array_search($company->full_name, $matches);
            if($match !== FALSE) {
                $contact_id = $match;
            } else {
                return FALSE;
            }

        }
        elseif($response->data->{'$total'})
        {

            // Get the name_and_company_name of the existing company/contact
            if(!isset($response->data->{'$items'}[0]->id)) {
                return FALSE;
            }
            $contact_id = $response->data->{'$items'}[0]->id;

        }
        else
        {

            // Create a new company/contact
            $company_types = array();
            $contact_types = array();
            if($company->type_customer)
            {
                $company_types[] = 'CUSTOMER';
                if($contact->role_transport)   $contact_types[] = 'PURCHASING';
            }

            if($company->type_haulier)
            {
                $company_types[] = 'VENDOR';
                if($contact->role_transport)   $contact_types[] = 'SALES';
            }

            if($contact->role_accountant)  $contact_types[] = 'ACCOUNTS';


            // Due to an issue with sage types must have only one entry
            if(count($company_types) != 1)
                $company_types = array('CUSTOMER');
            if(count($contact_types) != 1)
                $contact_types = array('ACCOUNTS');


            $sageContact = array('contact' => array(

                'name'                     => $company->full_name,
                'contact_type_ids'         => $company_types,
                'notes'                    => 'Automatically created by the Go-Haul Booking System (originally for job J'.$job->ref.')',
                'main_address'             => array('address_type_id' => 'ACCOUNTS') + $this->sage_address_array($company),

                'main_contact_person'      => array(
                    'contact_person_type_ids'  => $contact_types,
                    'name'                     => trim($contact->first_names.' '.$contact->last_name),
                    'telephone'                => $contact->office_phone,
                    'mobile'                   => $contact->mobile_phone,
                    'email'                    => $contact->email_address,
                    'is_main_contact'          => TRUE,
                ),

            ));

            if($company->vat_no) {
                $sageContact['contact']['tax_number'] = substr(trim(strtr($company->vat_no, array('GB' => '', ' ' => '', '-' => '', ':' => ''))), 0, 14);
            }

            // Strip blank fields
            foreach($sageContact as $key => $value)
                if(!is_array($value) && $value == '')
                    unset($sageContact[$key]);

            // Create it
            $response = $this->exec('POST', 'accounts','contacts', $sageContact);
            if(!isset($response->data->id)) {
                return FALSE;
            }

            $contact_id = $response->data->id;
        }

        // Specify the invoice
        $invoice = array('sales_invoice' => array(

            'contact_id'               => $contact_id,
            'date'                     => date('d/m/Y'),

            'invoice_lines' => array( array(
                'description'            => 'Haulage on '.date('d/m/Y', strtotime($job->job_date)),
                'ledger_account_id'      => $franchise->sage_ledger_id,
                'quantity'               => 1,
                'unit_price'             => $job->cost,
                'net_amount'             => $job->cost,
                'tax_rate_id'            => 'GB_STANDARD',
                'tax_amount'             => number_format($this->config->app->sage->vat_rate * $job->cost, 2, '.', ''),
            )),

            'main_address'             => array('address_type_id' => 'ACCOUNTS') + $this->sage_address_array($company),
            'due_date'                 => date('d/m/Y', strtotime('30 days')),
            'reference'                => $franchise->short_name . "/" . date('ym', strtotime($purchaseOrder->created_date)) . '/' . $purchaseOrder->ref,
            'notes'                    => implode("\n", $this->locations_array($job)),

        ));

        // Create the invoice
        $response = $this->exec('POST', 'accounts', 'sales_invoices', $invoice);
        if(!isset($response->data->invoice_number)) {
            return FALSE;
        }

        // Update the job
        $job->invoiced_date = date('Y-m-d');
        $job->invoice_no = $response->data->invoice_number;
        $job->invoiced_amount = number_format($response->data->total_amount, 2, '.', '');
        $job->amount_paid = number_format($response->data->total_paid, 2, '.', '');
        if($job->isValid()) {
            $job->store();
            return $job;
        }
        return FALSE;
    }

    /**
     * @param jobEntity $job
     * @return bool
     */
    public function sync(jobEntity $job)
    {
        // Search for the invoice
        $response = $this->exec('GET', 'accounts', 'sales_invoices', array('search' => $job->invoice_no));
        if (!isset($response->data->{'$total'})) {
            return FALSE;
        }
        if(is_array($response->data->{'$items'})) {
            foreach ($response->data->{'$items'} as $item) {
                if (isset($item['displayed_as']) && $item['displayed_as'] == $job->invoice_no && isset($item['id']) && $item['id']) {
                    $response = $this->exec('GET', 'accounts', 'sales_invoices/' . $item['id']);
                    if (!isset($response->data->id)) {
                        return FALSE;
                    }
                    if (isset($response->data->total_paid) && isset($response->data->total_amount)) {
                        $job->invoiced_amount = number_format($response->data->total_amount, 2, '.', '');
                        $job->amount_paid = number_format($response->data->total_paid, 2, '.', '');
                        $job->completed_date = date('Y-m-d');

                        if ($job->isValid()) {
                            $job->store();
                            return TRUE;
                        }
                    }
                }
            }
        }
        return FALSE;
    }

    protected function sage_address_array($company)
    {
        $address = array();

        $address['address_line_1'] = $company->address_1 ?? '';
        $address['address_line_2'] = $company->address_2 ?? '';
        $address['city']           = $company->city ?? '';
        $address['postcode']       = $company->postcode ?? '';
        $address['region']         = $company->state ?? '';
        $address['country_id']     = $company->country ?? '';

        return $address;
    }

    protected function locations_array($job)
    {
        $locations = array();

        for($i = 1; $i <= $this->LOCATIONS; $i++)
            if(mb_strlen($job->{'location_'.$i})) $locations['location_'.$i] = $job->{'location_'.$i};

        return $locations;
    }
}


/* End */