<?php
namespace apexl\Io\modules\display\components;

use apexl\Io\includes\Entity;
use apexl\Io\includes\Routes;
use apexl\Io\modules\component\entities\componentEntity;

class EntityDisplayTable extends componentEntity {

    protected $dataSrc;
    protected $filters;
    protected $entity;
    protected $hideDisabled = FALSE;

    public function __construct(Entity $entity = null, $replacements = [])
    {
        parent::__construct();
        $this->setName("DataTable");
        if($entity){
            $this->setEntity($entity, $replacements);
        }
        $this->displayPagination();
    }

    public function setEntity(Entity $entity, $replacements = []){
        $this->entity = $entity;
        $this->dataSrc = Routes::getRoutePattern($entity->getEntityName().'.get', $replacements);
        $this->src($this->dataSrc);
        return $this;
    }

    public function setFilters($filters){
        $this->filters = $filters;
        $this->src($this->dataSrc);
        return $this;
    }

    public function showCols(array $cols): EntityDisplayTable
    {
        $this->addProperty('showCols', $cols);
        return $this;
    }

    public function hideCols(array $cols): EntityDisplayTable
    {
        $this->addProperty('hideCols', $cols);
        return $this;
    }

    public function displayPagination($display = TRUE){
        $this->addProperty('showPagination', $display);
        return $this;
    }

    public function displayShortTotals($display = TRUE){
        $this->addProperty('displayShortTotals', $display);
        return $this;
    }

    public function colspans($colspans = []){
        $this->addProperty('colspans', $colspans);
        return $this;
    }

    public function hideIdColumn($display = TRUE){
        $this->addProperty('hideIdColumn', $display);
        return $this;
    }

    public function src($dataSrc){
        $this->dataSrc = $dataSrc;
        $src = $this->dataSrc ?? '';
        if($this->filters){
            $src .= '?'.ltrim($this->filters, '?').'&asTable=1';
        } else {
            $src .= '?asTable=1';
        }
        $this->addProperty('dataSrc', $src);
        return $this;
    }

    /**
     * @return $this
     */
    public function addCommonButtons(): EntityDisplayTable
    {
        $view = $this->entity->getEntityName() . '.display.get';
        $edit = $this->entity->getEntityName() . '.display.put';
        $delete = $this->entity->getEntityName() . '.display.delete';
        $this->addViewButton($view);
        $this->addEditButton($edit);
        $this->addDeleteButton($delete);
        return $this;
    }

    /**
     * @param $routeName
     * @return $this
     */
    public function addViewButton($routeName): EntityDisplayTable
    {
        $this->addButton($routeName, 'eye', $this->getAccessFromRoute($routeName));
        return $this;
    }

    /**
     * @param $routeName
     * @return $this
     */
    public function addEditButton($routeName): EntityDisplayTable
    {
        $this->addButton($routeName, 'edit', $this->getAccessFromRoute($routeName));
        return $this;
    }

    /**
     * @param $routeName
     * @return $this
     */
    public function addDeleteButton($routeName): EntityDisplayTable
    {
        $this->addButton($routeName, 'trash', $this->getAccessFromRoute($routeName));
        return $this;
    }

    /**
     * @param $routeName
     * @param array $replacements
     * @param string $icon
     * @param false $disabled
     * @return $this
     */
    public function addPostButton($routeName, array $replacements = [], string $icon ='',bool $disabled = NULL): EntityDisplayTable
    {
        if(is_null($disabled)){
            $disabled = $this->getAccessFromRoute($routeName);
        }
        $button = (new TableButton())
            ->addRoute(Routes::getRoutePattern($routeName, $replacements, FALSE))
            ->addIcon($icon)
            ->isPost()
            ->addClass($this->entity->getEntityTrueName().'_table__btn')
            ->addClass($this->entity->getEntityTrueName().'_table__btn-'.$icon)
            ->addClass('table__btn')
            ->addClass('table__btn-'.$icon);
        if($disabled) {
            $button->addRoute('#');
            $button->addClass($this->entity->getEntityTrueName() . '_table__btn-disabled');
            $button->addClass('table__btn-disabled');
            $button->disabled($disabled);
            if($this->hideDisabled){
                //do not add the button to the component
                return $this;
            }
        }
        $this->addComponent($button);
        return $this;
    }

    /**
     * @param $routeName
     * @param string $icon
     * @param false $disabled
     * @return $this
     */
    public function addButton($routeName, string $icon ='', bool $disabled = FALSE, bool $openInNewWindow = FALSE): EntityDisplayTable
    {
        $entityTrueName = isset($this->entity) && $this->entity ? $this->entity->getEntityTrueName() : '';

        $button = (new TableButton())
            ->addRoute(Routes::getRoutePattern($routeName, [], FALSE))
            ->addIcon($icon)
            ->addClass($entityTrueName.'_table__btn')
            ->addClass($entityTrueName.'_table__btn-'.$icon)
            ->addClass('table__btn')
            ->addClass('table__btn-'.$icon);
        if ($openInNewWindow) $button->openInNewWindow();
        if($disabled) {
            $button->addRoute('#');
            $button->addClass($entityTrueName . '_table__btn-disabled');
            $button->addClass('table__btn-disabled');
            $button->disabled($disabled);
            if($this->hideDisabled){
                //do not add the button to the component
                return $this;
            }
        }
        $this->addComponent($button);
        return $this;
    }

    /**
     * @param bool $value
     * @return $this
     */
    public function hideDisabled(bool $value = TRUE): EntityDisplayTable
    {
        $this->hideDisabled = $value;
        return $this;
    }

    /**
     * @param $routeName
     * @return bool
     */
    protected function getAccessFromRoute($routeName): bool
    {
        $route = Routes::getRoute($routeName);
        if(($route ?? FALSE) && $permission = $route->getArgs('permission')){
            //we need to pass the opposite here as disabled is false if access it true.
            return !$this->currentUser->isAllowed($permission);
        }
        //no permission? just return true.
        return $route ? TRUE : FALSE; //no route? return false.
    }
}
