<?php

declare(strict_types=1);

namespace apexl\Io\modules\email\services;

use apexl\Io\includes\System;

class templateService
{
    /** @var string[]  */
    public static array $locations = [];

    private const SYSTEM_VAR_PREFIX = 'io_email_';

    /**
     * @param array<string, string> $replacements
     */
    public static function fetch(string $templateName, array $replacements): ?string
    {
        if (!$content = self::load($templateName)) {
            return null;
        }

        return self::replace($content, $replacements);
    }

    protected static function load(string $templateName): ?string
    {
        $templateFilename = sprintf('%s.html', $templateName);

        $locations = static::locations();

        foreach ($locations as $location) {
            $path = sprintf('%s/%s', $location, $templateFilename);
            if (file_exists($path)) {
                return file_get_contents($path);
            }
        }

        return null;
    }

    /**
     * @return string[]
     */
    protected static function locations(): array
    {
        return [
            sprintf('%s/../template/email', $_SERVER['DOCUMENT_ROOT']),
            ...self::$locations,
            sprintf('%s/../templates', dirname(__FILE__)),
        ];
    }

    /**
     * @param array<string, string> $replacements
     */
    protected static function replace(string $content, array $replacements): string
    {
        list($content, $replacements) = self::processLoops($content, $replacements);

        // Add system variables to replacements
        // Only use vars prefixed io_email_
        $systemVariables = array_filter(
            System::getVariables(),
            fn (string $key) => strpos($key, self::SYSTEM_VAR_PREFIX) !== false,
            ARRAY_FILTER_USE_KEY,
        );

        foreach ($systemVariables as $key => $value) {
            $key = str_replace(self::SYSTEM_VAR_PREFIX, '', $key);
            $replacements[$key] = $value;
        }

        return preg_replace(
            array_map(
                fn (string $key) => sprintf('/\{\{\s*%s\s*}}/', $key),
                array_keys($replacements)
            ),
            array_values($replacements),
            $content
        );
    }


    /**
     * This function duplicates looped elements in an HTML doc and make replacements according to looped object
     * e.g. data-for="item in lineItems"
     * lineItems is an array of php objects
     * {{ item.description }} in the html file will be replaced by object property "description"
     * @param $content
     * @param $replacements
     * @return false|string
     */
    protected static function processLoops($content, $replacements)
    {
        try {
            $doc = new \DOMDocument();
            $doc->loadXml($content);
            $xpath = new \DOMXPath($doc);
            $nodes = $xpath->query("//*[@data-for]"); // Get all nodes requiring duplication
            foreach ($nodes as $node) {
                $nodeHtml = $node->ownerDocument->saveHTML($node);
                list($subItemName, $in, $varName) = explode(" ", $node->getAttribute('data-for'));
                if (isset($replacements[$varName])) {
                    foreach ($replacements[$varName] as $item) {
                        $newNodeHtml = $nodeHtml;
                        foreach ($item as $prop => $val) {
                            $newNodeHtml = preg_replace("/\{\{\s*" . $subItemName . "." . $prop . "\s*}}/", $val, $newNodeHtml);
                        }
                        self::appendHTML($node->parentNode, $newNodeHtml);
                    }
                }
                $node->parentNode->removeChild($node);
            }

            // Remove array replacements
            foreach ($replacements as $id => $replacement) {
                if (is_array($replacement)) {
                    unset($replacements[$id]);
                }
            }

            $content = $doc->saveXML();

        } catch (\Exception $e) {
            // Exceptions most likely due to invalid content. NOTE domdocument requires tags closed e.g. <img /> works but <img> doesn't!
        }

        return [$content, $replacements];
    }

    protected static function appendHTML($parent, $source) {
        $tmpDoc = new \DOMDocument();
        $tmpDoc->loadHTML($source);
        foreach ($tmpDoc->getElementsByTagName('body')->item(0)->childNodes as $node) {
            $node = $parent->ownerDocument->importNode($node, true);
            $parent->appendChild($node);
        }
    }
}
