<?php

declare(strict_types=1);

namespace apexl\Io\modules\email\services;

use apexl\Io\includes\System;

class templateService
{
    /** @var string[]  */
    public static array $locations = [];

    private const SYSTEM_VAR_PREFIX = 'io_email_';

    /**
     * @param array<string, string> $replacements
     */
    public static function fetch(string $templateName, array $replacements): ?string
    {
        if (!$content = self::load($templateName)) {
            return null;
        }

        return self::replace($content, $replacements);
    }

    protected static function load(string $templateName): ?string
    {
        $templateFilename = sprintf('%s.html', $templateName);

        $locations = static::locations();

        foreach ($locations as $location) {
            $path = sprintf('%s/%s', $location, $templateFilename);
            if (file_exists($path)) {
                return file_get_contents($path);
            }
        }

        return null;
    }

    /**
     * @return string[]
     */
    protected static function locations(): array
    {
        return [
            sprintf('%s/../template/email', $_SERVER['DOCUMENT_ROOT']),
            ...self::$locations,
            sprintf('%s/../templates', dirname(__FILE__)),
        ];
    }

    /**
     * @param array<string, string> $replacements
     */
    protected static function replace(string $content, array $replacements): string
    {
        // Add system variables to replacements
        // Only use vars prefixed io_email_
        $systemVariables = array_filter(
            System::getVariables(),
            fn (string $key) => str_starts_with($key, self::SYSTEM_VAR_PREFIX),
            ARRAY_FILTER_USE_KEY,
        );

        foreach ($systemVariables as $key => $value) {
            $key = str_replace(self::SYSTEM_VAR_PREFIX, '', $key);
            $replacements[$key] = $value;
        }

        return preg_replace(
            array_map(
                fn (string $key) => sprintf('/\{\{\s*%s\s*}}/', $key),
                array_keys($replacements)
            ),
            array_values($replacements),
            $content
        );
    }
}
