<?php

declare(strict_types=1);

namespace apexl\Io\modules\formbuilder\FormKitValidation;

use apexl\dto\Dto;
use ValueError;

final readonly class FormKitValidation extends Dto
{
    public function __construct(
        public string $name,
        public string $config = '',
    ) {}

    public static function required(): FormKitValidation
    {
        return self::create('required');
    }

    public static function create(string $validation, string $config = ''): FormKitValidation
    {
        return new FormKitValidation($validation, $config);
    }

    /**
     * @param string $dateAfter YYYY-MM-DD format
     */
    public static function dateAfter(string $dateAfter): FormKitValidation
    {
        return self::create('date_after', $dateAfter);
    }

    /**
     * @param string $dateBefore YYYY-MM-DD format
     */
    public static function dateBefore(string $dateBefore): FormKitValidation
    {
        return self::create('date_before', $dateBefore);
    }

    public static function email(): FormKitValidation
    {
        return self::create('email');
    }

    public static function minNumber(int $min): FormKitValidation
    {
        return self::create('min', (string) $min);
    }

    public static function maxNumber(int $max): FormKitValidation
    {
        return self::create('max', (string) $max);
    }

    public static function charLength(int $min = 0, ?int $max = null): FormKitValidation
    {
        if ($min === 0 && $max === null) {
            throw new ValueError('`$min` or `$max` must be supplied');
        }

        $config = trim(sprintf('%s,%s', $min, $max ?? ''), ',');

        return self::create('length', $config);
    }

    public static function confirm(string $field = ''): FormKitValidation
    {
        return self::create('confirm', $field);
    }

    public function toString(): string
    {
        return sprintf('%s%s', $this->name, $this->config ? sprintf(':%s', $this->config) : '');
    }
}