<?php

namespace apexl\Io\modules\xero\services;

use apexl\Config\Singleton;
use apexl\Io\includes\System;
use Calcinai\OAuth2\Client\Provider\Xero;

use XeroPHP\Application as XeroApp;
use XeroPHP\Models\Accounting\Account;
use XeroPHP\Models\Accounting\BrandingTheme;
use XeroPHP\Models\Accounting\Contact;
use XeroPHP\Models\Accounting\Invoice;

class xeroService {

    protected $xero;
    protected $config;

    public function __construct()
    {
        $this->config = Singleton::getInstance();
        $this->instantiate();
    }

    public function getInvoices($email)
    {
        $contacts = $this->getContact((object)['email' => $email], 'email');

        if (!count($contacts)) return [];

        foreach ($contacts as $contact) {
            break;
        }

        $invoices = $this->xero->load(Invoice::class)
            ->where('Contact.ContactID', $contact->getContactID())
            ->execute();

        $data = [];
        foreach ($invoices as $invoice) {
            $data[] = [
                'id' => $invoice->getInvoiceID(),
                'name' => $invoice->getInvoiceNumber(),
                'email' => $email,
                'date' => $invoice->getDate()->format('jS M \'y'),
                'reference' => $invoice->getReference(),
                'subtotal' => number_format($invoice->getSubTotal(), 2),
                'totalTax' => number_format($invoice->getTotalTax(), 2),
                'total' => number_format($invoice->getTotal(), 2),
                //$this->getInvoiceUrl($invoice) ?? 'Unavailable',
            ];
        }

        return $data;
    }

    public function getBrandingThemes()
    {
        $themes = (object)[
            'timestamp' => time(),
            'themes' => []
        ];
        $brandingThemes =  $this->xero->load(BrandingTheme::class)->execute();
        foreach ($brandingThemes as $theme) {
            $themes->themes[] = (object)[
                'name' => $theme->getName(),
                'id' => $theme->getBrandingThemeID()
            ];
        }

        return $themes;
    }

    public function getAccounts()
    {
        $accounts =  $this->xero->load(Account::class)->execute();
        foreach ($accounts as $account) {
            echo"\n".$account->getName() . ' ' . $account->getCode();
        }

        die();
    }

    public function createInvoice($lineitems, $contactData, $dueDate = null, $desc = '', $branding = '5d4dd402-c851-497e-aae1-9ff265c0d15a', $sendEmail=FALSE, $type = 'ACCREC', $currency = 'GBP') {

        $this->instantiate();

        $dueDateTime = new \DateTime("now", new \DateTimeZone('Europe/London'));
        if($dueDate) $dueDateTime->setTimestamp(strtotime($dueDate));

        $addContact = TRUE;
        $xeroContact = $this->getContact($contactData);

        // check existing / create new contact
        foreach($xeroContact as $contact){
            if(!empty($contact)) {
                $addContact = FALSE; // Contact found - don't add a new one
                break;
            }
        }
        if($addContact) $contact = $this->createContact($contactData); //  create a new contact

        $invoice = new \XeroPHP\Models\Accounting\Invoice($this->xero);
        $invoice->setStatus("AUTHORISED");
        $invoice->setLineAmountType('Inclusive');
        $invoice->setType($type);
        $invoice->setCurrencyCode($currency);
        $invoice->setDueDate($dueDateTime);
        $invoice->setContact($contact);
        $invoice->setBrandingThemeID($branding);

        //create the Line items
        foreach($lineitems as $lineItem){
            $invoice->addLineItem($this->createInvoiceLineItem($lineItem, $desc));
        }

        $response = $invoice->save();

        if ($sendEmail) $invoice->sendEmail();

        return [$invoice->getInvoiceId(), $invoice->getInvoiceNumber(), $this->getDataFromContact($contact)];
    }

    public function getInvoiceUrl($invoiceId)
    {
        try {

            $invoices = $this->xero->load(Invoice::class)
                ->where('InvoiceID', $invoiceId)
                ->execute();

            foreach ($invoices as $invoice) {
                $url = new \XeroPHP\Remote\URL($this->xero, 'Invoices/' . $invoice->getGUID() . '/OnlineInvoice');
                $request = new \XeroPHP\Remote\Request($this->xero, $url);
                $request->send();
                return $request->getResponse()->getElements()[0]['OnlineInvoiceUrl'] ?? null;
            }

        } catch (NotFoundException $e) {
            // handle not found error
        } catch (RateLimitExceededException $e) {
            // handle rate limit error
        } catch (\Exception $e) {

        }

        return null;
    }

    protected function getDataFromContact($xeroContact)
    {
        $data = (object)[
            'firstName' => $xeroContact->getFirstName(),
            'lastName' => $xeroContact->getLastName(),
            'emailAddress' => $xeroContact->getEmailAddress(),
            'taxNumber' => $xeroContact->getTaxNumber()
        ];

        return $data;
    }

    protected function createContact($details){
        $this->instantiate();
        $contact = new \XeroPHP\Models\Accounting\Contact($this->xero);
        $contact->setName($details->name)->setFirstName($details->firstName)->setLastName($details->lastName)->setEmailAddress($details->emailAddress)->setTaxNumber($details->taxNumber);
        $contact->save();

        return $contact;
    }

    protected function getProvider()
    {
        if (!isset($provider)) {
            static $provider;

            $provider = new Xero([
                'clientId'          => $this->config->app->invoicing->xero->client_id,
                'clientSecret'      => $this->config->app->invoicing->xero->client_secret,
                'redirectUri'       => $this->config->app->invoicing->xero->redirect_url,
            ]);
        }

        return $provider;
    }

    /**
     * Get access token
     * @return void
     */
    protected function getAccesstoken($oauthMethod='client_credentials')
    {
        $tokenData = System::getVariable('xero_shortlived_client_credentials_access_token');
        if ($tokenData) {
            if (isset($tokenData->expires) && $tokenData->expires && $tokenData->expires > time()) {
                return $tokenData;
            }
        }

        $provider = $this->getProvider();
        $token = $provider->getAccessToken($oauthMethod);
        $tokenData = $this->getTokenData($token);

        if ($tokenData && isset($tokenData->token)) {
            System::setVariable('xero_shortlived_client_credentials_access_token', json_encode($tokenData));
        }

        return $tokenData;
    }

    protected function refreshToken($tokenData)
    {
        $provider = $this->getProvider();

        $newAccessToken = $provider->getAccessToken('refresh_token', [
            'refresh_token' => $tokenData['refreshToken']
        ]);

        $tokenData = $this->getTokenData($newAccessToken);
    }

    protected function getTokenData($token)
    {
        $provider = $this->getProvider();
        $tenants = $provider->getTenants($token);
        return (object)[
            'token' => $token->getToken(),
            'expires' => $token->getExpires(),
            'refreshToken' => $token->getRefreshToken(),
            'tenantId' => $tenants[0]->tenantId,
        ];
    }

    protected function instantiate()
    {
        $tokenData = $this->getAccesstoken();
        $this->xero = new XeroApp($tokenData->token, $tokenData->tenantId);
    }

    protected function createInvoiceLineItem($lineInfo){
        $lineItem = new \XeroPHP\Models\Accounting\Invoice\LineItem($this->xero);
        $lineItem->setDescription($lineInfo->description)
            ->setQuantity($lineInfo->quantity)
            ->setLineAmount($lineInfo->amount)
            ->setAccountCode($lineInfo->accountCode);
        if(isset($lineInfo->taxType)) {
            $lineItem->setTaxType($lineInfo->taxType);
        }
        $lineItem->save();

        return $lineItem;
    }

    protected function getContact($contactDetails, $useField='name'){

        switch ($useField) {
            case 'email':
                return $this->xero->load(Contact::class)->where('EmailAddress', $contactDetails->email)->execute();
                break;
            default:
                return $this->xero->load(Contact::class)->where('Name', $contactDetails->name)->execute();
                break;
        }
    }


}