<?php

namespace apexl\Io\modules\invoice\services;

use apexl\Config\Singleton;
use apexl\Io\includes\Hook;
use apexl\Io\includes\Routes;
use apexl\Io\includes\System;
use apexl\Io\modules\email\services\templateService;
use apexl\Io\modules\invoice\entities\invoiceEntity;
use apexl\Io\modules\invoice\entities\invoicePaymentEntity;
use apexl\Io\modules\invoice\entities\invoiceSubscriptionEntity;
use apexl\Io\modules\invoice\providers\xeroProvider;
use apexl\Io\modules\payment\entities\paymentEntity;
use apexl\Io\modules\payment\entities\userCreditEntity;
use apexl\Io\modules\payment\providers\stripeProvider;
use apexl\Io\modules\payment\services\cardValidationService;
use apexl\Io\modules\payment\services\paymentService;
use apexl\Io\modules\product\entities\productEntity;
use apexl\Io\modules\subscription\entities\subscriptionCreditEntity;
use apexl\Io\modules\subscription\entities\subscriptionEntity;
use apexl\Io\modules\subscription\services\subscriptionService;
use apexl\Io\modules\user\entities\userEntity;
use apexl\Io\services\Logger;
use apexl\Io\services\Mailer;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;

class invoiceService {

    protected $config;
    protected $providerContactData = [];

    public function __construct()
    {
        $this->config = Singleton::getInstance();
        $this->setProvider();

        if (!defined('REMOTE_INVOICE_REF')) define('REMOTE_INVOICE_REF', 'Dashboard Inv Ref #{ID}');
    }

    /**
     * Set provider class (xero etc.)
     * @return void
     */
    public function setProvider()
    {
        $this->provider = null;
        $provider = $this->config->app->invoicing->provider ?? null;

        switch ($provider){
            case 'xero':
                $this->provider = new xeroProvider($this);
                break;
        }
    }

    /**
     * Create an invoice.  If provider config is set, create an invoice in the provider service
     * @param $userId
     * @param $lineItems
     * @param $subscriptions
     * @return invoiceEntity
     * @throws \Exception
     */
    public function createInvoice($userId, $lineItems, $sendReceipt=false, $subscriptions=[], $remoteOrderId=null, $rollForwardSubscriptions=true, $billingEnabled=true, $brandingThemeId=null, $lookupContactBy='email', $sendEmail=FALSE,$lineAmountType='Exclusive', $contactData=null)
    {
        if (count($lineItems) > 0) {

            list($lineItems, $creditApplied) = $this->applyCreditToLineItems($userId, $lineItems);
            list($total, $vat) = $this->getTotalFromLineItems($lineItems);

        $invoice = new invoiceEntity();
            $invoice->user_id = $userId;
        $invoice->invoice_date = date('Y-m-d');
        $invoice->total = $total;
        $invoice->vat = $vat;
        $invoice->line_items = json_encode($lineItems);
            $invoice->provider = $this->config->app->invoicing->provider ?? null;
            if ($remoteOrderId) $invoice->remote_order_id = $remoteOrderId;
            $invoice->credit_applied = $creditApplied > 0 ? $creditApplied : 0;
            if (!$billingEnabled) $invoice->billing_enabled = 0;
            $invoice->send_receipt = $sendReceipt;

            $invoice->store();
            $invReference = str_replace('{ID}', $invoice->id, REMOTE_INVOICE_REF);

            // Roll forward subscriptions
            if (count($subscriptions) > 0 && $rollForwardSubscriptions) {
                $subscriptionService = new subscriptionService();
                $subscriptionService->rollForwardSubscriptions($subscriptions);
            }

            if ($creditApplied > 0) {
                $this->subtractCredit($userId, $creditApplied, "Credit applied to invoice #" . $invoice->id);
            }

            // Link subscriptions to invoice
            foreach ($subscriptions as $subscription) {
                $invoiceSubscriptionEntity = new invoiceSubscriptionEntity();
                $invoiceSubscriptionEntity->invoice_id = $invoice->id;
                $invoiceSubscriptionEntity->subscription_id = $subscription->id;
                $invoiceSubscriptionEntity->store();
            }

            if ($this->provider) {
                if (!$contactData) {
                $userEntity = new userEntity();
                $userEntity->load($userId);
                $contactData = (object)[
                    'name' => trim($userEntity->first_name . ' ' . $userEntity->last_name),
                    'firstName' => $userEntity->first_name,
                    'lastName' => $userEntity->last_name,
                    'emailAddress' => $userEntity->email,
                    'taxNumber' => ""
                ];
                }

                $brandingTheme = $brandingThemeId ?? $this->config->app->invoicing->xero->default_branding_theme;
                list($invoiceId, $invoiceNumber, $contactData) = $this->createProviderInvoice($lineItems, $contactData, $lookupContactBy, null, '', $brandingTheme, $sendEmail, $invReference, 'ACCREC', 'GBP', $lineAmountType);

                $this->providerContactData[$invoiceNumber] = $contactData;

                $invoice->provider_id = $invoiceId;
                $invoice->invoice_number = $invoiceNumber;
                $invoice->store();
            }
        }


        return $invoice;
    }

    public function getProviderContactData($invoiceNumber)
    {
        return $this->providerContactData[$invoiceNumber] ?? null;
    }

    public function sendInvoiceToProvider($invoiceId)
    {

        $result = (object)[
            'invoices_sent' => 0,
            'invoice_payments_sent' => 0,
        ];

        if ($this->provider) {
            $invoiceEntity = new invoiceEntity();
            $invoiceEntity->load($invoiceId);

            if ((isset($invoiceEntity->provider_id) || is_null($invoiceEntity->provider_id)) && trim($invoiceEntity->provider_id) == "") {

                $lineItems = json_decode($invoiceEntity->line_items);

                $userEntity = new userEntity();
                $userEntity->load($invoiceEntity->user_id);
                $contactData = (object)[
                    'name' => trim($userEntity->first_name . ' ' . $userEntity->last_name),
                    'firstName' => $userEntity->first_name,
                    'lastName' => $userEntity->last_name,
                    'emailAddress' => $userEntity->email,
                    'taxNumber' => ""
                ];

                list($invoiceId, $invoiceNumber, $contactData) = $this->createProviderInvoice($lineItems, $contactData, 'email', null, '', $this->config->app->invoicing->xero->default_branding_theme);

                $result->invoices_sent++;

                $invoiceEntity->provider_id = $invoiceId;
                $invoiceEntity->invoice_number = $invoiceNumber;
                $invoiceEntity->store();

                $invoicePaymentEntity = new invoicePaymentEntity();
                $payments = $invoicePaymentEntity->getPayments($invoiceEntity->id);


                foreach ($payments as $invoicePayment) {
                    $invoicePaymentEntity = new invoicePaymentEntity();
                    $invoicePaymentEntity->loadByPaymentId($invoicePayment->id);
                    if (trim($invoicePaymentEntity->provider_payment_id == "")) {
                        $paymentEntity = new paymentEntity();
                        $paymentEntity->load($invoicePaymentEntity->payment_id);
                        if ($paymentEntity->status == 'successful') {
                            $this->submitPaymentToProvider($paymentEntity, $invoiceEntity, $invoicePaymentEntity);
                            $result->invoice_payments_sent++;
                        }
                    }
                }
            }
        }

        return $result;
    }

    /**
     * Create an invoice line item object for a given product ID
     * @param $name
     * @param $quantity
     * @param $productId
     * @param $total
     * @return void
     */
    public function createProductLineItem($name, $quantity, $productId, $total=null)
    {
        $productEntity = new productEntity();
        $productEntity->load($productId);
        if (isset($productEntity->id) && $productEntity->id) {
            $price = is_numeric($total) ? $total : $productEntity->price;
            $this->createInvoiceLineItem($name, $quantity, $productEntity->price, $productEntity->invoice_nominal_code, $productId);
        }
    }

    /**
     * Create an invoice line item
     * @param $name
     * @param $quantity
     * @param $perUnitCost
     * @param $nominalCode
     * @param $productId
     * @return array|object
     */
    public function createInvoiceLineItem($description, $quantity, $perUnitCost, $nominalCode, $productId=null, $taxType=null)
    {
        if ($quantity == 0) return [];
        $total = number_format($quantity*$perUnitCost, 2);
        //$nominalCode = '200';
        $lineItem = (object)[
            'description' => $description,
            'accountCode' => $nominalCode,
            'quantity' => $quantity,
            'perUnit' => $perUnitCost,
            'amount' => $total,
            'vat' => $taxType != 'NONE' ? number_format(0.2*$total, 2) : 0,
            'productId' => $productId
        ];
        if ($taxType) {
            $lineItem->taxType = $taxType;
        }

        return $lineItem;
    }

    /**
     * Apply a user's existing credit to a group of line items
     * @param $userId
     * @param $lineItems
     * @return array
     */
    public function applyCreditToLineItems($userId, $lineItems)
    {
        $credit = $this->getTotalCredit($userId);
        $initialCredit = $credit;

        foreach ($lineItems as $id => $lineItem) {
            if ($credit > 0) {
                $totalBeforeCredit = $lineItem->amount;
                list($lineItem->amount, $credit) = $this->applyCreditToAmount($lineItem->amount, $credit);
                $lineItem->vat = round($lineItem->vat*($lineItem->amount/$totalBeforeCredit), 2); // reduce VAT accordingly
                $lineItem->creditApplied = round($totalBeforeCredit - $lineItem->amount, 2); // Note credit applied on lineItem
                $creditApplied = $lineItem->creditApplied;
                $credit -= $creditApplied;
                $lineItems[$id] = $lineItem;
            }
        }

        return [$lineItems, ($initialCredit - $credit)];
    }

    /**
     * Get total and total VAT from a group of line items
     * @param $lineItems
     * @return array
     */
    public function getTotalFromLineItems($lineItems)
    {
        $total = 0.00;
        $vat = 0.00;

        foreach ($lineItems as $id => $lineItem) {
            $total += $lineItem->amount;
            $vat += $lineItem->vat;
        }

        return [round($total,2), round($vat,2)];
    }

    /**
     * Function to handle automated creation of subscription invoices
     * @return Response
     * @throws \Exception
     */
    public function triggerSubscriptionInvoices(Request $request, Response $response, $args)
    {
        if (!self::checkStartCron()) return System::asJson($response, ['error' => 'exit due to cron duplication. process: '.getmypid()]);

        $subscriptionEntity = new subscriptionEntity();
        $productEntity = new productEntity();
        $subscriptionService = new subscriptionService();
        $toBill = $subscriptionEntity->getSubscriptionsToBill(date('Y-m-d'), 1, 24);
        $subscriptionsProcessed = 0;

        foreach ($toBill as $userId => $subscriptions) {

            $lineItems = [];

            $subscriptionsBilled = [];
            $gracePeriodSubscriptions = [];

            foreach ($subscriptions as $subscription) {
                $incCharge = !$subscription->grace_period_ends || $subscription->next_billing_date >= $subscription->grace_period_ends; // If this is the grace period, subtract the base subscription charge
                if (!$incCharge) $gracePeriodSubscriptions[$subscription->id] = $subscription;

                $productEntity->load($subscription->product_id);
                if ($incCharge && isset($productEntity->id)) {// Get invoice line items
                    $billingDate = \DateTime::createFromFormat('Y-m-d', $subscription->next_billing_date);

                    $price = !empty($subscription->product_price_override) && $subscription->product_price_override > 0 && $subscription->product_price_override_ends > $subscription->next_billing_date ? $subscription->product_price_override : $productEntity->price;

                    $lineItems[] = $this->createInvoiceLineItem("Subscription Fee: {$productEntity->product_name}, scheduled billing date: {$billingDate->format('d M Y')}", $subscription->product_quantity, $productEntity->price, $productEntity->invoice_nominal_code, $subscription->product_id);
                    $subscriptionsBilled[$subscription->id] = $subscription;
                    $newLineItems = Hook::processHook('trigger_subscription_invoice_line_items', $subscription, $lineItems);
                    if (is_array($newLineItems)) $lineItems = $newLineItems;
                }

                $subscriptionsProcessed++;
            }

            if (count($lineItems) > 0) { // Create the invoice
                if (!self::checkContinueCron()) return System::asJson($response, ['error' => 'exit due to cron duplication. process: '.getmypid()]);
                $invoice = $this->createInvoice($userId, $lineItems, $this->config->app->invoicing->send_receipts ?? false, $subscriptionsBilled);
                if (isset($invoice->id) && $invoice->id) $lineItems = Hook::processHook('confirm_subscription_invoice_line_items', $invoice, $lineItems, $userId);
            }

            // Roll forward subscription to next billing period
            $subscriptionService->rollForwardSubscriptions($gracePeriodSubscriptions);

            foreach ($subscriptions as $subscription) {
                $subscriptionEntity->load($subscription->id);
                $subscriptionEntity->last_checked = date('Y-m-d H:i:s');
                $subscriptionEntity->store();
            }
        }

        return System::asJson($response, ['subscriptionsProcessed' => $subscriptionsProcessed]);
    }

    /**
     * Function to handle automated billing of invoices via payments provider
     * @return Response
     * @throws \Exception
     */
    public function triggerInvoicePayments(Request $request, Response $response, $args)
    {
        if (!self::checkStartCron('invoice_payment')) return System::asJson($response, ['error' => 'exit due to cron duplication. process: '.getmypid()]);

        $invoiceEntity = new invoiceEntity();
        $invoices = $invoiceEntity->getChargeable(1);
        $cardValidationService = new cardValidationService();

        $cutoff = new \DateTime('now');
        $timeInterval = 24*60; // Don't attempt to charge a user more than once per day
        $cutoff->sub(new \DateInterval("PT".$timeInterval."M"));
        $result = (object)['invoice_charge_attempts' => 0, 'invoice_charge_passes' => 0];

        foreach ($invoices as $invoice) {

            $invoiceEntity = new invoiceEntity();
            $invoiceEntity->load($invoice->id);
            $ref = 'InvId-' . $invoiceEntity->id;

            if (!$this->invoiceSafetyNetChecks($invoiceEntity, $ref)) { // Perform safety net checks
            $paymentEntity = new paymentEntity();
                $paymentEntity->getMostRecentForUser($invoice->user_id, [] /*['failed', 'three_d_secure_required']*/);
            $lastPaymentForUser = \DateTime::createFromFormat('U', $paymentEntity->created); // Get the last payment attempt for this user

            if ($lastPaymentForUser < $cutoff) {
                    if (!self::checkContinueCron('invoice_payment')) return System::asJson($response, ['error' => 'exit due to cron duplication. process: '.getmypid()]);
                    $this->attemptInvoicePayment($invoiceEntity, $ref, true, $this->getPaymentNotesFromInvoice($invoiceEntity));
                $result->invoice_charge_attempts++;
                } else {
                    $result->invoice_charge_passes++;
                }
            }

                $invoiceEntity->last_payment_attempt = date('Y-m-d H:i:s'); // Set last payment attempt date and time
                $invoiceEntity->store();
        }

        return $result;
    }

    protected static function checkStartCron($process='subscription')
    {
        $lastStarted = System::getVariable($process.'_cron_process_started');
        if (!$lastStarted || microtime(true) - (float)$lastStarted > 60) {
            // continue if no other process started in the last minute
            System::setVariable($process.'_cron_process_started', microtime(true));
            System::setVariable($process.'_cron_process_pid', getmypid());
            sleep(3);
            return self::checkContinueCron($process);
        }

        return false;
    }

    protected static function checkContinueCron($process='subscription')
    {
        $activePid = System::getVariable($process.'_cron_process_pid');
        if ($activePid == getmypid()) {
            return true;
        }

        return false;
    }

    /**
     * This function provides a safety net against inadvertant billing of the same invoice multiple times
     * @param $invoiceEntity
     * @param $ref
     * @return bool
     */
    protected function invoiceSafetyNetChecks($invoiceEntity, $ref)
    {

        // 1. Check invoice payments
        $totalPaid = 0.0;
        $paymentEntity = new paymentEntity();
        $payments = $paymentEntity->fetchAllByRef($ref);
        foreach ($payments as $payment) {
            if ($payment->status == 'successful') {
                $totalPaid += (float)$payment->amount;
            }
        }
        $invTotal = (float)($invoiceEntity->total + $invoiceEntity->vat);
        if ($invTotal - $totalPaid <= 0.015) {
            $invoiceEntity->paid = 1;
            $invoiceEntity->store();
            return true;
        }

        // 2. Check balance in Xero
        if (isset($invoiceEntity->provider_id) && $invoiceEntity->provider_id) { // Submit payment information to provider
            $invoice = $this->provider->getInvoice($invoiceEntity->provider_id);
            if ($invoice->AmountDue < 0.015) {
                $invoiceEntity->paid = 1;
                $invoiceEntity->store();
                return true;
            }
        }

        return false;

    }

    protected function getPaymentNotesFromInvoice($invoiceEntity)
    {
        $user = new userEntity();
        $user->load($invoiceEntity->user_id);
        $lineItems = json_decode($invoiceEntity->line_items);
        foreach ($lineItems as $id => $item) {
            $lineItems[$id]->email = $user->email;
        }

        if(strlen(json_encode($lineItems)) > 999){
            $lineItems = [];
            $lineItems[] = 'Data Too long for stripe - Can be retrieved from dashboard if needed.';
        }

        return $lineItems;
}
    /**
     * Attempt to chrage payment for a single invoice
     * @param $invoiceEntity
     * @param $ref
     * @param $isSubscriptionPayment
     * @param $data
     * @param $meta
     * @param $userPresent
     * @return paymentEntity|null
     */
    public function attemptInvoicePayment($invoiceEntity, $ref="", $isSubscriptionPayment=true, $data=[], $meta=null, $userPresent=false)
    {
        $paymentEntity = null;

        if ($invoiceEntity->billing_enabled == '1') {

            $paymentService = new paymentService();
            $cardValidationService = new cardValidationService();
            $subscriptionEntity = new subscriptionEntity();
            $subscriptions = $subscriptionEntity->getByInvoiceId($invoiceEntity->id);
            $cardValidationService->setSubscriptions($subscriptions);
            $billingToken = $paymentService->getBillingToken($invoiceEntity->user_id);

            if (!$billingToken) {
                $invoiceEntity->on_hold = 1;
                $invoiceEntity->store();
                $cardValidationService->processSubscriptionPaymentFailure();
            } else {
                $paymentEntity = $cardValidationService->capturePaymentWithToken($invoiceEntity->user_id, ($invoiceEntity->total + $invoiceEntity->vat), $ref, $isSubscriptionPayment, $data, $meta, $userPresent);
                if (isset($paymentEntity->id) && trim($invoiceEntity->provider_id) != "") {
                    $invoicePaymentEntity = $this->addPaymentToInvoice($invoiceEntity->id, $paymentEntity);
                    if (trim($invoiceEntity->provider_id) != "") {
                        if ($paymentEntity->status == 'successful') {
                            $this->submitPaymentToProvider($paymentEntity, $invoiceEntity, $invoicePaymentEntity);
                        }
                    }
                }
            }

            $this->checkAndMarkAsPaid($invoiceEntity);
        }

        return $paymentEntity;
    }

    public function reenableUserInvoices($userId)
    {
        $invoiceEntity = new invoiceEntity();
        $invoices = $invoiceEntity->getForUser($userId);
        foreach ($invoices as $invoice) {
            if ($invoice->on_hold == 1) {
                $invoiceEntity->load($invoice->id);
                $invoiceEntity->on_hold = 0;
                $invoiceEntity->store();
            }
        }
    }

    public function reenableSubscriptionInvoices($subscriptionId)
    {
        $invoiceEntity = new invoiceEntity();
        $invoices = $invoiceEntity->getForSubscription($subscriptionId);
        foreach ($invoices as $invoice) {
            if ($invoice->on_hold == 1) {
                $invoiceEntity->load($invoice->id);
                $invoiceEntity->on_hold = 0;
                $invoiceEntity->store();
            }
        }
    }

    /**
     * Submit payment to invoice provider and store the ID
     * @param $paymentEntity
     * @param $invoiceEntity
     * @param $invoicePaymentEntity
     * @return invoicePaymentEntity|void
     * @throws \Exception
     */
    public function submitPaymentToProvider($paymentEntity, $invoiceEntity=null, $invoicePaymentEntity=null)
    {
        if (!$invoicePaymentEntity) {
            $invoicePaymentEntity = new invoicePaymentEntity();
            $invoicePaymentEntity->loadByPaymentId($paymentEntity->id);
            if (!isset($invoicePaymentEntity->id)) return null;
        }

        if (!$invoiceEntity) {
            $invoiceEntity = new invoiceEntity();
            $invoiceEntity->load($invoicePaymentEntity->invoice_id);
        }
        if (isset($invoiceEntity->provider_id) && $invoiceEntity->provider_id) { // Submit payment information to provider
            $providerPaymentId = $this->provider->createPayment($invoiceEntity->provider_id, $paymentEntity->amount, $paymentEntity->provider_ref_2, $this->config->app->invoicing->sale_account_code);
            $invoicePaymentEntity->provider_payment_id = $providerPaymentId;
            $invoicePaymentEntity->store();
        }
    }

    /**
     * log a payment against an invoice
     * @param $invoiceId
     * @param $paymentEntity
     * @return void
     * @throws \Exception
     */
    public function addPaymentToInvoice($invoiceId, $paymentEntity)
    {
        $invoicePaymentEntity = new invoicePaymentEntity();
        $invoicePaymentEntity->invoice_id = $invoiceId;
        $invoicePaymentEntity->payment_id = $paymentEntity->id;
        $invoicePaymentEntity->store();

        return $invoicePaymentEntity;
    }

    /**
     * Check whether an invoice has been paid and mark it as such
     * @param $invoiceEntity
     * @return void
     */
    public function checkAndMarkAsPaid($invoiceEntity)
    {
        if (isset($invoiceEntity->id) && $invoiceEntity->id) {
            $invoicePaymentEntity = new invoicePaymentEntity();
            $payments = $invoicePaymentEntity->getPayments($invoiceEntity->id);

            //echo 'NUM Payments:' . count($payments);

            $totalPaid = 0.0;
            $chargeIds = [];
            foreach ($payments as $payment) {
                if ($payment->status == 'successful') {
                    $totalPaid += $payment->amount;
                    $chargeIds[] = $payment->provider_ref_2;
                }
            }

            $totalPaid = (float)$totalPaid;
            $invTotal = (float)($invoiceEntity->total + $invoiceEntity->vat);

            //echo "\nTot Due: " . $invTotal;
            //echo "\nTot Paid: " . $totalPaid;
            //echo "\nDifference: " . ($invTotal - $totalPaid);

            if ($invTotal - $totalPaid <= 0.001) {
                $invoiceEntity->paid = 1;
                $invoiceEntity->store();
                if ($invoiceEntity->send_receipt) {
                    $this->sendReceipt($invoiceEntity);
                }
                Hook::processHook('on_successful_invoice_payment', $invoiceEntity, $chargeIds);
            }
        }
    }

    protected function sendReceipt($invoiceEntity)
    {
        if (!$invoiceEntity->receipt_sent_date) {

            $userEntity = new userEntity();
            $userEntity->load($invoiceEntity->user_id);
            if (isset($userEntity->id) && $userEntity->id > 0) {

                $from = $this->config->app->site->email_address ?? 'no-reply@localhost.com';
                $fromName = $this->config->app->site->name ?? 'localhost';
                $frontEndDomain = $this->config->app->site->frontend_domain ?? 'localhost';
                $link = rtrim($frontEndDomain, '/');// . '/sim-activation/verify-email/' . $session;

                $emailData = [
                    'first_name' => $userEntity->first_name,
                    'last_name' => $userEntity->last_name,
                    'email' => $userEntity->email,
                    'lineItems' => json_decode($invoiceEntity->line_items),
                    'subtotal' => number_format($invoiceEntity->total, 2),
                    'vat' => number_format($invoiceEntity->vat, 2),
                    'total' => number_format($invoiceEntity->total + $invoiceEntity->vat, 2),
                ];

                $mailer = new Mailer();
                $mailer->setFrom($from, $fromName);
                $mailer->addAddress($userEntity->email);     // Add a recipient

                $mailer->Subject = 'GeoSIM - Your Receipt';
                $mailer->Body = templateService::fetch("receipt_email", $emailData);

                // Attach invoice
                if ($this->provider) {
                    if ($content = $this->provider->getInvoicePDF($invoiceEntity->provider_id)){
                        $mailer->addStringAttachment($content, $invoiceEntity->invoice_number . '.pdf');
                    }
                }

                $mailer->IsHTML(true);
                if ($result = $mailer->send()) {
                    $invoiceEntity->receipt_sent_date = date('Y-m-d H:i:s');
                    $invoiceEntity->store();
                }
                Logger::log('send_invoice_receipt', ['sent' => $result, 'invoice' => $invoiceEntity->getData()]);
            }

        }
    }

    /**
     * Update a users total credit
     * @param $userId
     * @param $newCreditAmount
     * @param $data
     * @return void
     * @throws \Exception
     */
    public function updateTotalCredit($userId, $newCreditAmount, $data)
    {
        // Delete existing credits for this user
        $userCreditEntity = new userCreditEntity();
        $credits = $userCreditEntity->getByUser($userId);
        foreach ($credits as $credit) {
            $userCreditEntity->load($credit->id);
            $userCreditEntity->delete();
        }

        $this->addCredit($userId, $newCreditAmount, $data);
    }

    /**
     * Add credit for a user
     * @param $userId
     * @param $newCreditAmount
     * @param $data
     * @return void
     * @throws \Exception
     */
    public function addCredit($userId, $newCreditAmount, $data)
    {
        // Add credit
        $userCreditEntity = new userCreditEntity();
        $userCreditEntity->user_id = $userId;
        $userCreditEntity->credit_amount = $newCreditAmount;
        $userCreditEntity->data = json_encode($data);
        $userCreditEntity->store();
    }

    /**
     * Subtract credit from a user's credit balance
     * @param $userId
     * @param $subtractAmount
     * @param $reason
     * @return void
     * @throws \Exception
     */
    public function subtractCredit($userId, $subtractAmount, $reason)
    {
        $totalCredit = $this->getTotalCredit($userId);
        $this->updateTotalCredit($userId, ((float)$totalCredit - (float)$subtractAmount), ['reason' => $reason]);
    }

    /**
     * OLD - FOR REF ONLY:   NOTE THIS FUNCTION ASSUMES STORED CREDIT VALUES EXCLUDE VAT
     * AND THAT $AMOUNT INCLUDES VAT
     * @param $amount
     * @param $userId
     * @return array
     */
    /* public function applyCreditToAmount($amount, $userId)
     {
         $totalCredit = $this->getTotalCredit($userId);
         $totalCreditWithVAT = $totalCredit*1.2;
         if ($totalCreditWithVAT >= $amount) {
             return [0, $totalCreditWithVAT-round($amount/1.2,2)];
         } else {
             return [($amount - $totalCreditWithVAT), $totalCredit];
         }
     }*/

    /**
     * Apply credit to a total and return the new total and remaining credit
     * @param $amount
     * @param $totalCredit
     * @return array
     */
    public function applyCreditToAmount($amount, $totalCredit)
    {
        if ($totalCredit >= $amount) {
            return [0, $amount];
        } else {
            return [($amount - $totalCredit), $totalCredit];
        }
    }

    /**
     * Get a user's credit balance
     * @param $userId
     * @return float
     */
    public function getTotalCredit($userId)
    {
        $userCreditEntity = new userCreditEntity();
        $credits = $userCreditEntity->getByUser($userId);
        $totalCredit = 0.0;
        foreach ($credits as $credit) {
            $totalCredit += $credit->credit_amount;
        }

        return $totalCredit;
    }

    /**
     * Find an invoice entity using a provider ID (e.g. Xero ID)
     * @param $providerId
     * @return invoiceEntity|null
     */
    public function getInvoiceByProviderId($providerId)
    {
        $invoiceEntity = new invoiceEntity();
        $invoiceEntity->loadByProviderId($providerId);
        if (isset($invoiceEntity) && $invoiceEntity->provider_id == $providerId) {
            return $invoiceEntity;
        }

        return null;
    }

    public function getInvoiceByReference($reference)
    {
        if (str_replace("{ID}", "", REMOTE_INVOICE_REF) == preg_replace('/[0-9]+/', '', $reference)) {
            $invoiceId = preg_replace("/[^0-9]/", "", $reference );
            if (is_numeric($invoiceId)) {
                $invoiceEntity = new invoiceEntity();
                $invoiceEntity->load($invoiceId);
                return $invoiceEntity;
            }
        }

        return null;
    }

    public function getInvoiceByDashboardInvoiceRef($providerId)
    {
        $invoiceEntity = new invoiceEntity();
        $invoiceEntity->loadByProviderId($providerId);
        if (isset($invoiceEntity) && $invoiceEntity->provider_id == $providerId) {
            return $invoiceEntity;
        }

        return null;
    }

    public function getProviderContactById($id)
    {
        if ($this->provider) {
            return $this->provider->getContact((object)['id' => $id], 'id');
        }
    }

    /**
     * Get provider invoices for a given email address
     * @param $email
     * @return void
     */
    public function getProviderInvoices($email)
    {
        if ($this->provider) {
            return $this->provider->getInvoices($email);
        }
    }

    /**
     * Create an invoice in the provider service (e.g. on Xero)
     * @param $lineitems
     * @param $contactData
     * @param $dueDate
     * @param $desc
     * @param $branding
     * @param $sendEmail
     * @param $type
     * @param $currency
     * @return void
     */
    public function createProviderInvoice($lineitems, $contactData, $lookupContactBy='name', $dueDate = null, $desc = '', $branding = '5540a201-bccd-4585-a66d-c5d2aa40cc56', $sendEmail=FALSE, $reference=null, $type = 'ACCREC', $currency = 'GBP', $lineAmountType='Exclusive')
    {
        if ($this->provider) {
            return $this->provider->createInvoice($lineitems, $contactData, $lookupContactBy, $lineAmountType, $dueDate, $desc, $branding, $sendEmail, $reference, $type, $currency);
        }
    }

    /**
     * Get public URL of provider invoice
     * @param $providerInvoiceId
     * @return void
     */
    public function getProviderInvoiceUrl($providerInvoiceId)
    {
        if ($this->provider) {
            return $this->provider->getInvoiceUrl($providerInvoiceId);
        }
    }

    /**
     * Get provider branding themes
     * @return void
     */
    public function getProviderBrandingThemes()
    {
        if ($this->provider) {
            return $this->provider->getBrandingThemes();
        }
    }

    /**
     * Get provider accounts
     * @return void
     */
    public function getProviderAccounts()
    {
        if ($this->provider) {
            $this->provider->getAccounts();
        }
    }
}


