<?php

declare(strict_types=1);

namespace apexl\Io\modules\jobs\Entity;

use apexl\entityCore\enums\Casts;
use apexl\entityCore\traits\hasCasts;
use apexl\Io\modules\jobs\Interface\QueueTaskHandlerInterface;
use app\module\newtonsTrailers\Dto\EtlSyncJobData;
use app\module\newtonsTrailers\Entity\Entity;
use DateTimeImmutable;
use Exception;

/**
 * @property int $id
 * @property int $sequence
 * @property class-string $handler
 * @property DateTimeImmutable $created
 * @property bool $locked
 * @property int $attempts
 * @property DateTimeImmutable $processed
 * @property DateTimeImmutable $failed
 * @property EtlSyncJobData $contents
 */
class QueueJob extends Entity
{
    use hasCasts;

    public const TABLE = 'job_queue';

    /**
     * @throws Exception
     */
    public static function create(
        mixed $sequence,
        string $handler,
        DateTimeImmutable $created,
        array $contents
    ): QueueJob {
        $job = new QueueJob();

        $job->sequence = $sequence;
        $job->handler = $handler;
        $job->created = $created;
        $job->contents = new EtlSyncJobData(
            table: $contents['table'],
            data: $contents['data']
        );
        $job->store();

        return $job;
    }

    public function casts(): array
    {
        return [
            ...parent::casts(),
            'created' => Casts::DATETIME,
            'contents' => Casts::SERIALIZE,
        ];
    }

    public function run(): void
    {
        /** @var QueueTaskHandlerInterface $handler */
        $handler = new $this->handler($this->contents);

        $handler->run();
    }

    public function lock(): void
    {
        $this->locked = true;

        $this->store();
    }

    public function unlock(): void
    {
        $this->locked = false;

        $this->store();
    }

    public function markProcessed(): void
    {
        $this->processed = new DateTimeImmutable();

        $this->store();
    }

    public function handleFailure(): void
    {
        $this->incrementAttempts();

        if ($this->attempts >= 3) {
            $this->markFailed();
        }
    }

    private function incrementAttempts(): void
    {
        $this->attempts++;

        $this->store();
    }

    private function markFailed(): void
    {
        $this->failed = new DateTimeImmutable();

        $this->store();
    }
}
