<?php

declare(strict_types=1);

namespace apexl\Io\modules\jobs\Service;

use apexl\Io\modules\jobs\Interface\QueuedTaskFinderInterface;
use apexl\Io\modules\jobs\Interface\QueueTaskInterface;
use Spatie\Async\Pool;
use Throwable;

final readonly class TaskRunnerService
{
    private Pool $pool;

    public function __construct(
        private QueuedTaskFinderInterface $queuedTaskFinder
    ) {
        $this->pool = Pool::create();
    }

    public function run(): void
    {
        $tasks = $this->queuedTaskFinder->tasks();

        foreach ($tasks as $task) {
            $this->addTaskToPool($task);
        }

        $this->pool->wait();
    }

    private function addTaskToPool(QueueTaskInterface $task): void
    {
        $this->pool->add(function () use ($task): void {
            if ($task->isLocked()) {
                return;
            }
            $task->lock();
            logger('queue')->debug('Starting {task}', [
                'task' => $task->name(),
            ]);
            $task->run();
        })
            ->then(function () use ($task) {
                logger('queue')->debug('Completed {task}', [
                    'task' => $task->name(),
                ]);
                $task->markProcessed();
                $task->unlock();
            })->catch(function (Throwable $throwable) use ($task) {
                logger('queue')->error('Error running {task}: {error}', [
                    'task' => $task->name(),
                    'error' => $throwable->getMessage(),
                    'exception' => $throwable,
                ]);
                $task->handleFailure();
                $task->unlock();
            });
    }

}
