<?php

declare(strict_types=1);

namespace apexl\Io\modules\jobs\Entity;

use apexl\entityCore\enums\Casts;
use apexl\entityCore\traits\hasCasts;
use apexl\Io\includes\System;
use apexl\Io\modules\jobs\Interface\QueueTaskHandlerInterface;
use app\module\newtonsTrailers\Dto\EtlSyncJobData;
use app\module\newtonsTrailers\Entity\Entity;
use DateTimeImmutable;
use Exception;
use Throwable;

/**
 * @property int $id
 * @property int $sequence
 * @property class-string $handler
 * @property DateTimeImmutable $created
 * @property bool $locked
 * @property int $attempts
 * @property DateTimeImmutable $processed
 * @property DateTimeImmutable $failed
 * @property EtlSyncJobData $contents
 */
class QueueJob extends Entity
{
    use hasCasts;

    public const TABLE = 'job_queue';

    /**
     * @throws Exception
     */
    public static function create(
        mixed $sequence,
        string $handler,
        DateTimeImmutable $created,
        array $contents
    ): QueueJob {
        $job = new QueueJob();

        $job->sequence = $sequence;
        $job->handler = $handler;
        $job->created = $created;
        $job->contents = new EtlSyncJobData(
            table: $contents['table'],
            data: $contents['data']
        );
        $job->store();

        return $job;
    }

    public function casts(): array
    {
        return [
            ...parent::casts(),
            'created' => Casts::DATETIME,
            'contents' => Casts::SERIALIZE,
        ];
    }

    public function run(): void
    {
        $this->handler()->run();
    }

    private function handler(): QueueTaskHandlerInterface
    {
        /** @var ?QueueTaskHandlerInterface $handler */
        static $handler = null;

        if ($handler === null) {
            $handler = System::makeRegisteredService($this->handler, [
                'data' => $this->contents,
            ]);
        }

        return $handler;
    }

    public function lock(): void
    {
        $this->locked = true;

        $this->store();
    }

    public function unlock(): void
    {
        $this->locked = false;

        $this->store();
    }

    public function markProcessed(): void
    {
        $this->processed = new DateTimeImmutable();

        $this->store();
    }

    public function handleFailure(string $task, Throwable $throwable): void
    {
        $this->incrementAttempts();
        $this->emptyHandler()->handleFailure($task, $throwable);
    }

    private function incrementAttempts(): void
    {
        $this->attempts++;

        $this->store();
    }

    private function emptyHandler(): QueueTaskHandlerInterface
    {
        /** @var ?QueueTaskHandlerInterface $handler */
        return System::makeRegisteredService($this->handler, [
            'data' => null,
        ]);
    }

    private function markFailed(): void
    {
        $this->failed = new DateTimeImmutable();

        $this->store();
    }
}
