<?php

declare(strict_types=1);

namespace apexl\Io\modules\jobs\Entity;

use apexl\entityCore\enums\Casts;
use apexl\entityCore\traits\hasCasts;
use apexl\Io\includes\Entity;
use apexl\Io\includes\System;
use apexl\Io\modules\jobs\Interface\QueueTaskHandlerInterface;
use apexl\Vault\Vault;
use DateTimeImmutable;
use Exception;
use Throwable;

/**
 * @property int $id
 * @property int $sequence
 * @property class-string $handler
 * @property DateTimeImmutable $created
 * @property bool $locked
 * @property int $attempts
 * @property DateTimeImmutable $processed
 * @property DateTimeImmutable $failed
 * @property mixed $contents
 */
class QueueJob extends Entity
{
    use hasCasts {
        hasCasts::casts as traitCasts;
    }

    public const TABLE = 'job_queue';
    private const ATTEMPTS_BEFORE_FAILURE = 1;

    public function __construct()
    {
        parent::__construct(self::TABLE);
    }

    /**
     * @throws Exception
     */
    public static function create(
        string $handler,
        mixed $contents,
        string|int|null $sequence = null,
        ?DateTimeImmutable $created = null,
    ): QueueJob {
        $job = new QueueJob();

        $job->sequence = $sequence ?? self::nextSequence();
        $job->handler = $handler;
        $job->created = $created ?? new DateTimeImmutable();
        $job->contents = $contents;
        $job->store();

        return $job;
    }

    public static function nextSequence(): int
    {
        return (int) Vault::getInstance()
            ->select('job_queue')
            ->fields(['(IFNULL(MAX(sequence),0) + 1) AS sequence'])
            ->execute()
            ->fetch()->sequence;
    }

    public function casts(): array
    {
        return [
            ...$this->traitCasts(),
            'created' => Casts::DATETIME,
            'contents' => Casts::SERIALIZE,
        ];
    }

    public function run(): void
    {
        $this->handler()->run();
    }

    private function handler(): QueueTaskHandlerInterface
    {
        return System::makeRegisteredService($this->handler, [
            'data' => $this->contents,
        ]);
    }

    public function lock(): void
    {
        $this->locked = true;

        $this->store();
    }

    public function unlock(): void
    {
        $this->locked = false;

        $this->store();
    }

    public function markProcessed(): void
    {
        $this->processed = new DateTimeImmutable();

        /** @Todo Add config to turn clearing contents on / off */
        $this->data['contents'] = '';

        $this->store();
    }

    public function handleFailure(string $task, Throwable $throwable): void
    {
        $this->incrementAttempts();
        if ($this->attempts > self::ATTEMPTS_BEFORE_FAILURE) {
            $this->markFailed();
        }

        $this->emptyHandler()->handleFailure($task, $throwable);
    }

    private function incrementAttempts(): void
    {
        $this->attempts++;

        $this->store();
    }

    private function markFailed(): void
    {
        $this->failed = new DateTimeImmutable();

        $this->store();
    }

    private function emptyHandler(): QueueTaskHandlerInterface
    {
        /** @var ?QueueTaskHandlerInterface $handler */
        return System::makeRegisteredService($this->handler, [
            'data' => null,
        ]);
    }
}
