<?php

declare(strict_types=1);

namespace apexl\Io\modules\logger\services;

use apexl\Config\Singleton as Config;
use apexl\Io\includes\System;
use apexl\Io\modules\logger\dto\handlerConfig;
use apexl\Io\modules\logger\dto\handlerConfigParams\fileHandlerConfigParams;
use apexl\Io\modules\logger\dto\handlerConfigParams\logstashHandlerConfigParams;
use apexl\Io\modules\logger\dto\streamConfig;
use apexl\Io\modules\logger\enums\Handler;
use apexl\Io\modules\logger\interfaces\loggerInterface;
use Closure;
use Monolog\Formatter\LogstashFormatter;
use Monolog\Handler\ErrorLogHandler;
use Monolog\Handler\HandlerInterface;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Handler\SocketHandler;
use Psr\Log\AbstractLogger;
use Throwable;
use UnexpectedValueException;

class logger extends AbstractLogger implements loggerInterface
{
    private \Monolog\Logger $driver;
    private streamConfig $config;
    private string $stream;

    public function __construct(string $stream, streamConfig $config)
    {
        $this->stream = $stream;
        $driver = new \Monolog\Logger($stream);
        $driver->setExceptionHandler(Closure::fromCallable(function (\Throwable $throwable) : void {
            $this->handleThrowable($throwable);
        }));
        $this->config = $config;

        $driver->setHandlers($this->handlers());

        $this->driver = $driver;
    }

    public function log($level, $message, array $context = [])
    {
        $this->driver->log($level, $message, $context);
    }

    /**
     * @return HandlerInterface[]
     */
    private function handlers(): array
    {
        $handlers = [];
        foreach ($this->config->handlers as $handler) {
            $handlers[] = $this->handler($handler);
        }

        return $handlers;
    }

    private function handler(handlerConfig $handlerConfig): ?HandlerInterface
    {
        try {
            $destination = Handler::from($handlerConfig->destination);

            $level = ($handlerConfig->level ?? $this->config->level)->getValue();

            if ($destination->equals(Handler::ERROR_LOG())) {
                return new ErrorLogHandler(
                    ErrorLogHandler::OPERATING_SYSTEM,
                    $level,
                );
            }

            if ($destination->equals(Handler::FILE())) {
                /** @var fileHandlerConfigParams $params */
                $params = $handlerConfig->params;
                $file = $params->file ?? sprintf('%s.log', $this->stream);
                if (substr($file, 0, 1) !== '/') {
                    $file = sprintf('%s/../%s/%s', System::getBasePath(), $this->config->logDirectory, $file);
                }

                return new RotatingFileHandler($file, $params->maxFiles ?? 7, $level);
            }

            if ($destination->equals(Handler::LOGSTASH())) {
                /** @var logstashHandlerConfigParams $params */
                $params = $handlerConfig->params;
                $connectionString = sprintf(
                    '%s:%s',
                    $params->host,
                    $params->port
                );

                $handler = new SocketHandler($connectionString, $level);

                $handler->setFormatter(new LogstashFormatter(
                    $params->application ?? Config::getInstance()->app->site->name,
                    $params->system,
                ));

                return $handler;
            }

            $this->unknownHandler($handlerConfig->destination);
        } catch (UnexpectedValueException $e) {
            $this->unknownHandler($handlerConfig->destination, $e->getMessage());
        }

        return null;
    }

    private function unknownHandler(string $handler, string $append = '')
    {
        $message = sprintf('Unknown Handler \'%s\'', $handler);

        if ($append !== '' && $append !== '0') {
            $message = sprintf('%s: %s', $message, $append);
        }

        error_log(
            $message,
            E_USER_WARNING
        );
    }

    private static function formatThrowable(Throwable $throwable, string $prepend = ''): string
    {
        $message = sprintf('(%d) %s', $throwable->getCode(), $throwable->getMessage());

        if ($prepend !== '' && $prepend !== '0') {
            $message = sprintf('%s: %s', $prepend, $message);
        }

        return $message;
    }

    public static function throwable(
        Throwable $throwable,
        string $prepend = '',
        $context = [],
        $stream = loggerStore::DEFAULT__ERROR_STREAM
    ): void {
        $logger = loggerStore::getInstance()->logger($stream);

        $message = self::formatThrowable($throwable, $prepend);

        $logger->error($message, $context);
    }

    /**
     * @return never
     */
    public static function throwableAndExit(
        Throwable $throwable,
        string $prepend = '',
        $context = [],
        $httpCode = 500,
        $stream = loggerStore::DEFAULT__ERROR_STREAM
    ): void {
        logger::throwable($throwable, $prepend, $context, $stream);

        http_response_code($httpCode);
        exit;
    }

    private function handleThrowable(Throwable $throwable): void
    {
        error_log(
            self::formatThrowable($throwable, 'Error handling log message'),
            E_USER_WARNING
        );
    }
}
