<?php

declare(strict_types=1);

namespace apexl\Io\modules\logger\services;

use apexl\Config\Singleton as Config;
use apexl\Io\includes\System;
use apexl\Io\modules\logger\dto\handlerConfig;
use apexl\Io\modules\logger\dto\handlerConfigParams\fileHandlerConfigParams;
use apexl\Io\modules\logger\dto\handlerConfigParams\logstashHandlerConfigParams;
use apexl\Io\modules\logger\dto\streamConfig;
use apexl\Io\modules\logger\enums\Handler;
use apexl\Io\modules\logger\interfaces\loggerInterface;
use Exception;
use Monolog\Formatter\LogstashFormatter;
use Monolog\Handler\ErrorLogHandler;
use Monolog\Handler\HandlerInterface;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Handler\SocketHandler;
use Monolog\Processor\PsrLogMessageProcessor;
use Psr\Log\AbstractLogger;
use Throwable;
use UnexpectedValueException;

class logger extends AbstractLogger implements loggerInterface
{
    private readonly \Monolog\Logger $driver;

    /**
     * @throws Exception
     */
    public function __construct(private readonly string $stream, private readonly streamConfig $config)
    {
        $driver = new \Monolog\Logger($stream);
        $driver->setExceptionHandler(
            (function (Throwable $throwable): void {
                $this->handleThrowable($throwable);
            })(...)
        );

        $driver->setHandlers($this->handlers());
        $driver->pushProcessor(new PsrLogMessageProcessor(removeUsedContextFields: true));

        $this->driver = $driver;
    }

    private function handleThrowable(Throwable $throwable): void
    {
        error_log(
            self::formatThrowable($throwable, 'Error handling log message'),
            E_USER_WARNING
        );
    }

    private static function formatThrowable(Throwable $throwable, string $prepend = ''): string
    {
        $message = sprintf('(%d) %s', $throwable->getCode(), $throwable->getMessage());

        if ($prepend !== '' && $prepend !== '0') {
            $message = sprintf('%s: %s', $prepend, $message);
        }

        return $message;
    }

    /**
     * @return HandlerInterface[]
     * @throws Exception
     */
    private function handlers(): array
    {
        $handlers = [];
        foreach ($this->config->handlers as $handler) {
            $handlers[] = $this->handler($handler);
        }

        return $handlers;
    }

    /**
     * @throws Exception
     */
    private function handler(handlerConfig $handlerConfig): ?HandlerInterface
    {
        try {
            $destination = Handler::from($handlerConfig->destination);

            $level = ($handlerConfig->level ?? $this->config->level)->value;

            if ($destination === Handler::ERROR_LOG) {
                return new ErrorLogHandler(
                    ErrorLogHandler::OPERATING_SYSTEM,
                    $level,
                );
            }

            if ($destination === Handler::FILE) {
                /** @var fileHandlerConfigParams $params */
                $params = $handlerConfig->params;
                $file = $params->file ?? sprintf('%s.log', $this->stream);
                if (!str_starts_with($file, '/')) {
                    $file = sprintf('%s/../%s/%s', System::getBasePath(), $this->config->logDirectory, $file);
                }

                return new RotatingFileHandler($file, $params->maxFiles ?? 7, $level);
            }

            if ($destination === Handler::LOGSTASH) {
                /** @var logstashHandlerConfigParams $params */
                $params = $handlerConfig->params;
                $connectionString = sprintf(
                    '%s:%s',
                    $params->host,
                    $params->port
                );

                $handler = new SocketHandler($connectionString, $level);

                $handler->setFormatter(
                    new LogstashFormatter(
                        $params->application ?? Config::getInstance()->app->site->name,
                        $params->system,
                    )
                );

                return $handler;
            }

            $this->unknownHandler($handlerConfig->destination);
        } catch (UnexpectedValueException $e) {
            $this->unknownHandler($handlerConfig->destination, $e->getMessage());
        }

        return null;
    }

    private function unknownHandler(string $handler, string $append = ''): void
    {
        $message = sprintf('Unknown Handler \'%s\'', $handler);

        if ($append !== '' && $append !== '0') {
            $message = sprintf('%s: %s', $message, $append);
        }

        error_log(
            $message,
            E_USER_WARNING
        );
    }

    public static function throwableAndExit(
        Throwable $throwable,
        string $prepend = '',
        $context = [],
        $httpCode = 500,
        $stream = loggerStore::DEFAULT__ERROR_STREAM
    ): never {
        logger::throwable($throwable, $prepend, $context, $stream);

        http_response_code($httpCode);
        exit;
    }

    /** @noinspection PhpUnused */

    public static function throwable(
        Throwable $throwable,
        string $prepend = '',
        $context = [],
        $stream = loggerStore::DEFAULT__ERROR_STREAM
    ): void {
        $logger = loggerStore::getInstance()->logger($stream);

        $message = self::formatThrowable($throwable, $prepend);

        $logger->error($message, $context);
    }

    public function log($level, $message, array $context = []): void
    {
        $this->driver->log($level, $message, $context);
    }
}
