<?php

declare(strict_types=1);

namespace apexl\Io\modules\logger\services;

use apexl\Io\modules\logger\dto\handlerConfig;
use apexl\Io\modules\logger\dto\streamConfig;
use apexl\Io\modules\logger\enums\Handler;
use apexl\Io\modules\logger\interfaces\loggerInterface;
use apexl\Io\modules\logger\services\handlerCreator\handlerCreatorFactory;
use Exception;
use Monolog\Handler\HandlerInterface;
use Monolog\Processor\PsrLogMessageProcessor;
use Psr\Log\AbstractLogger;
use Throwable;
use UnexpectedValueException;

class logger extends AbstractLogger implements loggerInterface
{
    private readonly \Monolog\Logger $driver;

    /**
     * @throws Exception
     */
    public function __construct(
        private readonly string $stream,
        private readonly streamConfig $config,
        private readonly handlerCreatorFactory $handlerCreatorFactory
    ) {
        $driver = new \Monolog\Logger($stream);
        $driver->setExceptionHandler(
            (function (Throwable $throwable): void {
                $this->handleThrowable($throwable);
            })(...)
        );

        $driver->setHandlers($this->handlers());
        $driver->pushProcessor(new PsrLogMessageProcessor(removeUsedContextFields: true));

        $this->driver = $driver;
    }

    private function handleThrowable(Throwable $throwable): void
    {
        error_log(
            self::formatThrowable($throwable, 'Error handling log message'),
            E_USER_WARNING
        );
    }

    private static function formatThrowable(Throwable $throwable, string $prepend = ''): string
    {
        $message = sprintf('(%d) %s', $throwable->getCode(), $throwable->getMessage());

        if ($prepend !== '' && $prepend !== '0') {
            $message = sprintf('%s: %s', $prepend, $message);
        }

        return $message;
    }

    /**
     * @return HandlerInterface[]
     * @throws Exception
     */
    private function handlers(): array
    {
        $handlers = [];
        foreach ($this->config->handlers as $handler) {
            $handlers[] = $this->handler($handler);
        }

        return $handlers;
    }

    /**
     * @throws Exception
     */
    private function handler(handlerConfig $handlerConfig): ?HandlerInterface
    {
        try {
            $destination = Handler::from($handlerConfig->destination);

            return $this->handlerCreatorFactory->make(
                $destination,
            )->create(
                $this->stream,
                $handlerConfig,
                $this->config,
            );
        } catch (UnexpectedValueException $e) {
            $this->unknownHandler($handlerConfig->destination, $e->getMessage());
        }

        return null;
    }

    private function unknownHandler(string $handler, string $append = ''): void
    {
        $message = sprintf('Unknown Handler \'%s\'', $handler);

        if ($append !== '' && $append !== '0') {
            $message = sprintf('%s: %s', $message, $append);
        }

        error_log(
            $message,
            E_USER_WARNING
        );
    }
    
    public function log($level, $message, array $context = []): void
    {
        $this->driver->log($level, $message, $context);
    }
}
