<?php

namespace apexl\Io\modules\menu\entities;

use apexl\Io\includes\Entity;
use apexl\Io\includes\Hook;
use apexl\Io\includes\Routes;
use apexl\Io\modules\menu\entities\operators\menuOperator;
use apexl\Io\modules\menu\includes\menuItem;
use apexl\Io\modules\user\entities\userEntity;
use apexl\Io\services\pathUtility;
use apexl\Utils\Arrays\Merge;

class menuEntity extends Entity
{
    protected $menuItems;
    protected $parentKeys = [];
    protected $storeLinks = [];

    public function __construct()
    {
        parent::__construct('menus');
        $this->setOperator(new menuOperator('menus'));
        //set Defaults
        $this->name = 'navigation';
        $this->module = 'menu';
        $this->permission = 'AllowAll';
        $this->active = 1;
        $this->links = [];
    }

    /**
     * @param $name
     * @param false $skipAccessCheck
     * @throws \Exception
     */
    public function loadByName($name, $skipAccessCheck = false)
    {
        parent::__call('loadByName', [$name, $skipAccessCheck]);
        $this->buildMenu();
    }

    /**
     * @param $routeName
     * @param string $title
     * @param string $icon
     * @param int $weight
     * @param bool $parentRoute
     * @return $this
     */
    public function addToMenu($routeName, $title = '', $icon = '', $weight = 0, $parentRoute = false, $replacements = [])
    {
        $link = [
            'routeName' => $routeName,
            'title' => $title,
            'icon' => $icon,
            'weight' => $weight,
            'replacements' => $replacements
        ];

        $key = md5(json_encode($link));

        // If this is a child, set sub-items of parent
        if ($parentRoute) {
            if (isset($this->parentKeys[$parentRoute])) { // Parent already processed - add as child
                if (!isset($this->data['links'][$this->parentKeys[$parentRoute]])) {
                //set some defaults, just incase.
                    $this->data['links'][$this->parentKeys[$parentRoute]] = [
                    'routeName' => $parentRoute,
                    'title' => $parentRoute,
                ];
            }
                $this->data['links'][$this->parentKeys[$parentRoute]]['links'][$key] = $link;
            } else { // Parent not yet added, store and add it later
                $this->storeLinks[$parentRoute][$key] = $link;
            }

        } elseif (isset($this->data['links'][$key]['links'])) { // if we're overriding, keep child links if this is already set.
            $childLinks = $this->data['links'][$key]['links'];
            $this->data['links'][$key] = $link;
            $this->data['links'][$key]['links'] = $childLinks;
            $this->parentKeys[$routeName] = $key;
        } else { // Add a non-child link
            if (isset($this->storeLinks[$routeName])) {
                $this->data['links'] = $this->storeLinks[$routeName];
            }
            $this->data['links'][$key] = $link;
            $this->parentKeys[$routeName] = $key;
        }
        return $this;
    }

    /**
     * Build the entity into a menu structure for display.
     * @todo perhaps move this into something different like a display module?
     * @param userEntity $user
     * @param bool $filterByPermission
     * @return array
     */
    public function getAllItemsAsMenu(userEntity $user, $filterByPermission = true): array
    {
        $this->buildMenu();
        $menu = [];
        foreach ($this->menuItems as $menuGroup) {
            if ($filterByPermission) {
                $item = $this->checkAccess($menuGroup, $user);
                if (!empty((array)$item)) { //typecast in case of empty object.
                    $menu[] = $item;
                }
            }
        }
        return $menu;
    }

    /**
     * Generate breadcrumbs for the provided path.
     * @param $route
     * @return array
     */
    public function generateBreadcrumbs($route): array
    {
        //get the parent item and return the stack (usually needed for breadcrumbs)
        $pathUtility = pathUtility::getInstance();
        $currentRouteName = $pathUtility->getCurrentRouteName();
        $currentPath = $pathUtility->getPath();
        //echo $currentRouteName . ' ||| ' . $currentPath . "\n\n";
        $this->buildMenu();
        $parts = explode('/', ltrim($currentPath, '/'));
        //we want to check each component for a matching route. If the component contains a dynamic key, we need to check for callbacks
        $reconstructed = '';
        $stack = [];

        foreach ($parts as $part) {
            $reconstructed = $reconstructed.'/'.$part;
            if (!empty($this->menuItems)) {
                foreach ($this->menuItems as $item) {
                    //  echo "\n".$item->href . ' --- ' . $reconstructed;
                    // print_r($item);
                    if ($item->href == $reconstructed) {
                        $stack[] = $item;
                    } elseif (isset($item->links)) {
                        foreach ($item->links as $childItem) {
                            if ($childItem->href == $reconstructed) {
                                $parentItem = clone $item;
                                unset($parentItem->href);
                                $stack[] = $parentItem;
                                $stack[] = $childItem;
                            }
                        }
                    }
                }
            }
        }

        return $stack;
    }

    /**
     * Check access to the provided menu items.
     * @param \stdClass $menuItem
     * @param userEntity $user
     * @return menuItem|object
     */
    protected function checkAccess($menuItem, userEntity $user)
    {
        $allowed = (object)[];
        if (isset($menuItem->href)) {
            if ($user->isAllowed($menuItem->permission)) {
                $allowed = clone $menuItem;
                if (isset($menuItem->links)) {
                    //clear the links from the allowed var since we want to check access.
                    unset($allowed->links);
                    foreach ($menuItem->links as $link) {
                        if (!empty((array)$this->checkAccess($link, $user))) {
                            $allowed->links[] = $link;
                        }
                    }
                }
            }
        } else {
            $allowed = $menuItem;
        }
        return $allowed;
    }

    /**
     * @param $routeName
     * @return false|mixed
     */
    public function getMenuItem($routeName)
    {
        //strip any params
        if (($params = strpos($routeName, '?')) !== false) { //Bool safe check
            list($routeName) = explode('?', $routeName);
        }
        if (empty($routeName)) {
            $routeName = '/';
        }

        //check for callbacks and override if available.
        return $this->links[$routeName] ?? $this->checkSublinks($routeName) ?? false;
    }

    /**
     * @param $routeName
     * @return false|mixed
     */
    protected function checkSublinks($routeName)
    {
        foreach ($this->links as $item) {
            if (!$links = $item->links ?? false) {
                continue;
            }
            foreach ($item->links as $subLink) {
                if ($subLink->routeName == $routeName) {
                    return $subLink;
                }
            }
        }

        return false;
    }

    /**
     * @todo This needs to be cached.
     * @return void
     */
    protected function buildMenu()
    {
        $menuItems = [];
        if (!empty($this->data['links'])) {
            foreach ($this->data['links'] as $item) {
                $menuItem = new menuItem($item, Routes::getRoute($item['routeName']));
                $menuItem = Hook::processHook('menuItemAlter', $menuItem);

                //do we have child links? Is so, loop and load.
                if ($item['links'] ?? false) {
                    $subLinks = [];
                    foreach ($item['links'] as $subItem) {
                        $subMenuItem = new menuItem($subItem, Routes::getRoute($subItem['routeName']));
                        $subMenuItem = Hook::processHook('menuItemAlter', $subMenuItem);
                        //weighting, treat 0 as 'stack' so we dont do this on every pass..
                        if ($menuItem->getWeight() !== 0) {
                            Merge::arrayByWeight($subMenuItem->getWeight(), (object)get_object_vars($subMenuItem), $subLinks);
                        } else {
                            $subLinks[] = (object)get_object_vars($subMenuItem);
                        }
                    }
                    $menuItem->setLinks($subLinks);
                }
                //test if this is an empty parent. If so, and the parent doesn't link, omit it completely.
                //parent weighting, treat 0 as 'stack' so we dont do this on every pass..
                if ($menuItem->getWeight() !== 0) {
                    Merge::arrayByWeight($menuItem->getWeight(), (object)get_object_vars($menuItem), $menuItems);
                } else {
                    $menuItems[] = (object)get_object_vars($menuItem);
                }
            }
        }

        //finally loop over the re-weighted items to key by path.
        if (!empty($menuItems)) {
            foreach ($menuItems as $menuItem) {
                $this->menuItems[$menuItem->href] = $menuItem;
            }
        }
    }

    public function isCurrentMenu(): bool
    {
        $pathUtility = pathUtility::getInstance();
        $currentRouteName = $pathUtility->getCurrentRouteName();

        if (!empty($this->menuItems)) {
            foreach ($this->menuItems as $item) {
                if ($item->name == $currentRouteName) {
                    return true;
                }
            }
        }

        return false;
    }
}
