<?php

declare(strict_types=1);

namespace app\module\apollo\Helper;

use apexl\Io\modules\file\entities\fileEntity;
use apexl\Io\modules\mirror\Interface\Uploadable;
use apexl\Vault\operators\drivers\mysql;
use apexl\Vault\Vault;
use Mimey\MimeTypes;

class UploadableFileEntity implements Uploadable
{
    public const TABLE__LOCK = 'file_locks';
    public const TYPE__LOCKED = 'locked';
    public const TYPE__UPLOADED = 'uploaded';

    private fileEntity $fileEntity;
    private Vault $vault;

    public function __construct(fileEntity $fileEntity)
    {
        $this->vault = Vault::getInstance();
        $this->fileEntity = $fileEntity;
    }

    public static function fromId(int $id): UploadableFileEntity
    {
        $fileEntity = new fileEntity();
        $fileEntity->load($id);

        return self::fromFileEntity($fileEntity);
    }

    public static function fromFileEntity(fileEntity $fileEntity): UploadableFileEntity
    {
        return new UploadableFileEntity($fileEntity);
    }

    public function getKey(): string
    {
        return sprintf('%s/%s', dirname($this->fileEntity->path), $this->fileEntity->name);
    }

    public function getPath(): string
    {
        return $this->fileEntity->getLocalPath();
    }

    public function unlockUpload(): void
    {
        $this->removeLock(self::TYPE__UPLOADED);
    }

    public function removeLock(string $type): void
    {
        /** @var mysql $vault */
        $vault = $this->vault;
        $vault->delete(self::TABLE__LOCK)
            ->where('file', $this->fileEntity->id)
            ->where('type', $type)
            ->execute();
    }

    public function lockUpload(): void
    {
        $this->addLock(self::TYPE__LOCKED);
    }

    private function addLock(string $type): void
    {
        /** @var mysql $vault */
        $vault = $this->vault;
        $vault->insert(self::TABLE__LOCK)
            ->fields([
                'file' => $this->fileEntity->id,
                'type' => $type,
                'datetime' => date('Y-m-d H:i:s'),
            ])
            ->execute();
    }

    public function markUploaded(): void
    {
        $this->addLock(self::TYPE__UPLOADED);
    }

    public function isUploaded(): bool
    {
        return $this->hasLock(self::TYPE__UPLOADED);
    }

    private function hasLock(string $type): bool
    {
        /** @var mysql $vault */
        $vault = $this->vault;
        $result = $vault->select(self::TABLE__LOCK)
            ->fields('id')
            ->where('file', $this->fileEntity->id)
            ->where('type', $type)
            ->limit(1)
            ->execute()
            ->fetchAll();

        return count($result) > 0;
    }

    public function isLocked(): bool
    {
        return $this->hasLock(self::TYPE__LOCKED);
    }

    public function getMimeType(): string
    {
        $mimey = new MimeTypes();
        preg_match('/\.(\w+)$/', $this->fileEntity->name, $match);
        [, $ext] = $match;

        return $mimey->getMimeType($ext);
    }
}
