<?php

namespace apexl\Io\modules\payment\providers;

use apexl\Config\Singleton;
use apexl\Io\modules\payment\entities\paymentLogEntity;
use Stripe\Checkout\Session;
use Stripe\Stripe;
use Stripe\StripeClient;

class stripeProvider implements providerInterface {

    protected $stripe;

    public function __construct()
    {
        $config = Singleton::getInstance();
        Stripe::setApiKey($config->app->payments->stripe->secret_key);
        $this->stripe = new StripeClient($config->app->payments->stripe->secret_key);
    }

    public function getOneTimePaymentUrl($products, $successUrl, $cancelUrl)
    {
        $options = [
            'line_items' => [],
            'mode' => 'payment',
            'success_url' => $successUrl.'/{CHECKOUT_SESSION_ID}',
            'cancel_url' => $cancelUrl.'/{CHECKOUT_SESSION_ID}',
        ];

        foreach ($products as $product) {
            $options['line_items'][] = [
                'price_data' => [
                    'currency' => 'gbp',
                    'product_data' => [
                        'name' => $product['name'],
                    ],
                    'unit_amount' => ($product['unit_price']*100), // stripe amounts are in pence
                ],
                'quantity' => $product['quantity'],
            ];
        }

        $session = Session::create($options);

        return $session->url;
    }
    
    public function getCardValidationUrl($successUrl, $cancelUrl, $localTokenId, $customerRef, $meta)
    {
        $options = [
            'payment_method_types' => ['card'],
            'mode' => 'setup',
            'success_url' => "$successUrl/{CHECKOUT_SESSION_ID}/$localTokenId",
            'cancel_url' => $cancelUrl,
        ];
        if ($customerRef) $options['customer'] = $customerRef;
        if ($meta) $options['metadata'] = $meta;

        $session = Session::create($options);

        return $session->url;
    }

    public function getPaymentIntentIdFromSessionId($sessionId)
    {
        $session = $this->makeRequest('checkout', 'sessions', $sessionId, [], "retrieve");

        return $session->payment_intent ?? null;
    }

    public function retrieveToken($sessionId)
    {
        $session = $this->makeRequest('checkout', 'sessions', $sessionId, [], "retrieve");
        //$session = $this->stripe->checkout->sessions->retrieve($sessionId, []);
        $paymentMethodId = $this->retrievePaymentMethodFromIntent($session->setup_intent);
        $this->setupOffSessionUsage($session->setup_intent);

        return [$session->setup_intent ?? null, $paymentMethodId];
    }

    public function attachCustomerToPaymentMethod($paymentMethodId, $email, $fullName, $customerId=null)
    {
        if (!$customerId) $customerId = $this->createCustomer($email, $fullName);
        if ($customerId) {
            $response = $this->makeRequest('paymentMethods', 'attach', $paymentMethodId, ['customer' => $customerId]);
            /*$response = $this->stripe->paymentMethods->attach(
                $paymentMethodId,
                ['customer' => $customerId]
            );*/

            return $response->customer ?? false;
        }

        return false;
    }
    
    public function capturePaymentWithToken($paymentMethodId, $customerRef=null, $amount=0, $description = '', $currency='gbp', $metaData=null)
    {
        $paymentIntentId = $this->createPaymentIntent($paymentMethodId, $customerRef, $amount, $description, $currency='gbp', $metaData);

        if ($paymentIntentId) {
            $response = $this->makeRequest('paymentIntents', 'confirm', $paymentIntentId, ['payment_method' => $paymentMethodId]);
            /*$response = $this->stripe->paymentIntents->confirm(
                $paymentIntentId,
                ['payment_method' => $paymentMethodId]
            );*/
            $chargeId = $response->charges->data[0]->id ?? null;

            if (isset($response->status) && $response->status == 'succeeded') return [$paymentIntentId, $chargeId];
        }

        return [false, false];
    }

    protected function createPaymentIntent($paymentMethodId, $customerRef, $amount, $description = '', $currency='gbp', $metaData=null)
    {
        if ($amount > 0) {
            $options = [
                'amount' => ($amount*100), // Stripe deals in pence or cents etc.  i.e. the smallest currency unit
                'currency' => $currency,
                'payment_method' => $paymentMethodId,
                'setup_future_usage' => 'off_session',
                'description' => $description,
                'metadata' => $metaData,
            ];
            if ($customerRef) $options['customer'] = $customerRef;

            $response = $this->makeRequest('paymentIntents', 'create', null, $options);
            //$response = $this->stripe->paymentIntents->create($options);
        }

        return $response->id ?? null;
    }

    protected function retrievePaymentMethodFromIntent($setupIntentId)
    {
        $setupIntent = $this->makeRequest('setupIntents', 'retrieve', $setupIntentId);
        //$setupIntent = $this->stripe->setupIntents->retrieve($setupIntentId, []);

        return $setupIntent->payment_method ?? null;
    }

    protected function setupOffSessionUsage($setupIntentId)
    {
        $setupIntent = $this->makeRequest('setupIntents', 'retrieve', $setupIntentId);
        //$setupIntent = $this->stripe->setupIntents->retrieve($setupIntentId, []);
        if (isset($setupIntent->usage) && $setupIntent->usage != "off_session") {
            $this->makeRequest('setupIntents', 'update', $setupIntentId, ['usage' => 'off_session']);
            /*$this->stripe->setupIntents->update(
                $setupIntentId,
                ['usage' => 'off_session']
            );*/
        }
        return;
    }

    protected function createCustomer($email, $fullName)
    {
        $response = $this->makeRequest('customers', 'create', null, [
            'email' => $email,
            'name' => $fullName,
        ]);

        /*$response = $this->stripe->customers->create([
            'email' => $email,
            'name' => $fullName,
        ]);*/

        return $response->id ?? null;
    }

    protected function makeRequest($apiName, $methodName, $id=null, $data=[], $subMethodName=null)
    {
        $logData = $data;

        try {

            if (!is_null($id)) {
                if ($subMethodName) {
                    $response = $this->stripe->$apiName->$methodName->$subMethodName($id, $data);
                    $methodName = "$methodName > $subMethodName";
                } else {
                    $response = $this->stripe->$apiName->$methodName($id, $data);
                }
            } else {
                $response = $this->stripe->$apiName->$methodName($data);
            }

            $logResponse = $response;

        } catch(\Stripe\Exception\CardException $e) { // Payment decline
            $logResponse = (object)[
                'Error' => 'Stripe API: Payment Exception. ' . $e->getMessage(),
                'Status' => $e->getHttpStatus(),
                'Type' => $e->getError()->type,
                'Code' => $e->getError()->code,
                'Param' => $e->getError()->param,
                'Message' => $e->getError()->message
            ];
        } catch (\Stripe\Exception\RateLimitException $e) {
            $logResponse = (object)['Error' => 'Stripe API: rate limit exceeded. ' . $e->getMessage()];
        } catch (\Stripe\Exception\InvalidRequestException $e) {
            $logResponse = (object)['Error' => 'Stripe API: Invalid parameters were supplied. ' . $e->getMessage()];
        } catch (\Stripe\Exception\AuthenticationException $e) {
            $logResponse = (object)['Error' => 'Stripe API: Authentication failed. ' . $e->getMessage()];
        } catch (\Stripe\Exception\ApiConnectionException $e) {
            $logResponse = (object)['Error' => 'Stripe API: Network communication failed. ' . $e->getMessage()];
        } catch (\Stripe\Exception\ApiErrorException $e) {
            $logResponse = (object)['Error' => 'Stripe API: Generic API failure. ' . $e->getMessage()];
        } catch (\Exception $e) {
            $logResponse = (object)['Error' => $e->getMessage()];
        }

        $hideFields = [
            'payment_method'
        ];
        foreach ($hideFields as $field) {
            if (isset($logResponse->$field)) $logResponse->field = 'hidden_for_security';
            if (isset($logData[$field])) $logData[$field] = 'hidden_for_security';
        }

        $hideIdsForApis = [
            'paymentMethods'
        ];
        foreach ($hideIdsForApis as $method) {
            if ($apiName == $method) {
                $id = 'hidden_for_security';
                if (isset($logResponse->id)) $logResponse = ['hidden_for_security'];
                if (isset($logData['id'])) $logData['id'] = 'hidden_for_security';
            }
        }

        $paymentLog = new paymentLogEntity();
        $paymentLog->ts = date('Y-m-d H:i:s');
        $paymentLog->provider = 'stripe';
        $paymentLog->api_name = $apiName;
        $paymentLog->method_name = $methodName;
        if ($id) $paymentLog->provider_id = $id;
        elseif (isset($logResponse->id)) $paymentLog->provider_id = $logResponse->id;
        $paymentLog->request_data = json_encode($logData);
        $paymentLog->response_data = json_encode($logResponse);
        $paymentLog->store();

        return $response ?? $logResponse;
    }


}