<?php

namespace apexl\Io\modules\payment\services;

use apexl\Config\Singleton;
use apexl\encryption\Encrypt;
use apexl\Io\includes\Hook;
use apexl\Io\includes\Routes;
use apexl\Io\modules\company\entities\companyEntity;
use apexl\Io\modules\payment\entities\paymentEntity;
use apexl\Io\modules\payment\entities\paymentProviderRefEntity;
use apexl\Io\modules\payment\entities\paymentTokenEntity;
use apexl\Io\modules\payment\providers\stripeProvider;
use apexl\Io\modules\product\entities\operators\productRemoteReferenceIdsOperator;
use apexl\Io\modules\product\entities\productEntity;
use apexl\Io\modules\product\entities\productRemoteReferenceIdsEntity;
use apexl\Io\modules\subscription\entities\subscriptionEntity;
use apexl\Io\modules\user\entities\userEntity;
use apexl\Io\modules\user\services\simService;
use apexl\Io\modules\user\services\userTools;
use apexl\Io\services\Database;
use apexl\Io\services\Mailer;
use apexl\Io\services\Output;
use apexl\Io\services\pathUtility;
use Psr\Http\Message\ServerRequestInterface as Request;

class cardValidationService {

    protected $config;
    protected $provider;
    protected $providerName;
    protected $mailer;
    protected $output;
    protected $path;
    protected $subscriptionId;
    protected $maxPaymentFailures = 3;  // If payment fails this many times, card will be abandoned and user will need to re-enter details
    protected $subscriptions = [];
    protected $chargeId = null;

    public function __construct()
    {
        $this->config = Singleton::getInstance();
        $this->provider = new stripeProvider();
        $this->providerName = 'stripe';
        $this->output = Output::getInstance();
        $this->path = pathUtility::getInstance();
    }

    public function createNewTokenEntity($userId, $data)
    {
        $paymentTokenEntity = new paymentTokenEntity();
        $paymentTokenEntity->user_id = $userId;
        $paymentTokenEntity->created = time();
        $paymentTokenEntity->provider = $this->getProviderName();
        $paymentTokenEntity->data = json_encode($data);
        $paymentTokenEntity->store();

        return $paymentTokenEntity;
    }

    public function getOneTimePaymentUrl($products, $successUrl, $cancelUrl)
    {
        if (!$successUrl) $successUrl = $this->config->app->site->backend_domain . Routes::getRoutePattern('payment.validation.success');
        if (!$cancelUrl) $cancelUrl = $this->config->app->site->backend_domain . Routes::getRoutePattern('payment.validation.cancel');

        return $this->provider->getOneTimePaymentUrl($products, $successUrl, $cancelUrl);
    }

    public function getPaymentIntentIdFromSessionId($sessionId)
    {
        return $this->provider->getPaymentIntentIdFromSessionId($sessionId);
    }

    public function getProviderPortalUrl($paymentToken, $successUrl=null, $cancelUrl=null, $meta=null)
    {
        if (!$successUrl) $successUrl = $this->config->app->site->backend_domain . Routes::getRoutePattern('payment.validation.success');
        if (!$cancelUrl) $cancelUrl = $this->config->app->site->backend_domain . Routes::getRoutePattern('payment.validation.cancel');
        $providerCustomerRef = $paymentToken->user_id ? $this->getProviderUserRef($paymentToken->user_id) : null;

        return $this->provider->getCardValidationUrl($successUrl, $cancelUrl, $paymentToken->id, $providerCustomerRef, $meta);
    }

    public function getProviderTokenUsingSessionId($sessionId)
    {
        return $this->provider->retrieveToken($sessionId);
    }

    public function comleteValidation(Request $request, $paymentToken, $userEmail, $userFullName, $password=null, $createSubscription=true, $startDate=null, $gracePeriodEnds=null, $nextBillingDate=null)
    {
        $this->subscriptions = [];
        $user = new userEntity();
        $user->getUserByEmail($userEmail);
        if (!$user->id){
            // Create account
            $userTools = new userTools();
            list($firstName, $lastName) = $userTools->splitNameToFirstAndLast($userFullName);
            $user = $userTools->createCompanyUserWithPassword($request, $userEmail, $firstName, $lastName, $password, "$firstName $lastName");
        }

        if ($user) {
            $encrypt = new Encrypt();
            $paymentMethodRefDecrypted = $encrypt->decrypt($paymentToken->payment_method_ref, $paymentToken->payment_method_ref_iv, $this->config->app->encryption->key);
            if ($customerId = $this->attachCustomerToPaymentMethod($paymentMethodRefDecrypted, $user)) {
                //$paymentToken->company_id = $company->id;
                $paymentToken->user_id = $user->id;
                $paymentToken->setup_complete = 1; // Mark as setup complete
                $paymentToken->store();
                if ($createSubscription) {
                    $companyId = null;
                    if (class_exists('apexl\Io\modules\company\entities\companyEntity')) {
                    $company = new companyEntity();
                    $company->loadByUser($user->id);
                        $companyId = $company ? $company->id : null;
                    }
                    $data = json_decode($paymentToken->data);
                    if (isset($data->products) && is_array($data->products)) { //i.e. if this is a woocommerce order
                        $productRemoteReferenceIdsEntity = new productRemoteReferenceIdsEntity();
                        $productEntity = new productEntity();
                        foreach ($data->products as $product) {
                            if (isset($product->product_id)) {
                            $productId = $productRemoteReferenceIdsEntity->getProductIdUsingRemoteReferenceId($product->product_id); // Get Dashboard product ID from Woo Product ID
                            } elseif (isset($product->product_name)) { // If no ID provided, load by name
                                $productEntity->loadByName($product->product_name);
                                if (!isset($productEntity->id) || !$productEntity) { // Automatically create new product if it doesn't exist
                                    $productEntity = new productEntity();
                                    $productEntity->type = $product->type ?? "";
                                    $productEntity->product_name = $product->product_name;
                                    $productEntity->price = $product->price ?? 0;
                                    $productEntity->code = $product->code ?? "";
                                    $productEntity->created = time();
                                    $productEntity->store();
                                }
                                $productId = $productEntity->id;
                            }
                            $this->createSubscription($companyId, $user->id, $productId, $product->quantity, $startDate, $gracePeriodEnds, $nextBillingDate);
                        }
                    } else { // None-woocommerce order - may be extinct now? Not sure...
                        $this->createSubscription($companyId, $user->id, $this->getProductIdFromToken($paymentToken), 1, $startDate, $gracePeriodEnds, $nextBillingDate);
                    }
                }

                $this->output::addMessage($this->path->getRouteName($request), 'success','Payment setup complete');
            }
        }

        return $user;
    }

    public function reenableUserSubscriptions($user)
    {
        $subscriptionEntity = new subscriptionEntity();
        $subscriptions = $subscriptionEntity->getByUser($user->id);
        $now = date('Y-m-d');
        foreach ($subscriptions as $subscription) {
            $subscriptionEntity->load($subscription->id);
            if (!$subscriptionEntity->enabled && $subscriptionEntity->suspended) {
                if (!$subscriptionEntity->date_expires || $subscriptionEntity->date_expires > $now) {
                    $subscriptionEntity->enabled = true;
                    $subscriptionEntity->suspended = false;
                    $subscriptionEntity->store();
                }
            }
        }
    }

    public function getSubscriptions()
    {
        return $this->subscriptions;
    }

    public function getChargeId()
    {
        return $this->chargeId;
    }

    /*public function getSubscriptionId()
    {
        return $this->subscriptionId;
    }

    public function getSubscription()
    {
        return $this->subscription;
    }*/

    public function getProviderName()
    {
        return $this->providerName;
    }


    public function capturePaymentWithToken($userId, $amount, $ref="", $isSubscriptionPayment=true, $data=null, $meta=null)
    {
        $this->chargeId = null;
        $paymentTokenEntity = new paymentTokenEntity();
        $latestToken = $paymentTokenEntity->getLastForUser($userId); // Get last payment token for user
        $description = '';

        if (isset($latestToken['payment_method_ref'])
            && isset($latestToken['setup_complete']) && $latestToken['setup_complete']
            && (!isset($latestToken['abandoned']) || !$latestToken['abandoned'])) {

            $encrypt = new Encrypt();
            $paymentMethodRefDecrypted = $encrypt->decrypt($latestToken['payment_method_ref'], $latestToken['payment_method_ref_iv'], $this->config->app->encryption->key);
            if ($paymentMethodRefDecrypted) {
                $paymentProviderRefEntity = new paymentProviderRefEntity();
                $customerRef = $paymentProviderRefEntity->getUserRef($this->providerName, $userId);
                if (!is_null($data)) {
                    $description = json_encode($data);
                }
                if ($isSubscriptionPayment) {
                    if (!isset($meta->subscription_ids)) $meta['subscription_ids'] = [];
                    foreach ($this->subscriptions as $subscription) {
                        $meta['subscription_ids'][] = $subscription->id;
                    }
                    $meta['subscription_ids'] = implode(",", $meta['subscription_ids']);
                }
                list($paymentId, $chargeId) = $this->provider->capturePaymentWithToken($paymentMethodRefDecrypted, $customerRef['ref'], $amount, $description, 'gbp', $meta);
                $this->chargeId = $chargeId;
                //$paymentId = false;
                $payment = new paymentEntity();
                $payment->ref = $ref;
                $payment->provider = $this->providerName;
                $payment->provider_ref = $paymentId;
                $payment->status = $paymentId ? 'successful' : 'failed';
                $payment->amount = $amount;
                if (!is_null($data)) {
                    $payment->data = json_encode($data);
                    if (isset($data->order_id)) $payment->remote_order_id = $data->order_id;
                }

                $payment->store();

                $paymentTokenEntity->load($latestToken['id']);

                if (!$paymentId) {
                    $paymentTokenEntity->num_failures++;
                    $paymentTokenEntity->store();
                    if ($paymentTokenEntity->num_failures >= $this->maxPaymentFailures) {
                        $paymentTokenEntity->abandoned = true;
                        $paymentTokenEntity->store();
                        if ($isSubscriptionPayment) $this->processSubscriptionPaymentFailure();
                    }
                } else {
                    $paymentTokenEntity->num_failures = 0; // reset number of failures
                    $paymentTokenEntity->store();
                    if ($isSubscriptionPayment) $this->processSubscriptionPaymentSuccess();

                    // Temporary if statement to get quick results for ED - code shouldn't really be here
                    if (!is_null($data) && isset($data->order_id)) {
                        $woocommerceService = new woocommerceIntegrationService();
                        $woocommerceService->notifyOrderSuccess($data->order_id, $chargeId);
                    }
                }

                if ($isSubscriptionPayment) $this->createSubscriptionPaymentLinks($payment);

                return $payment;
            }
        }

        return null;
    }

    public function setSubscriptions(array $subscriptions)
    {
        $entities = [];
        foreach ($subscriptions as $subscription) {
            if ($subscription instanceof subscriptionEntity) {
                $entities[] = $subscription;
            } else {
                $subscriptionEntity = new subscriptionEntity();
                $subscriptionEntity->load($subscription->id);
                $entities[] = $subscriptionEntity;
            }
        }
        $this->subscriptions = $entities;
    }

    public function createSubscriptionPaymentLinks($payment)
    {
        $upserts = [];
        foreach ($this->subscriptions as $subscription) {
            $upserts[] = [
                'subscription_id' => $subscription->id,
                'payment_id' => $payment->id,
            ];
        }
        if (count($upserts) > 0) {
            Database::persistRawBatch($upserts, 'subscription_payment');
        }
    }

    public function processSubscriptionPaymentFailure()
    {
        foreach ($this->subscriptions as $subscription) {
            $subscription->suspended = 1;
            $subscription->enabled = 0;
            $subscription->store();
            $this->paymentMethodFailureNotification($subscription);
            //Hook::processHook('paymentMethodFailed', $subscription);
        }
    }

    public function processSubscriptionPaymentSuccess()
    {
        foreach ($this->subscriptions as $subscription) {
            $date = new \DateTime('now'); // Set last and next billing dates for subscription
            $subscription->last_billing_date = $date->format('Y-m-d');
            if (isset($subscription->next_billing_date) && $subscription->next_billing_date) {
                $date = \DateTime::createFromFormat('Y-m-d', $subscription->next_billing_date);
            }

            // Set next billing date
            if (isset($this->config->app->payments->subscriptions->billingInterval) && $this->config->app->payments->subscriptions->billingInterval) {
                $date->add(new \DateInterval($this->config->app->payments->subscriptions->billingInterval));
            } elseif (isset($this->config->app->payments->subscriptions->specificBillingDate) && $this->config->app->payments->subscriptions->specificBillingDate) {
                $date->modify($this->config->app->payments->subscriptions->specificBillingDate);
            } else {
            $date->add(new \DateInterval('P1M'));
            }
            $subscription->next_billing_date = $date->format('Y-m-d');
            $subscription->enabled = 1;
            $subscription->store();
            //Hook::processHook('paymentMethodSuccess', $subscription);
        }
    }

    protected function paymentMethodFailureNotification(subscriptionEntity $subscription)
    {
        $from = $this->config->app->site->email_address ?? 'no-reply@localhost.com';
        $fromName = $this->config->app->site->name ?? 'localhost';
        $backEndDomain = $this->config->app->site->backend_domain ?? 'localhost';
        $link = rtrim($backEndDomain, '/').'/api/v1/payment/action/payment/validation/trigger/-/'.$subscription->code;

        $user = new userEntity();
        $user->load($subscription->user_id);

        if ($user->id) {
            $mailer = new Mailer();
            $mailer->setFrom($from, $fromName);
            $mailer->addAddress($user->email);     // Add a recipient

            $mailer->Subject = 'Payment Failure Notification';
            $mailer->Body = 'We are writing to let you know that your chosen payment method has failed. In order to avoid disruption to your service, please click the link below and follow the on screen instructions to update your payment details. <br><br><a href="' . $link . '">Update Payment Details</a>';
            $mailer->IsHTML(true);
            $result = $mailer->send();
        }
    }

    /*protected function setSubscriptionId($subscriptionId)
    {
        $this->subscriptionId = $subscriptionId;
    }

    protected function setSubscription($subscription)
    {
        $this->subscription = $subscription;
    }*/

    protected function createSubscription($companyId, $userId, $productId, $productQuantity, $startDate=null, $gracePeriodEnds=null, $nextBillingDate=null)
    {
        $subscription = new subscriptionEntity();

        // Look for existing subscriptions for this user, if exists, update to trigger hooks, and use that
        if (isset($this->config->app->payments->subscriptions->reuseExistingSubscriptions) && $this->config->app->payments->subscriptions->reuseExistingSubscriptions) {
        $sub = $subscription->getByUserCompanyProductAndDate($userId, $companyId, $productId, date('Y-m-d'));
        }
        if (isset($sub) && isset($sub['id']) && $sub['id']) {
            $subscription->load($sub['id']);
            $subscription->product_quantity = $subscription->product_quantity + $productQuantity;
        } else {
            $subscription->company_id = $companyId;
            $subscription->user_id = $userId;
            $now = date('Y-m-d h:i:s');
            $subscription->date_starts = date('Y-m-d');
            $subscription->enabled = 1;
            $subscription->created = $now;
            $subscription->product_id = $productId;
            $subscription->product_quantity = $productQuantity;
            $subscription->code = bin2hex(random_bytes(32)); // unique code to use in email links

            if ($nextBillingDate) {
                $date = \DateTime::createFromFormat('Y-m-d', $nextBillingDate);
            }  else {
            if (is_null($startDate)) {
                $date = new \DateTime('now');
            } else {
                $date = $startDate;
            }
            }
            $subscription->next_billing_date = $date->format('Y-m-d');

            if ($gracePeriodEnds) $subscription->grace_period_ends = $gracePeriodEnds->format('Y-m-d');
        }

        $subscription->store(); // Always store (even for existing subs), in order to trigger hooks

        //$this->setSubscription($subscription);
        //$this->setSubscriptionId($subscription->id);
        $this->subscriptions[$subscription->id] = $subscription;

        return $subscription;
    }

    protected function getProductIdFromToken($paymentToken)
    {
        $data = json_decode($paymentToken->data);
        return $data->productId ?? null;
    }

    public function attachCustomerToPaymentMethod($token, $user, $forceAttachInStripe=FALSE)
    {
        if ($ref = $this->getProviderUserRef($user->id) && !$forceAttachInStripe) return $ref;

        $customerId = $this->provider->attachCustomerToPaymentMethod($token, $user->email, $user->getNiceName, $ref);
        if ($customerId && !$ref) {
            $paymentProviderRef = new paymentProviderRefEntity();
            $paymentProviderRef->user_id = $user->id;
            $paymentProviderRef->provider = $this->providerName;
            $paymentProviderRef->ref = $customerId;
            $paymentProviderRef->store();
        }

        return $customerId;
    }

    protected function getProviderUserRef($userId)
    {
        $paymentProviderRef = new paymentProviderRefEntity();
        $paymentProviderRef->getUserRef($this->providerName, $userId);
        return $paymentProviderRef->ref ?? null;
    }
}