<?php declare(strict_types=1);

namespace apexl\Io\modules\queue\entities;

use apexl\entityCore\enums\Casts;
use apexl\entityCore\traits\hasCasts;
use apexl\Io\includes\Entity;
use apexl\Io\includes\System;
use apexl\Io\modules\queue\dto\queueItemDto;
use apexl\Io\modules\queue\exceptions\entityNotSavedException;
use apexl\Io\modules\notifier\exceptions\notifierException;
use apexl\Io\modules\queue\helpers\Condition;
use apexl\Io\modules\queue\interfaces\queueItemReadyNotifierInterface;
use apexl\Io\modules\queue\operators\queueItemOperator;
use apexl\Io\modules\queue\dto\queueSettings\queueSettingsQueueDto;
use apexl\Utils\Strings\Phone;
use DateTimeImmutable;
use DateTimeInterface;
use Exception;
use JsonSerializable;
use Psr\Http\Message\ServerRequestInterface;
use Ramsey\Uuid\Uuid;

/**
 * @property int $position
 * @property string $uuid
 * @property string $phone
 * @property string $email
 * @property DateTimeImmutable $fulfilled_at
 * @property DateTimeImmutable $skipped_at
 * @property DateTimeImmutable $notified_at
 * @property int $created
 * @property int $modified
 *
 * @mixin queueItemOperator
 */
class queueItem extends Entity implements JsonSerializable
{
    use hasCasts;

    public const TABLE = 'queue_items';
    private const KEY = 'position';

    protected array $show = [
        'position',
        'uuid',
    ];

    protected array $hide = [];

    public function __construct()
    {
        parent::__construct(self::TABLE, self::KEY);

        $this->setOperator(new queueItemOperator(self::TABLE, self::KEY));
    }

    public function getNiceName(): string
    {
        return 'Queue Items';
    }

    /**
     * @return queueItem[]
     */
    public static function all(array $conditions = []): array
    {
        $entity = new queueItem();

        try {
            return $entity->loadMultiple($conditions);
        } catch (Exception $e) {
            user_error(
                sprintf('Error loading queue items: %s', $e->getMessage()),
                E_USER_WARNING
            );

            return [];
        }
    }

    public static function allActive(): array
    {
        $settings = queueSettingsQueueDto::fromDb(true);

        $limit = null;
        if ($settings->display_upcoming) {
            $limit = $settings->active_count + $settings->display_upcoming;
        }

        return self::queryActive($limit);
    }

    /**
     * @return queueItem[]
     */
    public static function latestActiveNotNotified(): array
    {
        $settings = queueSettingsQueueDto::fromDb(true);

        return self::queryActive($settings->active_count, [Condition::isNull('notified_at')]);
    }

    private static function queryActive(?int $limit = null, array $conditions = []): array
    {
        try {
            $conditions = array_merge([
                Condition::isNull('fulfilled_at'),
                Condition::isNull('skipped_at'),
            ], $conditions);

            $entity = new queueItem();

            return $entity->loadMultiple(
                $conditions,
                [['created', 'ASC']],
                $limit ?? false,
                0
            );
        } catch (Exception $e) {
            logger()->warning(sprintf('Error loading queue items: %s', $e->getMessage()));

            return [];
        }
    }

    function casts(): array
    {
        return [
            'position' => Casts::INT(),
            'uuid' => Casts::STRING(),
            'phone' => Casts::STRING(),
            'email' => Casts::STRING(),
            'skipped_at' => Casts::DATETIME(),
            'fulfilled_at' => Casts::DATETIME(),
            'notified_at' => Casts::DATETIME(),
            'created' => Casts::INT(),
            'modified' => Casts::INT(),
        ];
    }

    public function markFulfilled(?DateTimeInterface $fulfilledAt = null)
    {
        $this->fulfilled_at = $fulfilledAt ?? new DateTimeImmutable();

        try {
            $this->store();
        } catch (Exception $e) {
            logger('error')
                ->error(sprintf('Error marking queueItem as fulfilled: %s', $e->getMessage()));
        }
    }

    public function markSkipped(?DateTimeInterface $skippedAt = null)
    {
        $this->skipped_at = $skippedAt ?? new DateTimeImmutable();

        try {
            $this->store();
        } catch (Exception $e) {
            logger()->error(sprintf('Error marking queueItem as skipped: %s', $e->getMessage()));
            http_response_code(500);

            exit;
        }
    }

    public function sendNotification(bool $force = false): void
    {
        if ($this->isNotified() && !$force) {
            logger()->debug(
                sprintf('queueItem %s has already sent notification; skipping.', $this->uuid)
            );

            return;
        }

        try {
            $this->notified_at = new DateTimeImmutable();
            $this->store();

            /** @var queueItemReadyNotifierInterface $notifier */
            $notifier = System::makeRegisteredService(queueItemReadyNotifierInterface::class, [
                'queueItem' => $this,
            ]);
            $notifier->send();
        } catch (Exception|notifierException $e) {
            logger('notifier')
                ->warning(sprintf('Error sending notification: %s', $e->getMessage()));
        }
    }

    /**
     * @throws entityNotSavedException
     */
    private function assertSaved(): void
    {
        if (!$this->{$this->primaryKey}) {
            throw new entityNotSavedException('Entity not saved!');
        }
    }

    public static function createFromRequest(ServerRequestInterface $request): queueItem
    {
        try {
            $queueItemDto = queueItemDto::fromRequest($request);
            $queueItem = new queueItem();
            $queueItem->phone = $queueItemDto->phone;
            $queueItem->email = $queueItemDto->email;
            $queueItem->store();

            return $queueItem;
        } catch (Exception $e) {
            logger()->error(sprintf('Error storing queueItem: %s', $e->getMessage()));
            http_response_code(500);

            exit;
        }
    }

    public function setStandardFields()
    {
        parent::setStandardFields();

        if (!isset($this->data['uuid'])) {
            $this->data['uuid'] = $this->uniqueUuid();
        }

        if (isset($this->data['phone'])) {
            $this->data['phone'] = Phone::normalise($this->data['phone']);
        }
    }

    private function uniqueUuid(): string
    {
        do {
            $uuid = Uuid::uuid4()->toString();
        } while ($this->uuidExists($uuid));

        return $uuid;
    }

    public static function fromUuid(string $uuid): ?queueItem
    {
        $entity = new self();
        try {
            if($queueItems = $entity->loadMultiple([['uuid', $uuid]])) {
                return array_pop($queueItems);
            }
        } catch (Exception $e) {
            logger()->warning(sprintf('Error loading queueItem: %s', $e->getMessage()));
        }

        return null;
    }

    public function isFulfilled(): bool
    {
        return (bool) $this->fulfilled_at;
    }

    public function isSkipped(): bool
    {
        return (bool) $this->skipped_at;
    }

    public function isNotified(): bool
    {
        return (bool) $this->notified_at;
    }

    public function jsonSerialize(): array
    {
        $fields = array_keys($this->data);

        if ($this->show) {
            $fields = array_filter(
                $fields,
                fn($key) => in_array($key, $this->show),
            );
        }

        if ($this->hide) {
            $fields = array_filter(
                $fields,
                fn($key) => !in_array($key, $this->hide),
            );
        }

        $data = [];
        foreach ($fields as $field) {
            $data[$field] = $this->{$field};
        }

        $data['ready'] = $this->isReady();

        return $data;
    }

    public function hrName($fieldName): string
    {
        switch ($fieldName) {
            case 'phone':
                return 'Phone';
            case 'fulfilled_at':
                return 'Fulfilled at';
            case 'skipped_at':
                return 'Skipped at';
            default:
                return parent::hrName($fieldName);
        }
    }

    public function getFieldConfig(): array
    {
        return $this->withCasts([
            'uuid' => [
                'formField' => fn() => null,
            ]
        ]);
    }

    private function isReady(): bool
    {
        $settings = queueSettingsQueueDto::fromDb(true);

        try {
            return $this->created < time() - ($settings->waiting_period * 60);
        } catch (Exception $e) {
            logger()->warning(
                sprintf('Unable to check if queueItem is ready: %s', $e->getMessage())
            );

            return false;
        }
    }
}
