<?php declare(strict_types=1);

namespace apexl\Io\modules\queue;

use apexl\Io\includes\System;
use apexl\Io\modules\basicApi\basicApiModule;
use apexl\Io\modules\notifier\notifiers\multipleNotifier;
use apexl\Io\modules\queue\controllers\fulfillQueueController;
use apexl\Io\modules\queue\controllers\joinQueueController;
use apexl\Io\modules\queue\controllers\queueCronController;
use apexl\Io\modules\queue\controllers\queueItemsController;
use apexl\Io\modules\queue\controllers\queueSettingsController;
use apexl\Io\modules\queue\controllers\queueStatsController;
use apexl\Io\modules\queue\controllers\skipQueueController;
use apexl\Io\modules\queue\entities\queueItem;
use apexl\Io\modules\queue\interfaces\queueItemReadyNotifierInterface;
use apexl\Io\modules\queue\listeners\getEntityDataListener;
use apexl\Io\modules\queue\listeners\postEntityUpdateListener;
use apexl\Io\modules\queue\notifiers\queueItemReadyEmailNotifier;
use apexl\Io\modules\queue\notifiers\queueItemReadySmsNotifier;
use apexl\Io\modules\user\services\Permissions;
use stdClass;

class queueModule extends basicApiModule
{
    public const PERMISSION__MANAGE_QUEUE = 'manage-queue';

    public function __construct()
    {
        parent::__construct(new queueItem());

        $this->registerHooks();
    }

    public function schema(): stdClass
    {
        return System::loadSchema(__DIR__ . '/schema/schema.json') ?: new stdClass();
    }

    public function addServices(): array
    {
        return [
            queueItemReadyNotifierInterface::class =>
                fn(queueItem $queueItem) =>
                new multipleNotifier([
                    System::makeRegisteredService(
                        queueItemReadyEmailNotifier::class,
                        ['queueItem' => $queueItem]
                    ),
                    System::makeRegisteredService(
                        queueItemReadySmsNotifier::class,
                        ['queueItem' => $queueItem]
                    ),
                ])
        ];
    }

    public function routes(): void
    {
        parent::routes();

        $this->route->post(
            'queue.join',
            '/queue/join',
            [joinQueueController::class, 'join']
        );

        $this->route->post(
            'queue.fulfill',
            '/queue/fulfill/{uuid}',
            [fulfillQueueController::class, 'fulfill']
        );

        $this->route->delete(
            'queue.skip',
            '/queue/skip/{uuid}',
            [skipQueueController::class, 'skip']
        );

        $this->route->get(
            'queue.active',
            '/queue/active',
            [queueItemsController::class, 'active']
        );

        $this->route->post(
            'queue.join',
            '/queue/join',
            [joinQueueController::class, 'join']
        );

        $this->addProtectedActionRoute(
            'post',
            'queue.add',
            'queue/add',
            [joinQueueController::class, 'add']
        );

        $this->addProtectedActionRoute(
            'post',
            'queue.settings.update',
            'settings',
            [queueSettingsController::class, 'update']
        );

        $this->addProtectedActionRoute(
            'post',
            'queue.settings.upload.video',
            'settings/upload/video',
            [queueSettingsController::class, 'upload']
        );

        $this->route->get(
            'queue.cron',
            'queue/cron',
            [queueCronController::class, 'process']
        );

        $this->route->get(
            'queue.stats.count',
            'queue/stats/count',
            [queueStatsController::class, 'count']
        );
    }

    public function registerPermissions(Permissions $permissions)
    {
        parent::registerPermissions($permissions);

        $permissions->registerPermission(
            self::PERMISSION__MANAGE_QUEUE,
            'Manage queue',
            ucfirst($this->getName())
        );
    }

    private function registerHooks()
    {
        postEntityUpdateListener::register();
        getEntityDataListener::register();
    }
}
