<?php

namespace apexl\Io\modules\subscription\controllers;

use apexl\Config\Singleton;
use apexl\Io\includes\Controller;
use apexl\Io\includes\Hook;
use apexl\Io\includes\System;
use apexl\Io\modules\product\entities\productEntity;
use apexl\Io\modules\subscription\entities\subscriptionCreditEntity;
use apexl\Io\modules\subscription\entities\subscriptionEntity;
use apexl\Io\modules\user\entities\userEntity;
use apexl\Io\modules\user\services\currentUser;
use apexl\Io\services\pathUtility;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;

class subscriptionController extends Controller {

    public function __construct(pathUtility $path, Singleton $config)
    {
        parent::__construct();
        $this->path = $path;
        $this->config = $config;
    }

    public function subscriptionListTableData(Request $request, Response $response){
        $params = $request->getQueryParams();
        $entity = new subscriptionEntity();
        $filters = $this->buildFilterConditions($entity, $params);
        $entityData = $entity->loadSubscriptionListByPage($params, $filters, $params['orderBy'] ?? []);

        $entityData['tableHeader'] = ['ID', 'User', 'Product', 'Qty', 'Amount', 'Created', 'Next Billing Date', 'Last Billed', 'State', 'SIMs', 'Transaction Status', 'Order ID', 'Order Status'];

        $rows = [];
        foreach($entityData['data'] as $result){
            $rowSub = new subscriptionEntity();
            $rowSub->load($result->id);
            $subData = $rowSub->getData();
            $orderData = $rowSub->getWordPressOrderStatus($result->remote_order_id);
            $row = [
                'id' => $subData['id'],
                'user' => $subData['user_id'],
                'product' => $subData['product_id'],
                'qty' => $subData['product_quantity'],
                'amount' => '£'.$result->amount,
                'created' => $subData['created'],
                'nbd' => $subData['next_billing_date'],
                'lb' => $subData['last_billing_date'],
                'state' => $subData['enabled'],
                'sims' => $result->assigned_sims,
                'transaction' => $result->status,
                'order' => $result->remote_order_id,
                'order_status' => isset($orderData['status']) ? "<div class='woostatus ".$orderData['status']."'>".$orderData['status'].'</div>' : ''
            ];
            $rows[] = $row;
        }

        $entityData['rows'] = $rows;
        unset($entityData['data']);

        return System::asJson($response, $entityData);
    }

    public function subscriptionListTableDataBasic(Request $request, Response $response){
        $params = $request->getQueryParams();
        $entity = new subscriptionEntity();

        $currentUser = currentUser::getCurrentUser();
        $email = $params['userEmail'] ?? $currentUser->email;
        $userEntity = new userEntity();
        $userEntity->getUserByEmail($email);
        $params['user_id'] = $userEntity->id ?? 0;
        $filters = $this->buildFilterConditions($entity, $params);

        $subscriptions = $entity->loadMultiple($filters, $params['orderBy'] ?? []); // Note - this inclueds access control with correct config

        $entityData['tableHeader'] = ['#', 'Created', 'Next Billing Date', 'Product', 'Quantity', 'Type', 'Amount', 'Status'];

        $rows = [];

        $productEntity = new productEntity();
        $products = $productEntity->getAll();
        $priceCache = [];
        foreach ($products as $product) {
            $priceCache[$product->product_name] = $product->price;
        }

        foreach($subscriptions as $subscription){

            $subData = $subscription->getData();

            $created = \DateTime::createFromFormat('Y-m-d', $subscription->date_starts);
            $nextBillingDate = \DateTime::createFromFormat('Y-m-d', $subscription->next_billing_date);
            $status = '';
            if ($subscription->enabled) {
                $status = 'enabled';
            }
            if ($subscription->suspended) {
                $status = 'suspended';
            }
            if (!$subscription->enabled) {
                $status = 'expired';
            }

            $row = [
                'id' => $subscription->id,
                'created' => $created->format('d M Y'),
                'next_billing_date' => $nextBillingDate->format('d M Y'),
                'product' => $subData['product_id'],
                'qty' => $subData['product_quantity'],
                'type' => 'CC',
                'amount' => isset($priceCache[$subData['product_id']]) ? '£'.$priceCache[$subData['product_id']] : 'N/A',
                'status' => $status,
            ];
            $rows[] = $row;
        }

        $entityData['rows'] = $rows;
        $entityData['totalData'] = count($rows);
        unset($entityData['data']);

        $entityData = Hook::processHook('subscriptionListBasicAlter', $entityData);

        return System::asJson($response, $entityData);
    }


    public function updateSubscription(Request $request, Response $response)
    {
        $body = $request->getParsedBody();
        if(!isset($body->subscription_id) || !$body->subscription_id){
            $this->output::addMessage('subscription.update', 'error', 'Subscription ID missing');
            $this->output::addResponse($request, [], FALSE); // added so we can hook into this elsewhere.
            return System::asJson($response, [], 400);
        }

        $subscription = new subscriptionEntity();
        $subscription->load($body->subscription_id);
        if (!$subscription->id) {
            $this->output::addMessage('subscription.update', 'error', 'Subscription does not exist');
            $this->output::addResponse($request, [], FALSE); // added so we can hook into this elsewhere.
            return System::asJson($response, [], 400);
        }

        if (isset($body->product_id) && $body->product_id > 0) $subscription->product_id = $body->product_id;
        if (isset($body->product_quantity) && $body->product_quantity > 0) $subscription->product_quantity = $body->product_quantity;
        $subscription->enabled = $body->enabled ? TRUE : FALSE;
        if ($subscription->enabled) {
            if (isset($this->config->app->payments->subscriptions->autoDeSuspendOnEnable) && $this->config->app->payments->subscriptions->autoDeSuspendOnEnable) {
                $subscription->suspended = FALSE;
            }
        }
        $subscription->next_billing_date = $body->next_billing_date;
        $subscription->store();
        $this->output::addMessage('subscription.update', 'success', 'Subscription Successfully updated!');

        return System::asJson($response);
    }

    public function remoteUpdate(Request $request, Response $response, $args)
    {
        $body = $request->getParsedBody();
        if (!isset($body->subscriptionId) || !$body->subscriptionId || !isset($body->remoteKey) || !isset($this->config->app->remoteUpdates->key) || $body->remoteKey != $this->config->app->remoteUpdates->key) {
            $this->output::addMessage('subscription.remote.disable', 'error', 'Required information is missing - requirement not stated explicitly for security reasons');
            $this->output::addResponse($request, [], FALSE); // added so we can hook into this elsewhere.
            return System::asJson($response, [], 400);
        }

        $subscriptionEntity = new subscriptionEntity();
        $subscriptionEntity->load((int)$body->subscriptionId);
        if (!isset($subscriptionEntity->id) || !$subscriptionEntity->id) {
            $this->output::addMessage('subscription.remote.disable', 'error', 'Subscription not found');
            $this->output::addResponse($request, [], FALSE); // added so we can hook into this elsewhere.
            return System::asJson($response, [], 400);
        }

        $responseData = [];

        if (isset($body->enabled) && is_bool($body->enabled)) $subscriptionEntity->enabled = $body->enabled;
        if (isset($body->product_price_override)) $subscriptionEntity->product_price_override = $body->product_price_override;
        if (isset($body->product_price_override_ends)) $subscriptionEntity->product_price_override_ends = $body->product_price_override_ends;
        if (isset($body->credits) && is_array($body->credits)) {
            foreach ($body->credits as $credit) {
                $creditEntity = new subscriptionCreditEntity();
                $creditEntity->credit_amount = $credit->amount;
                $creditEntity->subscription_id = (int)$body->subscriptionId;
                $creditEntity->credit_date = $credit->date;
                if (isset($credit->data)) $creditEntity->data = json_encode($credit->data);
                $creditEntity->store();
            }
        }
        if (isset($body->product_name) && $body->product_name != ""){
            $responseData['product_name'] = $body->product_name;
            $productEntity = new productEntity();
            $productEntity->loadByName($body->product_name);
            if (!isset($productEntity->id) || !$productEntity) { // Automatically create new product if it doesn't exist
                $productEntity = new productEntity();
                $productEntity->product_name = $body->product_name;
                $productEntity->price = $body->product_price;
                $productEntity->type = $product->product_type ?? "";
                $productEntity->code = $product->product_code ?? "";
                $productEntity->created = time();
                $productEntity->store();
                $responseData['new_product'] = 'yes';
            }
            $subscriptionEntity->product_id = $productEntity->id;
        };

        $subscriptionEntity->store();

        $responseData['enabled'] = $subscriptionEntity->enabled;

        return System::asJson($response, ['subscriptionsUpdated' => 1, 'data' => $responseData]);
    }
}