<?php

namespace apexl\Io\modules\subscription\services;

use apexl\Config\Singleton;
use apexl\Io\includes\Routes;
use apexl\Io\includes\System;
use apexl\Io\modules\company\entities\companyEntity;
use apexl\Io\modules\payment\entities\invoiceEntity;
use apexl\Io\modules\payment\entities\paymentEntity;
use apexl\Io\modules\payment\entities\userCreditEntity;
use apexl\Io\modules\payment\providers\stripeProvider;
use apexl\Io\modules\payment\services\cardValidationService;
use apexl\Io\modules\payment\services\paymentService;
use apexl\Io\modules\product\entities\productEntity;
use apexl\Io\modules\product\entities\productRemoteReferenceIdsEntity;
use apexl\Io\modules\subscription\entities\subscriptionCreditEntity;
use apexl\Io\modules\subscription\entities\subscriptionEntity;
use apexl\Io\modules\user\entities\userEntity;
use apexl\Io\services\Database;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;

class subscriptionService {

    protected $config;

    public function __construct()
    {
        $this->config = Singleton::getInstance();
    }

    public function rollForwardSubscriptions($subscriptions)
    {
        foreach ($subscriptions as $subscription) {
            $this->rollSubscriptionForward($subscription);
        }
    }

    public function rollSubscriptionForward($subscription)
    {
        if (!$subscription instanceof subscriptionEntity) {
            $id = $subscription->id;
            $subscription = new subscriptionEntity();
            $subscription->load($id);
        }

        $date = new \DateTime('now'); // Set last and next billing dates for subscription
        $subscription->last_billing_date = $date->format('Y-m-d');
        if (isset($subscription->next_billing_date) && $subscription->next_billing_date) {
            $date = \DateTime::createFromFormat('Y-m-d', $subscription->next_billing_date);
        }

        if (isset($this->config->app->payments->subscriptions->billingInterval) && $this->config->app->payments->subscriptions->billingInterval) {
            $date->add(new \DateInterval($this->config->app->payments->subscriptions->billingInterval));
        } elseif (isset($this->config->app->payments->subscriptions->specificBillingDate) && $this->config->app->payments->subscriptions->specificBillingDate) {
            $date->modify($this->config->app->payments->subscriptions->specificBillingDate);
        } else {
            $date->add(new \DateInterval('P1M'));
        }
        $subscription->next_billing_date = $date->format('Y-m-d');

        $subscription->store();
    }

    public function createSubscription($companyId, $userId, $productId, $productQuantity, $startDate=null, $gracePeriodEnds=null, $nextBillingDate=null, $temporaryPriceOverride=null)
    {
        $subscription = new subscriptionEntity();

        // Look for existing subscriptions for this user, if exists, update to trigger hooks, and use that
        if (isset($this->config->app->payments->subscriptions->reuseExistingSubscriptions) && $this->config->app->payments->subscriptions->reuseExistingSubscriptions) {
            $sub = $subscription->getByUserCompanyProductAndDate($userId, $companyId, $productId, date('Y-m-d'));
        }
        if (isset($sub) && isset($sub['id']) && $sub['id']) {
            $subscription->load($sub['id']);
            $subscription->product_quantity = $subscription->product_quantity + $productQuantity;
        } else {
            $subscription->company_id = $companyId;
            $subscription->user_id = $userId;
            if ($startDate instanceof \DateTime) {
                $subscription->date_starts = $startDate->format('Y-m-d');
            } else {
                $subscription->date_starts = !is_null($startDate) ? $startDate : date('Y-m-d');
            }

            if (isset($this->config->app->payments->subscriptions->disableOnCreation) && $this->config->app->payments->subscriptions->disableOnCreation) {
                $subscription->enabled = 0;
            } else {
            $subscription->enabled = 1;
            }
            $subscription->created = date('Y-m-d h:i:s');
            $subscription->product_id = $productId;
            $subscription->product_quantity = $productQuantity;
            $subscription->code = bin2hex(random_bytes(32)); // unique code to use in email links

            if ($nextBillingDate) {
                if ($nextBillingDate instanceof \DateTime) {
                    $date = $nextBillingDate;
                } else {
                $date = \DateTime::createFromFormat('Y-m-d', $nextBillingDate);
                }
            }  else {
                $date = new \DateTime('now');
            }
            $subscription->next_billing_date = $date->format('Y-m-d');

            if ($gracePeriodEnds) {
                if ($gracePeriodEnds instanceof \DateTime) {
                    $subscription->grace_period_ends = $gracePeriodEnds->format('Y-m-d');
                } else {
                    $subscription->grace_period_ends = $gracePeriodEnds;
                }
            }

            if (isset($temporaryPriceOverride) && $temporaryPriceOverride->price > 0) {
                $subscription->product_price_override = $temporaryPriceOverride->price;
                $subscription->product_price_override_ends = $temporaryPriceOverride->ends;
            }
        }

        $subscription->store(); // Always store (even for existing subs), in order to trigger hooks

        return $subscription;
    }

    public function createSubscriptionPaymentLinks($payment)
    {

        $upserts = [];
        foreach ($this->subscriptions as $subscription) {
            $upserts[] = [
                'subscription_id' => $subscription->id,
                'payment_id' => $payment->id,
            ];
        }
        if (count($upserts) > 0) {
            Database::persistRawBatch($upserts, 'subscription_payment');
        }
    }

    public function reenableUserSubscriptions($user)
    {
        $subscriptionEntity = new subscriptionEntity();
        $subscriptions = $subscriptionEntity->getByUser($user->id);
        $now = date('Y-m-d');
        foreach ($subscriptions as $subscription) {
            $subscriptionEntity->load($subscription->id);
            if ($subscriptionEntity->enabled == '0' && $subscriptionEntity->suspended == '1') {
                if (!$subscriptionEntity->date_expires || $subscriptionEntity->date_expires > $now) {
                    $subscriptionEntity->enabled = true;
                    $subscriptionEntity->store();
                }
            }
        }
    }

    public function createSubscriptionsWithToken($user, $paymentToken, $startDate=null, $gracePeriodEnds=null, $nextBillingDate=null)
    {
        $subscriptions = [];

        $companyId = null;
        if (class_exists('apexl\Io\modules\company\entities\companyEntity')) {
            $company = new companyEntity();
            $company->loadByUser($user->id);
            $companyId = $company ? $company->id : null;
        }
        $data = json_decode($paymentToken->data);
        if (isset($data->products) && is_array($data->products)) { //i.e. if this is a woocommerce order
            $productRemoteReferenceIdsEntity = new productRemoteReferenceIdsEntity();
            $productEntity = new productEntity();
            foreach ($data->products as $product) {
                if (isset($product->product_id)) {
                    $productId = $productRemoteReferenceIdsEntity->getProductIdUsingRemoteReferenceId($product->product_id); // Get Dashboard product ID from Woo Product ID
                } elseif (isset($product->product_name)) { // If no ID provided, load by name
                    $productEntity->loadByName($product->product_name);
                    if (!isset($productEntity->id) || !$productEntity) { // Automatically create new product if it doesn't exist
                        $productEntity = new productEntity();
                        $productEntity->type = $product->type ?? "";
                        $productEntity->product_name = $product->product_name;
                        $productEntity->price = $product->price ?? 0;
                        $productEntity->code = $product->code ?? "";
                        $productEntity->created = time();
                        $productEntity->store();
                    }
                    $productId = $productEntity->id;
                } else if (isset($product->dashboardProductId)) {
                    $productId = $product->dashboardProductId;
                }
                $subscriptions[] = $this->createSubscription($companyId, $user->id, $productId, $product->quantity, $startDate, $gracePeriodEnds, $nextBillingDate, $data->productPriceOverride ?? NULL);
            }
        } else { // None-woocommerce order - may be extinct now? Not sure...
            $subscriptions[] = $this->createSubscription($companyId, $user->id, $this->getProductIdFromToken($paymentToken), 1, $startDate, $gracePeriodEnds, $nextBillingDate, $data->productPriceOverride ?? NULL);
        }

        return $subscriptions;
    }

}