<?php

namespace apexl\Io\modules\subscription\entities\operators;

use apexl\Io\modules\component\entities\componentEntity;
use apexl\Io\modules\component\entities\componentTypeEntity;
use apexl\Io\modules\component\entities\propertyEntity;
use apexl\Io\operators\entityDatabaseOperator;

class subscriptionOperator extends entityDatabaseOperator{

    public function __construct($table, $primaryKey = 'id')
    {
        parent::__construct($table, $primaryKey);
    }

    public function getByCompany($companyId){
        return $this->vault->select('subscription')->fields()
            ->where('CompanyId', $companyId)
            ->execute()
            ->fetchAll();
    }

    public function getByUser($userId){
        return $this->vault->select('subscription')->fields()
            ->where('user_id', $userId)
            ->execute()
            ->fetchAll();
    }

    public function getByCode($code){
        $subscription = $this->vault->select('subscription')->fields()
            ->where('code', $code)
            ->execute()
            ->fetchAssoc();

        return (object)$subscription;
    }

    public function disableSubscription($id){
        return $this->vault->update('subscription')->fields(['enabled' => 0])->where('id', $id)->execute();
    }

    public function enableSubscription($id){
        return $this->vault->update('subscription')->fields(['enabled' => 1])->where('id', $id)->execute();
    }

    public function getTypesForSubscription($subscriptionId){
        return $this->vault->select('subscription_types')->fields()
            ->where('SubscriptionId', $subscriptionId)
            ->execute()
            ->fetchAll();
    }

    public function getByNextBillingDate($nextBillingDate, $limit=20){
        return $this->vault
            ->select($this->dbTable)
            ->fields()
            ->where('next_billing_date', $nextBillingDate, '<=')
            ->limit($limit)
            ->execute()
            ->fetchAll();
    }

    public function getByPaymentId($paymentId){
        $entity = $this->vault
            ->select('subscription')
            ->fields()
            ->leftjoin('subscription_payment', $this->dbTable.'.id=subscription_payment.subscription_id', '','')
            ->where('payment_id', $paymentId)
            ->execute()
            ->fetchAssoc();

        return ['updateEntityData' => true, 'data' => $entity];
    }

    /**
     * Get all subscriptions for which billing hasn't been attempted in the last X hours, indexed by user
     * @param $nextBillingDate
     * @param $limit
     * @return mixed
     */
    public function getSubscriptionsToBill($nextBillingDate, $limit=10, $cutoffHours=6){
        $time = new \DateTime('now');
        $timeInterval = $cutoffHours*60;
        $time->sub(new \DateInterval("PT".$timeInterval."M"));

        $rows = $this->vault
            ->select($this->dbTable)
            ->fields($this->dbTable.'.*')
            ->leftjoin('subscription_payment', $this->dbTable.'.id=subscription_payment.subscription_id', '','')
            ->leftjoin('payment', 'payment.id=subscription_payment.payment_id AND payment.created > '.$time->format('U'), '','')
            ->where('next_billing_date', $nextBillingDate, '<=')
            ->where('payment.id',NULL, "IS NULL")
            ->where('enabled', true)
            ->orderBy($this->dbTable.'.last_checked', 'ASC')
            ->limit($limit)
            ->execute()
            ->fetchAll();

        // Index by userId
        $subscriptionsByUser = [];
        $count = 0;
        foreach ($rows as $row) {
            if (!isset($subscriptionsByUser[$row->user_id])) {
                $subscriptionsByUser[$row->user_id] = [];
                $count++;
            }
            $subscriptionsByUser[$row->user_id][$row->id] = $row;

            if ($count >= $limit) break;
        }

        return $subscriptionsByUser;
    }

    /**
     * Get all subscriptions for which billing hasn't been attempted in the last X hours, indexed by user
     * @param $nextBillingDate
     * @param $limit
     * @return mixed
     */
    public function getSubscriptionIccidsToBill($nextBillingDate, $limit=2, $cutoffHours=6){
        //$time = new \DateTime('now');
        //$timeInterval = $cutoffHours*60;
        //$time->sub(new \DateInterval("PT".$timeInterval."M"));

        $rows = $this->vault
            ->select($this->dbTable)
            ->fields([$this->dbTable.'.*', 'GROUP_CONCAT(DISTINCT billingSimSettings.iccid) AS iccids'])
            //->leftjoin('subscription_payment', $this->dbTable.'.id=subscription_payment.subscription_id', '','')
            //->leftjoin('payment', 'payment.id=subscription_payment.payment_id AND payment.created > '.$time->format('U'), '','')
            ->leftjoin('billingSimSettings', 'billingSimSettings.subscriptionId='.$this->dbTable.'.id', '','')
            ->where('next_billing_date', $nextBillingDate, '<=')
            //->where('payment.id',NULL, "IS NULL")
            ->where('enabled', true)
            //->where($this->dbTable.'.id', 18)
            ->groupBy($this->dbTable.'.id')
            ->orderBy($this->dbTable.'.last_checked', 'ASC')
            ->limit($limit)
            ->execute()
            ->fetchAll();

        // Index by userId
        $subscriptionsByUser = [];
        $count = 0;
        foreach ($rows as $row) {
            if (!isset($subscriptionsByUser[$row->user_id])) {
                $subscriptionsByUser[$row->user_id] = [];
                $count++;
            }
            $subscriptionsByUser[$row->user_id][$row->id] = $row;

            if ($count >= $limit) break; // Only process $limit subscriptions at any one time
        }

        return $subscriptionsByUser;
    }

    public function getByUserCompanyProductAndDate($userId, $companyId, $productId, $dateStarts){

        $query = $this->vault->select('subscription')->fields()
            ->where('user_id', $userId)
            ->where('product_id', $productId)
            ->where('date_starts', $dateStarts)
            ;

        if ($companyId && $companyId > 0) {
            $query->where('company_id', $companyId);
        }

        return $query->execute()
            ->fetchAssoc();
    }

    public function getEndOfGracePeriodCache(){
        $data = $this->vault->select('subscription')->fields('id, grace_period_ends')
            ->execute()
            ->fetchAll();
        $results = [];
        foreach ($data as $item) {
            if ($item->grace_period_ends) {
                $results[$item->id] = $item->grace_period_ends;
            }
        }

        return $results;
    }

    public function loadAllActiveWithPaymentData($conditions = [], $orderBy = [], $limit = FALSE, $offset = FALSE){
        /**
        SELECT subscription.*, COUNT(billingSimSettings.iccid) AS assigned_sims, ABS(COUNT(billingSimSettings.iccid)-product_quantity) AS missing_sims, sp.payment_id, p.amount, p.status, p.remote_order_id
        FROM subscription
        LEFT JOIN billingSimSettings ON billingSimSettings.subscriptionId = subscription.id
        JOIN subscription_payment sp on sp.subscription_id = subscription.id and sp.payment_id = (
        select MIN(payment_id)
        from subscription_payment
        where subscription_id = subscription.id
        )
        JOIN payment p on sp.payment_id = p.id
        WHERE enabled = 1
        GROUP BY subscription.id, sp.id, p.id
         */

        $query = $this->vault->select('subscription')->fields(['subscription.*', 'COUNT(billingSimSettings.iccid) AS assigned_sims', 'ABS(COUNT(billingSimSettings.iccid)-product_quantity) AS missing_sims', 'sp.payment_id', 'p.amount', 'p.status', 'p.remote_order_id']);
        $query->leftjoin('billingSimSettings', 'billingSimSettings.subscriptionId='.$this->dbTable.'.id', '','');
        $query->join('subscription_payment sp', 'sp.subscription_id='.$this->dbTable.'.id and sp.payment_id = (
        select MIN(payment_id)
        from subscription_payment
        where subscription_id = subscription.id
        )', '','');
        $query->join('payment p', 'sp.payment_id', 'p.id')/*->where('enabled', 1)*/;
        foreach($conditions as $condition){
            $operator = $condition[2] ?? '=';
            $type = $condition[3] ?? 'AND';
            $postCondition = $condition[4] ?? '';
            $query->where($condition[0], $condition[1], $operator, $type, $postCondition);
        }
        $query->groupBy('subscription.id, sp.id, p.id');
        $query->orderBy('missing_sims', 'DESC')->orderBy('subscription.id', 'DESC');//orderBy[]=missing_sims&orderBy[]=DESC&orderBy[]=subscription.id&orderBy[]=DESC
        if(!empty($orderBy)){
            if(!is_array($orderBy)){
                //convert to an array
                $temp = $orderBy;
                $orderBy = [];
                $orderBy[0] = $temp;
            }
            for ($i=0; $i<12; $i+=2) { // Allow multiple Order By clauses
                if (isset($orderBy[$i]) && isset($orderBy[$i+1])) {
                    $direction = $orderBy[$i+1] ?? 'DESC';
                    $query->orderBy($orderBy[$i], $direction);
                }
            }
        }
        if($limit !== FALSE){
            $query->limit($limit, $offset);
        }
        return $query->execute()->fetchAll();
    }

    public function getWordPressOrderInfo($orderId){
        $this->setOperatorVault('geosim');
        $orderData = $this->vault->select('wp_postmeta')->fields()->where('post_id', $orderId)->execute()->fetchAll();
        $this->setOperatorVault('default');
        return $orderData;
    }

    public function getWordPressOrderStatus($orderId){
        $this->setOperatorVault('geosim');
        $orderData = $this->vault->select('wp_wc_order_stats')->fields()->where('order_id', $orderId)->execute()->fetchAssoc();
        $this->setOperatorVault('default');
        return $orderData;
    }

    /**
     * Function to get all active subscriptions that are:
     *   1. next payment overdue by 7 days
     *   2. have 3 payment attempts in the last 7 days which do not have status='successful'
     *
     *  These subscriptions are singled out for suspension
     * @param $numFailedPayments
     * @return void
     */
    public function getOverdueWithFailedPaymentAttempts($numFailedPayments=3)
    {
        $time = new \DateTime('now');
        $time->sub(new \DateInterval("P7D")); // Find all payment attempts in the last 7 days

        $subscriptions = $this->vault
            ->select($this->dbTable)
            ->fields([$this->dbTable.'.*', 'count(payment.id) as numPayments'])
            ->leftjoin('subscription_payment', $this->dbTable.'.id=subscription_payment.subscription_id', '','')
            ->leftjoin('payment', 'payment.id=subscription_payment.payment_id AND payment.created > '.$time->format('U'), '','')
            ->where('next_billing_date', $time->format('Y-m-d'), '<=')
            ->where('payment.status', 'successful', "!=")
            ->where('enabled', true)
            ->groupBy($this->dbTable.'.id')
            ->orderBy($this->dbTable.'.last_checked', 'ASC')
            ->execute()
            ->fetchAll();
        foreach ($subscriptions as $id => $subscription) {
            if ($subscription->numPayments < $numFailedPayments) {
               unset($subscriptions[$id]);
            }
        }

        return $subscriptions;
    }
}