<?php

namespace apexl\Io\modules\user\entities;

use apexl\entityCore\interfaces\EntityOperatorInterface;
use apexl\entityCore\traits\hasCasts;
use apexl\Io\includes\Entity;
use apexl\Io\includes\System;
use apexl\Io\includes\Utils;
use apexl\Io\modules\user\collections\permissionCollection;
use apexl\Io\modules\user\collections\roleCollection;
use apexl\Io\modules\user\collections\userMetaCollection;
use apexl\Io\modules\user\entities\operators\userOperator;
use apexl\Io\modules\user\services\currentUser;
use apexl\Io\modules\user\services\userMetaManagerService;
use apexl\Io\modules\user\services\userPermissionManagerService;
use apexl\Io\modules\user\services\userRoleManagerService;
use app\module\newtonsTrailers\Interface\PermissionInterface;
use DateTimeImmutable;
use Exception;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;

/**
 * @property ?int $id
 * @property string $first_name
 * @property string $last_name
 * @property bool $active
 * @property string $email
 * @property string $password
 * @property ?DateTimeImmutable $last_login
 * @property ?DateTimeImmutable $created
 * @property ?string $created_by
 * @property ?DateTimeImmutable $modified
 * @property ?string $modified_by
 * @property string $salt
 * @mixin userOperator
 */
class userEntity extends Entity
{
    use hasCasts;

    final public const TABLE = 'users';

    /** @var userOperator */
    protected EntityOperatorInterface $operator;

    private userMetaManagerService $userMetaManager;
    private userPermissionManagerService $permissionManager;
    private userRoleManagerService $roleManager;

    /**
     * @throws NotFoundExceptionInterface
     * @throws ContainerExceptionInterface
     * @throws Exception
     */
    public function __construct()
    {
        parent::__construct(self::TABLE);
        $this->setOperator(new userOperator(self::TABLE));
        $this->setSensitiveData(['password', 'salt']);
        $this->setRequiredData(['email', 'first_name', 'last_name', 'password']);
        $this->addReadOnlyField('salt')->addReadOnlyField('created');

        $this->userMetaManager = System::makeRegisteredService(userMetaManagerService::class, [
            'user' => $this,
        ]);

        $this->permissionManager = System::makeRegisteredService(userPermissionManagerService::class, [
            'user' => $this,
        ]);

        $this->roleManager = System::makeRegisteredService(userRoleManagerService::class, [
            'user' => $this,
        ]);
    }

    /**
     * @param  array  $conditions
     * @param  array  $orderBy
     * @param  false  $limit
     * @param  false  $offset
     * @param  bool  $asEntities
     * @return array
     * @throws Exception
     */
    public function loadMultiple(
        array $conditions = [],
        array $orderBy = [],
        $limit = false,
        $offset = false,
        bool $asEntities = true
    ): array {
        //we need to allow users with permission to bypass the default access control.
        $currentUser = (currentUser::getInstance())::getCurrentUser();
        if ($currentUser->isAllowed('ViewUsers')) {
            $conditions = array_merge($conditions, ["id" => ["users.id", null, "IS NOT"]]);
        }

        return parent::loadMultiple($conditions, $orderBy, $limit, $offset, $asEntities);
    }

    public function isAllowed(string|array $permissions): bool
    {
        $permissions = (array) $permissions;
        foreach ($permissions as $permission) {
            //we have a few core permissions to account for 'standard' functionality
            return match ($permission) {
                'canDoNothing' => false,
                'AllowAll' => true,
                'IsLoggedIn' => $this->isLoggedIn(),
                default => $this->permissionManager->can($permission),
            };
        }

        //no access? Return FALSE;
        return false;
    }

    public function isLoggedIn(): bool
    {
        return ($this->id !== null && !empty($this->id));
    }

    public function getUsersByRoleName($name)
    {
        $roles = new roleEntity();
        $roles->loadByName($name);

        return $this->operator->getUsersByRoleId($roles->id);
    }

    public function addRole(string|roleEntity|int $role): userEntity
    {
        $this->roleManager->add($role);

        return $this;
    }

    public function removeRoles(): userEntity
    {
        $this->roleManager->clear();
        $this->data['roles'] = [];

        return $this;
    }

    public function getNiceName(): string
    {
        return $this->id !== null ? sprintf('%s %s', $this->first_name, $this->last_name) : 'Missing User';
    }

    public function hasRole(string|int|roleEntity $roleName): bool
    {
        return $this->roleManager->has($roleName);
    }

    public function newForgottenPasswordLink($string): void
    {
        $this->operator->newForgottenPasswordLink($this->email, $string);
    }

    public function allMeta(): userMetaCollection
    {
        return $this->userMetaManager->allMeta();
    }

    public function hasMeta(string $key): bool
    {
        return $this->userMetaManager->hasMeta($key);
    }

    public function meta(string $key): ?userMetaEntity
    {
        return $this->userMetaManager->meta($key);
    }

    public function removeMeta(string $key): void
    {
        $this->userMetaManager->removeMeta($key);
    }

    public function addMeta($key, $value): void
    {
        $this->userMetaManager->addMeta($key, $value);
    }

    public function permissions(bool $includeRoles = true): permissionCollection
    {
        $permissions = $this->permissionManager->permissions();

        if ($includeRoles) {
            $this->roles()->each(fn(roleEntity $role) => $permissions->merge(
                $role->permissions()
            ));
        }

        return $permissions->unique();
    }

    public function roles(): roleCollection
    {
        return $this->roleManager->roles();
    }

    public function addPermission(PermissionInterface $permission): void
    {
        $this->permissionManager->add($permission);
    }

    public function removePermission(PermissionInterface $permission): void
    {
        $this->permissionManager->remove($permission);
    }

    protected function fieldConfig(): array
    {
        return $this->withCasts([
            "id" => [
                'name' => "ID",
            ],
            "first_name" => [
                'name' => "First Name",
            ],
            "last_name" => [
                'name' => "Last Name",
            ],
            "active" => [
                'name' => "Active",
                'display' => function ($active) {
                    return $active ? 'Yes' : 'No';
                },
            ],
            "email" => [
                'name' => "Email Address",
            ],
            "password" => [
                'name' => "Password",
                'display' => function (): string {
                    return 'Hidden';
                },
            ],
            "last_login" => [
                'name' => "Last Login",
                'display' => function ($lastLogin) {
                    return $lastLogin ? date('d-m-Y H:i:s', $lastLogin) : 'Never';
                },
            ],
            "salt" => [
                'name' => "Salt",
                'display' => function (): string {
                    return 'Hidden';
                },
            ],
            "modified" => [
                'name' => "Modified",
                'display' => function () {
                    return Utils::HIDE_FROM_DISPLAY;
                },
            ],
            "modified_by" => [
                'name' => "Modified By",
                'display' => function () {
                    return Utils::HIDE_FROM_DISPLAY;
                },
            ],
            "roles" => [
                'name' => "Roles",
                'display' => function ($roles) {
                    $niceNames = [];
                    $roleEntity = new roleEntity();
                    if (is_string($roles)) {
                        $roles = unserialize($roles) ?: [];
                    }
                    if (!empty($roles) && is_array($roles)) {
                        foreach ($roles as $rid) {
                            $roleEntity->load($rid);
                            $niceNames[] = $roleEntity->name;
                        }
                    }

                    return $niceNames === [] ? 'No Roles' : implode(', ', $niceNames);
                },
            ],
        ]);
    }
}
