<?php

declare(strict_types=1);

namespace apexl\Io\modules\user\services;

use apexl\Config\Singleton as Config;
use apexl\Io\modules\user\collections\permissionCollection;
use apexl\Io\modules\user\entities\userEntity;
use apexl\Vault\Vault;
use app\module\newtonsTrailers\Enum\Permission\Common;
use app\module\newtonsTrailers\Interface\PermissionInterface;

final readonly class userPermissionManagerService
{
    const TABLE__PERMISSIONS_USERS = 'permissions__users';
    private bool $isGodUser;

    public function __construct(private userEntity $user, private Vault $vault, Config $config)
    {
        $this->isGodUser = $this->isGodUser($config);
    }

    private function isGodUser(Config $config): bool
    {
        return ($config->app->permissions->enableGodUser ?? false) && $this->user->id == 1;
    }

    public function can(PermissionInterface|string $permission): bool
    {
        if ($this->isGodUser) {
            return true;
        }

        if ($this->permissions()->contains(Common::BYPASS->asString())) {
            return true;
        }

        if ($this->permissions()->contains(
            $permission instanceof PermissionInterface ? $permission->asString() : $permission
        )) {
            return true;
        }

        if ($this->user->roles()->can($permission)) {
            return true;
        }

        return false;
    }

    public function permissions(): permissionCollection
    {
        $userId = $this->user->id;

        return permissionCollection::createFromQueryResult(
            $this->vault->select(self::TABLE__PERMISSIONS_USERS)
                ->fields('permission')
                ->where('user', $userId)
                ->execute()
                ->fetchAll()
        );
    }

    public function add(PermissionInterface $permission): void
    {
        $userId = $this->user->id;
        $this->vault->insert(self::TABLE__PERMISSIONS_USERS)
            ->fields([
                'user' => $userId,
                'permission' => $permission->asString(),
            ])->execute();
    }

    public function remove(PermissionInterface $permission): void
    {
        $userId = $this->user->id;
        $this->vault->delete(self::TABLE__PERMISSIONS_USERS)
            ->where('user', $userId)
            ->where('permission', $permission->asString())
            ->execute();
    }


}
