<?php

declare(strict_types=1);

namespace apexl\Io\modules\user\services;

use apexl\Io\modules\user\exceptions\JwtException;
use apexl\Io\modules\user\interfaces\JwtServiceInterface;
use DateTimeImmutable;
use Exception;
use Firebase\JWT\JWT;
use Psr\Http\Message\ServerRequestInterface;
use function apexl\Io\config;

final readonly class JwtFromRequestService implements JwtServiceInterface
{
    public function __construct(private ServerRequestInterface $request) {}

    /**
     * @throws JwtException
     */
    public function token(): object
    {
        $tokenFromAuthHeader = $this->tokenFromAuthHeader();

        try {
            $token = JWT::decode(
                $tokenFromAuthHeader,
                config('auth.jwt.secret_key'),
                [config('auth.jwt.algorithm')]
            );

            $this->assertTokenIsValid($token);

            return $token;
        } catch (Exception $exception) {
            throw new JwtException($exception->getMessage(), $exception->getCode(), $exception);
        }

    }

    /**
     * @throws JwtException
     */
    private function tokenFromAuthHeader(): string
    {
        if (preg_match(
            '/^Bearer (\S+)/',
            $this->authHeader(),
            $matches,
        )) {
            return $matches[1];
        }

        throw new JwtException('Authorization header pattern not recognized');
    }

    /**
     * @throws JwtException
     */
    private function authHeader(): string
    {
        if ($this->request->hasHeader('Authorization')) {
            return $this->request->getHeader('Authorization')[0];
        }

        throw new JwtException('Authorization header not found');
    }

    /**
     * @throws JwtException
     */
    private function assertTokenIsValid(object $token): void
    {
        $serverName = $this->serverName();
        $now = new DateTimeImmutable();

        if (
            $token->iss === $serverName
            && $token->nbf <= $now->getTimestamp() && $token->exp >= $now->getTimestamp()
            && !empty($token->sessionId)
            && isset($token->userId)
            && $token->userId > 0
        ) {
            return;
        }

        throw new JwtException('JWT is not valid');
    }

    /**
     * @throws JwtException
     */
    private function serverName(): string
    {
        $serverName = $_SERVER['SERVER_NAME'] ?? null;

        if ($serverName === null) {
            throw new JwtException('Server name not set');
        }

        return $serverName;
    }
}
