<?php

namespace apexl\Io\modules\user\controllers;

use apexl\Io\exceptions\ValidationException;
use apexl\Io\factories\RequestHelperFactory;
use apexl\Io\includes\Controller;
use apexl\Io\modules\user\Actions\StartLoggedInSession;
use apexl\Io\modules\user\Actions\ValidateLoginAction;
use apexl\Io\modules\user\exceptions\LoginException;
use apexl\Io\modules\user\exceptions\UserAlreadyLoggedInException;
use apexl\Io\modules\user\RequestHelper\LoginRequestHelper;
use apexl\Io\modules\user\services\currentUser;
use Exception;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;

final readonly class LoginController extends Controller
{
    /**
     * @throws Exception
     */
    public function __invoke(
        RequestHelperFactory $requestHelperFactory,
        ValidateLoginAction $loginAction,
        StartLoggedInSession $startLoggedInSession,
        ServerRequestInterface $request,
        ResponseInterface $response
    ): ResponseInterface {
        try {
            $requestHelper = $requestHelperFactory->make(LoginRequestHelper::class, $request);


        } catch (ValidationException $exception) {
            $this->output->addMessage(
                'user.login.validation',
                'error',
                $exception->getMessage(),
            );

            return $this->json($response, [], 400);
        }
        try {
            $userId = $request->getAttribute('user'); // set in an earlier middleware
            if ($userId !== 0) {
                throw new UserAlreadyLoggedInException($userId);
            }

        } catch (UserAlreadyLoggedInException) {
            $this->addRedirect($requestHelper->redirect());

            return $this->json($response);
        }

        try {
            $user = $loginAction->handle($requestHelper->username(), $requestHelper->password());

            [$authToken, $refreshToken] = $startLoggedInSession->handle($user);

            $this->output->addResponse(
                $request,
                [
                    'access_token' => $authToken,
                    'refresh_token' => $refreshToken,
                    'expires_in' => currentUser::getTokenExpiryFromNow($authToken),
                ]
            );

            $this->addRedirect($requestHelper->redirect());

            return $this->json($response);
        } catch (LoginException $exception) {
            $this->output->addMessage('user.login.validation', 'error', $exception->getMessage());
            $this->output->addResponse($request, [], false); // added so we can hook into this elsewhere.

            return $this->json($response, [], $exception->getCode());
        }
    }

    private function addRedirect(?string $redirect): void
    {
        $this->output->addRedirect('user.login.redirect', $redirect ?? '/');
    }

}
