<?php

namespace apexl\Io\modules\user\classes;

use apexl\Io\enums\HttpMethod;
use apexl\Io\includes\Entity;
use apexl\Io\includes\RouteManager;
use apexl\Io\modules\display\controllers\displayController;
use apexl\Io\modules\user\enums\permissions\Common;
use apexl\Io\modules\user\interfaces\PermissionInterface;
use apexl\Io\recommendedIncludes\displayModule;

class userManagedDisplayModule extends displayModule
{
    /**
     * Method to add a route, allows the module to have a base path configured and for easy overrides.
     * This Method adds one extra option over the base, to permissions protect a route.
     */
    protected function addProtectedDisplayRoute(
        RouteManager $routeManager,
        HttpMethod $verb,
        string $name,
        string $pattern,
        $callable,
        PermissionInterface $permission = Common::ALLOW_ALL,
    ): static {
        $this->addRoute(
            $routeManager,
            $verb,
            $name,
            'display/' . ltrim($pattern, '/'),
            $callable
        );
        $routeManager->addRouteArg($name, 'permission', serialize($permission));

        return $this;
    }

    /**
     * Method to add a route, allows the module to have a base path configured and for easy overrides.
     * This Method adds one extra option over the base, to permissions protect a route.
     */
    protected function addProtectedDataRoute(
        RouteManager $routeManager,
        HttpMethod $verb,
        string $name,
        string $pattern,
        $callable,
        PermissionInterface $permission = Common::ALLOW_ALL,
    ): static {
        $this->addRoute(
            $routeManager,
            $verb,
            $name,
            'display/data/' . ltrim($pattern, '/'),
            $callable
        );
        $routeManager->addRouteArg($name, 'permission', serialize($permission));

        return $this;
    }

    /**
     * Method to add a route, allows the module to have a base path configured and for easy overrides.
     * This Method adds one extra option over the base, to permissions protect a route.
     */
    protected function addProtectedFormRoute(
        RouteManager $routeManager,
        string $name,
        string $pattern,
        $callable,
        PermissionInterface $permission = Common::ALLOW_ALL,
    ): static {
        $this->addDisplayRoute(
            $routeManager,
            HttpMethod::GET,
            $name,
            'forms/' . $pattern,
            $callable
        );
        $routeManager->addRouteArg($name, 'permission', serialize($permission));

        return $this;
    }

    /**
     * Method to add a route, allows the module to have a base path configured and for easy overrides.
     * This Method adds one extra option over the base, to permissions protect a route.
     */
    protected function addProtectedEntityFormRoute(
        RouteManager $routeManager,
        Entity $entity,
        string $pattern,
        $callable,
        PermissionInterface $permission = Common::ALLOW_ALL,
    ): static {
        $this->addDisplayRoute(
            $routeManager,
            HttpMethod::GET,
            $entity->getEntityName() . '.entity.form.createUpdate',
            'forms/entities/' . $this->getName() . '/' . $pattern,
            $callable
        );
        $routeManager->addRouteArg(
            $entity->getEntityName() . '.entity.form.createUpdate',
            'permission',
            serialize($permission)
        );
        $routeManager->addRouteArg(
            $entity->getEntityName() . '.entity.form.createUpdate',
            'entity',
            $entity::class
        );

        return $this;
    }

    /**
     * Method to add a route, allows the module to have a base path configured and for easy overrides.
     * This Method adds one extra option over the base, to permissions protect a route.
     */
    protected function addProtectedRootRoute(
        RouteManager $routeManager,
        HttpMethod $verb,
        string $name,
        string $pattern,
        $callable,
        PermissionInterface $permission = Common::ALLOW_ALL,
    ): static {
        $this->rootRoute($routeManager, $verb, $name, $pattern, $callable);

        if ($this->shouldAddRoutePermissions()) {
            $routeManager->addRouteArg($name, 'permission', serialize($permission));
        }

        return $this;
    }

    private function shouldAddRoutePermissions(): bool
    {
        if (config('auth.disable_route_permissions_locally') && config('app.environment') === 'local') {
            return false;
        }
        
        return true;
    }

    protected function addProtectedAdminRootRoute(
        RouteManager $routeManager,
        HttpMethod $verb,
        string $name,
        string $pattern,
        $callable,
        PermissionInterface $permission = Common::ALLOW_ALL,
    ): static {
        $this->rootRoute($routeManager, $verb, $name, 'admin/' . $pattern, $callable);

        if ($this->shouldAddRoutePermissions()) {
            $routeManager->addRouteArg($name, 'permission', serialize($permission));
        }

        return $this;
    }

    /**
     * Declare intention to build routes for this entity. Will take defaults if not previously defined.
     * @param null $callables
     * @param null $patterns
     * @return $this
     */
    protected function addProtectedEntityRoutes(RouteManager $routeManager, Entity $entity, $callables = null, $patterns = []): static
    {
        $this->setEntityPatterns($entity, $patterns);

        foreach ($this->entityPatterns[$entity->getEntityName()] as $verb => $pattern) {
            if (isset($callables[$verb]) && is_array($callables[$verb])) {
                $this->setEntityCallable(
                    $entity,
                    $verb,
                    ($callables[$verb][0] ?? displayController::class . ':' . $verb)
                );
            }
        }

        //build the routes.
        $this->buildEntityRoutes($routeManager, $entity, true);
        //protect the built routes
        foreach ($this->entityPatterns[$entity->getEntityName()] as $verb => $pattern) {
            $routeManager->addRouteArg(
                $entity->getEntityName() . '.display.' . strtolower((string) $verb),
                'permission',
                serialize($callables[$verb][1] ?? Common::ALLOW_ALL)
            );
            $routeManager->addRouteArg(
                $entity->getEntityName() . '.display.' . strtolower((string) $verb),
                'entity',
                $entity::class
            );
        }

        return $this;
    }

    /**
     * @param $entity
     * @param array $patterns
     * @return $this|userManagedDisplayModule
     */
    protected function setEntityPatterns($entity, array $patterns = []): static
    {
        $this->defaultEntityPatterns = [
            'all' => '/list', //Root display pattern, by default load all entities
            'post' => '/create', //no pattern here as we should POST to the base entity path (by default)
            'put' => '/{' . $entity->primaryKey . '}/edit',
            'delete' => '/{' . $entity->primaryKey . '}/delete',
            'get' => '[/{' . $entity->primaryKey . '}]',
        ];
        $this->patternsOverridden = $patterns !== [];
        $this->entityPatterns[$entity->getEntityName()] = array_merge($this->defaultEntityPatterns, $patterns);

        return $this;
    }

    /**
     * Normal structure is /ModuleName/EntityName/{pattern}
     * Build any default entity routes.
     */
    protected function buildEntityRoutes(RouteManager $routeManager, Entity $entity, bool $rootRoutes = false): void
    {
        $method = $rootRoutes ? 'rootRoute' : 'addRoute';
        foreach ($this->entityPatterns[$entity->getEntityName()] as $verb => $pattern) {
            if (isset($this->entityControllers[$entity->getEntityName()][$verb])) {
                $basePattern = $this->patternsOverridden ? '' : (str_replace(
                    'entity',
                    '',
                    strtolower((string) $entity->getEntityName())
                ));
                $this->{$method}(
                    $routeManager,
                    HttpMethod::GET, //Always Get for display.
                    $entity->getEntityName() . '.display.' . strtolower((string) $verb),
                    $basePattern . $pattern,
                    $this->entityControllers[$entity->getEntityName()][$verb]
                );
            }
        }
    }
}
