<?php

namespace apexl\Io\modules\user\controllers;

use apexl\hashing\Hash;
use apexl\Io\includes\Controller;
use apexl\Io\includes\Hook;
use apexl\Io\includes\Routes;
use apexl\Io\includes\System;
use apexl\Io\modules\company\entities\companyEntity;
use apexl\Io\modules\email\services\emailService;
use apexl\Io\modules\email\services\templateService;
use apexl\Io\modules\formbuilder\includes\inputField;
use apexl\Io\modules\subscription\entities\subscriptionEntity;
use apexl\Io\modules\subscription\services\subscriptionService;
use apexl\Io\modules\user\entities\refreshTokenEntity;
use apexl\Io\modules\user\entities\sessionEntity;
use apexl\Io\modules\user\entities\userContactDetailsEntity;
use apexl\Io\modules\user\entities\userEntity;
use apexl\Io\modules\user\services\apexlWPManageService;
use apexl\Io\modules\user\services\currentUser;

use apexl\Io\modules\user\services\userTools;
use apexl\Io\services\Logger;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;

class userController extends Controller
{
    protected $database;
    /** @var userEntity */
    protected userEntity $currentUser;
    protected userTools $userTools;

    public function __construct(currentUser $currentUser, userTools $userTools)
    {
        parent::__construct();
        $this->currentUser = $currentUser::getCurrentUser();
        $this->userTools = $userTools;
    }

    /**
     * @param Request $request
     * @param Response $response
     * @method GET
     * @return Response
     * @throws \Exception
     */
    public function login(Request $request, Response $response): Response
    {
        $body = $request->getParsedBody();
        //basic validation
        if (empty($body) || !isset($body->email) || !isset($body->password)) {
            $this->output::addMessage('user.login.validation', 'error', "Please provide an email address and a password.");
            $this->output::addResponse($request, [], false); //added so we can hook into this elsewhere.
            return System::asJson($response, [], 400);
        }

        //first, grab the user data.
        $user = userEntity::getUserByEmail($body->email);

        $errorText = 'The email address and password don\'t match';
        //if we have user data, grab the salt and password and check the provided one matches.
        //we also need the user to be active, so check that at the same time.
        if(isset($user->email)){
            if ($user->active == 1) {
            $hash = new Hash();
            $match = $hash->matchHash($body->password, $user->password, $user->salt);
            //if we match, complete the login process

            $match = Hook::processHook('userLogin', $match, $user, $request);
            } else {
                $match = FALSE;
                $errorText = "You can only login using a validated account";
            }
        } else {
            $match = FALSE;
            $match = Hook::processHook('userLoginNoMatchingUser', $match, $body->email, $body->password, $request); // This hook adds the possibility of third-party credential check and adding local user
            if ($match) {
                $user = userEntity::getUserByEmail($body->email);
            }
        }

        if($match && isset($user->email) && $user->active == 1){
            list($authToken, $refreshToken) = $this->userTools->startLoggedInSession($user);
            //allow others to act on this response. Hook is keyed by route name.
            if (isset($body->redirectToRoute)) {
                $body->redirectToRoute = base64_decode($body->redirectToRoute);
                $url = strpos($body->redirectToRoute, "/") !== false ? $body->redirectToRoute : Routes::getRoutePattern($body->redirectToRoute);
                $this->output->addMetadata('user.login.validation.redirect', 'redirect', $url);
            }
            $this->output::addResponse($request, ['authToken' => $authToken, 'refreshToken' => $refreshToken]);
            return System::asJson($response);
        }
        //if we end up here, then the login attempt failed, so return a 400
        $this->output::addMessage('user.login.validation', 'error', $errorText);
        $this->output::addResponse($request, [], false); // added so we can hook into this elsewhere.
        return System::asJson($response, [], 400);
    }

    public function loginAsUser(Request $request, Response $response){
        $body = $request->getParsedBody();

        //basic validation
        if(empty($body) || !isset($body->userId)){
            $this->output::addMessage('user.loginAsUser.validation', 'error', "Invalid User.");
            $this->output::addResponse($request, [], FALSE); //added so we can hook into this elsewhere.
            return System::asJson($response, [], 400);
        }

        //first, grab the user data.
        $user = userEntity::load($body->userId);
        if(!isset($user->id)){
            $this->output::addMessage('user.loginAsUser.validation', 'error', "Invalid User.");
            $this->output::addResponse($request, [], FALSE); //added so we can hook into this elsewhere.
            return System::asJson($response, [], 400);
        }

        list($authToken, $refreshToken) = $this->userTools->startLoggedInSession($user);
        //allow others to act on this response. Hook is keyed by route name.
        if (isset($body->redirectToRoute)) {
            $body->redirectToRoute = base64_decode($body->redirectToRoute);
            $url = strpos($body->redirectToRoute, "/") !== false ? $body->redirectToRoute : Routes::getRoutePattern($body->redirectToRoute);
            $this->output->addMetadata('user.loginAsUser.validation.redirect', 'redirect', $url);
        }
        $this->output::addMetadata('user.loginAsUser.post.redirect', 'redirect', '/');
        $this->output::addResponse($request,['authToken' => $authToken, 'refreshToken' => $refreshToken]);
        return System::asJson($response);

    }

    /**
     * @param Request $request
     * @param Response $response
     * @return Response
     * @throws \Exception
     */
    public function refresh(Request $request, Response $response): Response
    {
        $body = $request->getParsedBody();
        if (isset($body->refreshToken)) {
            $refreshTokenEntity = new refreshTokenEntity();
            $refreshTokenEntity->loadByToken(hash('sha256', $body->refreshToken));
            $claims = currentUser::getClaimsFromJWT($request);
            if (isset($refreshTokenEntity->user_id) && isset($claims->userId) && $refreshTokenEntity->user_id == $claims->userId) { // Refresh token exists and matches the current user?
                if ($refreshTokenEntity->expiry > time()) { // Refresh token still valid?
                    $user = userEntity::load($refreshTokenEntity->user_id);
                    if (($authToken = currentUser::createJWT($user, $claims->sessionId, $this->config->app->jwt->secret_key, $this->config->app->jwt->algorithm, ($this->config->app->jwt->lifetime ?? 3600))) !== '' && ($authToken = currentUser::createJWT($user, $claims->sessionId, $this->config->app->jwt->secret_key, $this->config->app->jwt->algorithm, ($this->config->app->jwt->lifetime ?? 3600))) !== '0') {
                        $this->output::addResponse($request, ['authToken' => $authToken]);
                        return System::asJson($response);
                    }
                }
            }
        }

        return System::asJson($response, [], 403);
    }

    /**
     * @param Request $request
     * @param Response $response
     * @return Response
     * @throws \Exception
     * @method POST
     * Automatic install method. Runs the core IO installation process, requires Database user and pass as a minimum to run.
     */
    public function logout(Request $request, Response $response)
    {
        list($token, $error) = currentUser::authenticateJWT($request, $this->config->app->jwt->secret_key, $this->config->app->jwt->algorithm);
        if ($token) {
            $session = sessionEntity::load($token->sessionId, true);
            if (isset($session->sessionId)) {
                $session->active = 0;
                $session->ended = time();
                $session->store();
            }

            $this->output::addResponse($request); //added so we can hook into this elsewhere.
            return System::asJson($response);
        }

        $this->output::addMessage('user.logout.validation', 'error', 'The user is not logged in');
        $this->output::addResponse($request, [], false); //added so we can hook into this elsewhere.
        return System::asJson($response, [], 403);
    }

    /**
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     * @throws \Exception
     */
    public function register(Request $request, Response $response, $args): Response
    {
        $body = $request->getParsedBody();
        $body = clone $body; // required so as not to interfere with $request->getParsedBody()
        $user = new userEntity();
        if ($id = $args['id'] ?? false) {
            //Permissions check
            if ($this->currentUser->id !== $id && !$this->currentUser->isAllowed('UpdateUsers')) {
                $this->output::addMessage($this->path->getRouteName($request).'.validation', 'error', 'The UpdateUsers permission is required to do this.');
                $this->output::addResponse($request, [], false);
                //added so we can hook into this elsewhere.
                return System::asJson($response, [], 403);
            }
            if ($this->currentUser->id === $id && (!$this->currentUser->isAllowed('UpdateUsers') && !$this->currentUser->isAllowed('UpdateSelf'))) {
                $this->output::addMessage($this->path->getRouteName($request).'.validation', 'error', 'The UpdateSelf permission is required to do this.');
                $this->output::addResponse($request, [], false);
                //added so we can hook into this elsewhere.
                return System::asJson($response, [], 403);
            }
            $user->load($id);
            //check we have all the data we need, make sure password is not required as this is an update.
            $user->setRequiredData(['email', 'first_name', 'last_name']);
        } else {
            if (!$this->currentUser->isAllowed('CanRegister') && !$this->currentUser->isAllowed('CreateUsers')) {
                $this->output::addMessage($this->path->getRouteName($request).'.validation', 'error', 'The CreateUsers permission is required to do this.');
                $this->output::addResponse($request, [], false); //added so we can hook into this elsewhere.
                return System::asJson($response, [], 403);
            }
            //does this user already exist?
            $user = new userEntity();
            $user->getUserByEmail($body->email);
            if (isset($user->id) && $user->id > 0) {
                $this->output::addMessage($this->path->getRouteName($request).'.validation', 'error', 'A user with that email address already exists.');
                $this->output::addResponse($request, [], false); //added so we can hook into this elsewhere.
                return System::asJson($response, [], 400);
            }
        }

        //check if passwords match, if we're providing them
        if ($body->password ?? false) {
            if ($body->password != $body->confirm_password) {
                $this->output::addMessage($this->path->getRouteName($request).'.validation', 'error', 'Passwords don\'t match');
                $this->output::addResponse($request, [], false); //added so we can hook into this elsewhere.
                return System::asJson($response, [], 400);
            }
            $passwordHash = '';
            $passwordSalt = '';
            if (isset($body->password) && !empty($body->password)) {
                $hash = new Hash();
                $hashData = $hash->hashString($body->password);
                $passwordHash = $hashData->hash;
                $passwordSalt = $hashData->salt;
                unset($body->password);
            }
            $user->password = $passwordHash;
            $user->salt = $passwordSalt;
        }
        //unset password
        if (isset($body->password)) {
            unset($body->password);
        }

        if (isset($body->confirm_password)) {
            //unset confirm_password so we dont try to store it.
            unset($body->confirm_password);
        }

        if (isset($body->active) && $this->currentUser->isAllowed('ChangeUserActiveState')) {
            $user->active = (int)$body->active;
            unset($body->active);
        }

        if (isset($body->suspended) && $this->currentUser->isAllowed('ChangeUserActiveState')) {
            $user->suspended = (int)$body->suspended;
            unset($body->suspended);
        }

        //we need to unset all roles if we're setting any at all, so check we can do this on first pass.
        if ($this->currentUser->isAllowed('CanSetUserRole')) {
            $user->removeRoles();
        }

        foreach ($body as $name => $value) {
            if ($this->currentUser->isAllowed('CanSetUserRole')) {
                if (strpos($name, 'role_') !== false) {
                    //Check we have permission to set user roles.
                    if ((int)$value !== 0) { //Ignore any FALSE/0/blank string values
                        $user->addRole(str_replace('role_', '', $name));
                    }
                } else {
                    $user->$name = $value;
                }
            } elseif (strpos($name, 'role_')) {
                //dont allow users without permission to set roles. Just ignore them and move on.
                continue;
            } else {
                $user->$name = $value;
            }
        }


        //Hook user pre save - allow other modules to act on the user save process, before storage.
        $user = Hook::processHook('userPreSave', $user, $request);

        if ($user->isValid()) {
            $user->store();

            // If new user and person creating is NOT an Admin
            if (!$id && class_exists('apexl\Io\modules\company\entities\companyEntity') && !$this->currentUser->isAllowed('administer')) {
                $user->getUserByEmail($body->email);
                $companyEntity = new companyEntity();
                $companyEntity->loadByUser($this->currentUser->id);
                $companyEntity->storeCompanyUser($companyEntity->id, $user->id);
                $user->roles = serialize($this->currentUser->roles); // Give new user same role as current user
                $user->store();
            }

            // Process user Contact Data
            if ($this->config->app->user->useContactDetails ?? null) {
                if ($user->id) {
                    $userContactDetailsEntity = new userContactDetailsEntity();
                    $userContactDetailsEntity->load($user->id);
                    foreach (['phone','shipping_address_1','shipping_address_2','shipping_address_3','shipping_town','shipping_county','shipping_postcode','shipping_country','billing_address_1','billing_address_2','billing_address_3','billing_town','billing_county','billing_postcode','billing_country'] as $field) {
                        if (isset($body->$field)) {
                            $userContactDetailsEntity->$field = trim($body->$field);
                        }
                    }
                    $userContactDetailsEntity->same_address = ($body->same_address ? 1 : 0);
                    if (!isset($userContactDetailsEntity->user_id) || !is_numeric($userContactDetailsEntity->user_id)) $userContactDetailsEntity->user_id = $user->id;
                    $userContactDetailsEntity->store();
                }
            }
            if (isset($body->company_name)) {
                $companyEntity = new companyEntity();
                $companyEntity->loadByUser($user->id);
                $companyEntity->CompanyName = trim($body->company_name);
                $companyEntity->store();
            }

            //Hook user post save - allow other modules to act on the user save process, after storage.
            $user = Hook::processHook('userPostSave', $user, $request);
            $this->output::addMetadata($this->path->getRouteName($request).'.complete', 'entityId', $user->id);
            $this->output::addMessage($this->path->getRouteName($request).'.complete', 'success', $user->getNiceName().' '. ($id ? 'updated' : 'created'));
            $this->output::addResponse($request, $user->getData()); //added so we can hook into this elsewhere.

            $routeName = 'userEntity.display.get';
            $path = Routes::getRoutePattern($routeName . '.override', ['id' => $user->id]);
            if ($route = ($path ? $path : Routes::getRoutePattern($routeName, ['id' => $user->id]))) {
                $this->output->addMetadata($this->path->getRouteName($request) . '.redirect', 'redirect', $route);
            }
            return System::asJson($response);
        }
        $this->output::addMessage($this->path->getRouteName($request).'.validation', 'error', 'Not all of the data provided is valid');
        $this->output::addResponse($request, ['missingFields' => $user->getMissingData()], false); //added so we can hook into this elsewhere.
        return System::asJson($response, [], 400);
    }

    public function forgotPassword(Request $request, Response $response)
    {
        $body = $request->getParsedBody();
        //first, grab the user data.
        $user = userEntity::getUserByEmail($body->email);

        //got a match? we need to generate a new random hash, then email it to the user.
        if(isset($user->email)){
            //we do this so the has can be passed back as a url param rather than a query string.
            $randomHash = str_replace('/', '@', (new Hash())->generateRandomHash());
            //write the hash to the database
            $user->newForgottenPasswordLink($randomHash);
            //now we send an email to the user.
            $this->forgottenPasswordEmail($user->email, $randomHash, $user->first_name);
            $this->output->addMetadata('user.forgot-password.redirect', 'redirect', Routes::getRoutePattern('user.display.forgot-password.check-email'));
        } else {
            $this->output::addMessage('user.forgot-password', 'error', 'User account does not exist');
        }
        $this->output::addResponse($request); //added so we can hook into this elsewhere.
        return System::asJson($response);
    }

    public function forgotPasswordWithLink(Request $request, Response $response, $args)
    {
        if (currentUser::getCurrentUser()->isLoggedIn()) {
            Output::addMetadata('user.login.redirect', 'redirect', '/');
            return System::asJson($response);
        }


        $hash = $args['hash'] ?? false;
        if ($hash) {
            $user = new userEntity();
            $match = $user->getForgottenPasswordLink($hash);
            if (!empty($match) && ((time() - $match['created']) <= 86400 && $match['used'] == 0)) {
                //log the user in and send them to their settings page to update the password. flag the hash as used.
                $user->markForgottenPasswordLinkUsed($hash);
                $user->getUserByEmail($match['email']);
                if ($user->id ?? false) {



                    $this->render::setActive("noUI");
                    $this->render::setMetaTitle("Forgotten Password | ". ($this->config->app->defaultMetaTitle ?? "Io Dashboard"));

                    $rowWrapper = (new RowWrapper())->addClass('d-block');
                    $rowWrapper->addComponent(
                        (new FormComponent())
                            ->src(Routes::getRoutePattern('user.display.forms.forgottenPassword'))
                            ->title("Enter your email to reset your password.")
                            ->addClass('forgotten-password-form')->setID('forgottenPassword')
                    );
                    $rowWrapper->addComponent(
                        (new BasicLink())->addClass('forgottenPasswordLink return mt-5 text-decoration-underline')->addRoute(Routes::getRoutePattern('user.display.login'))->addText(System::getVariable('user_login_return_login_text') ?? 'Return to login')
                    );

                    $this->output::addResponse($request, $this->render::build([$rowWrapper]));

                    return System::asJson($response);



                    //log the user in.
                    list($authToken, $refreshToken) = $this->userTools->startLoggedInSession($user);
                    $this->output::addResponse($request, ['authToken' => $authToken]); //added so we can hook into this elsewhere.
                    return System::asJson($response);
                }
            }
        }
        $this->output::addMessage($this->path->getRouteName($request).'.validation', 'error', "Invalid, unknown or expired hash");
        $this->output::addResponse($request, [], false); //added so we can hook into this elsewhere.
        return System::asJson($response, [], 400);
    }

    /**
     * @todo Replace mailer code with central mailer system
     * @param $email
     * @param $string
     * @param $first_name
     * @return void
     */
    protected function forgottenPasswordEmail($email, $string, $first_name){

        $from = System::getVariable('site_from_email') ?? $this->config->app->site->email_address ?? 'no-reply@localhost.com';
        $fromName = System::getVariable('site_from_name') ?? $this->config->app->site->name ?? 'localhost';
        $frontEndDomain = System::getVariable('site_from_domain') ?? $this->config->app->site->frontend_domain ?? 'localhost';

        //move to config.
        $link = rtrim($frontEndDomain, '/').'/'.ltrim('/'.Routes::getRoutePattern('user.display.reset-password', ['hash' => $string]), '/');

        $emailService = new emailService();
        $result = $emailService->send(
            (object) [
                'to' => $email,
                'subject' => 'Password Reset Request',
                'body' => templateService::fetch("password_reset", [
            'from_name' => $fromName,
            'reset_link' => $link,
            'name' => $first_name
                ]),
                'user_id' => 0,
            ],
            'generic'
        );

        Logger::log('password_reset_email', ['sent' => $result, 'email' => $email]);
    }

    public function userListTableData(Request $request, Response $response){
        $params = $request->getQueryParams();
        $entity = new userEntity();
        $filters = $this->buildFilterConditions($entity, $params);
        //$users = $entity->loadMultiple($filters, $params['orderBy'] ?? []); // Note - this inclueds access control with correct config

        $lookupKey = Hook::processHook('user_entity_search_lookup', null, $params);

        $entityData = $entity->loadByPage($params, $filters, $params['orderBy'] ?? [], $lookupKey);

        $entityData['tableHeader'] = ['ID', 'Name', 'Validated/ Active?', 'Email Address', 'Company', 'Last Login', 'Created', 'Roles'];

        $rows = [];
        foreach($entityData['data'] as $user){
            $user = (object)$user;

            $created = \DateTime::createFromFormat('d-m-Y H:i:s', $user->created);

            $row = [
                'id' => $user->id,
                'name' => trim($user->first_name . ' ' . $user->last_name),
                'validated' => $user->active,
                'email' => $user->email,
                'company' => $user->CompanyName,
                'lastLogin' => $user->last_login,
                'created' => $created->format('H:i - d M Y'),
                'Roles' => $user->roles,
            ];
            $rows[] = $row;
        }

        $entityData['rows'] = $rows;
        $entityData['totalData'] = count($rows);
        unset($entityData['data']);

        return System::asJson($response, $entityData);
    }

    public function buildFilterConditions($entity, $params, $validFields = []): array
    {
        $filters = parent::buildFilterConditions($entity, $params, ['customerName']);

        if(isset($filters['email'])) {
            $filters['email'][1] = '%' . $filters['email'][1] . '%';
            $filters['email'][2] = 'LIKE';
        }
        if(isset($filters['customerName'])) {
            $filters['customerName'] = ['CONCAT(first_name, " ", last_name)', '%'.$params['customerName'].'%', 'LIKE'];
        }

        return $filters;
    }

    public function resetPassword(Request $request, Response $response, $args)
    {
        $body = $request->getParsedBody();
        if ($body->password != $body->confirm_password) {
            $this->output::addMessage($this->path->getRouteName($request).'.validation', 'error', 'Passwords don\'t match');
            $this->output::addResponse($request, [], false); //added so we can hook into this elsewhere.
            return System::asJson($response, [], 400);
        }
        $isReset = false;

        $user = new userEntity();
        $match = $user->getForgottenPasswordLink($body->hash);
        if (!empty($match) && ((time() - $match['created']) <= 86400 && $match['used'] == 0)) {
            $user->markForgottenPasswordLinkUsed($body->hash);
            $user->getUserByEmail($match['email']);
            if (isset($user->id) && $user->id > 0 && isset($body->password) && !empty($body->password)) {

                $passwordHash = '';
                $passwordSalt = '';
                $hash = new Hash();
                $hashData = $hash->hashString($body->password);
                $passwordHash = $hashData->hash;
                $passwordSalt = $hashData->salt;
                unset($body->password);

                $user->password = $passwordHash;
                $user->salt = $passwordSalt;
                $user->store();

                try {
                    Logger::log('reset_remote_user_password', $body);
                    if (!apexlWPManageService::resetWPUserPassword($user->email, $body->confirm_password, $user->remote_user_id)) {
                        Logger::log('reset_remote_user_password_failed', $body);
                    }

                } catch (\Exception $e) {
                    Logger::log('reset_remote_user_password_error', $e->getMessage());
                }

                $this->output->addMetadata('user.login.validation.redirect', 'redirect', Routes::getRoutePattern('user.display.reset-password-success'));
                return System::asJson($response, ['success' => true]);
            }
        }

        $this->output::addMessage($this->path->getRouteName($request).'.validation', 'error', 'We were unable to reset your password');
        return System::asJson($response, [], 400);

    }

    public function updateSettings(Request $request, Response $response, $args)
    {
        $user = currentUser::getCurrentUser();
        if ($user->id ?? null) {
            $user->settings = $request->getParsedBody();
            $user->store();
            return System::asJson($response, ['success' => true]);
        }

        return System::asJson($response, ['success' => false]);
    }

    public function suspendUserNow(Request $request, Response $response, $args)
    {
        $userEntity = new userEntity();
        $userEntity->load($args['id']);
        if (isset($userEntity->id) && $userEntity->id > 0) {
            if (class_exists('apexl\Io\modules\subscription\entities\subscriptionEntity')) { // check to see if module is installed
                $subscriptionService = new subscriptionService();
                $subscriptionService->suspendAllUserSubscriptions($userEntity);
            }

        $userEntity->suspended = 1;
        $userEntity->active = 0;
        $userEntity->store();
        $this->output::addMessage('user.suspend', 'success', $userEntity->first_name . ' ' . $userEntity->last_name . ' access disabled and subscriptions pending imminent suspension.');

            return System::asJson($response, ['success' => true]);
        }

        return System::asJson($response, ['success' => false, 'reason' => 'User Not Found']);
    }

    public function validateUserNow(Request $request, Response $response, $args)
    {
        $userEntity = new userEntity();
        $userEntity->load($args['id']);
        if (isset($userEntity->id) && $userEntity->id > 0) {

            $userEntity->active = 1;
            $userEntity->store();
            $this->output::addMessage('user.validate', 'success', $userEntity->email . ' was successfully validated.');

            $this->output::addMetadata('user.validate','reloadPage', []);
            return System::asJson($response, ['success' => true]);
        }

        return System::asJson($response, ['success' => false, 'reason' => 'User Not Found']);
    }
}
