<?php

namespace apexl\Io\modules\userDisplay\controllers;

use apexl\Io\includes\Controller;
use apexl\Io\includes\Hook;
use apexl\Io\includes\Routes;
use apexl\Io\includes\System;
use apexl\Io\modules\display\components\ContentTitle;
use apexl\Io\modules\display\components\EntityFilteredDisplayTable;
use apexl\Io\modules\display\components\FormComponent;
use apexl\Io\modules\display\components\genericComponents;
use apexl\Io\modules\display\components\RowWrapper;
use apexl\Io\modules\display\components\SimpleButton;
use apexl\Io\modules\display\services\Render;
use apexl\Io\modules\formbuilder\entities\formEntity;
use apexl\Io\modules\formbuilder\includes\checkboxField;
use apexl\Io\modules\formbuilder\includes\inputField;
use apexl\Io\modules\userDisplay\callbacks\tableRowAlters;
use apexl\Io\modules\userDisplay\components\dashboardTiles;
use apexl\Io\modules\userDisplay\components\roleEntityDisplayTile;
use apexl\Io\modules\userDisplay\components\userEntityDisplayTile;
use apexl\Io\modules\userDisplay\forms\forgottenPasswordForm;
use apexl\Io\modules\userDisplay\forms\loginForm;
use apexl\Io\modules\user\entities\roleEntity;
use apexl\Io\modules\user\entities\userEntity;
use apexl\Io\modules\user\services\currentUser;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;

class userController extends Controller {

    protected $currentUser;
    protected $render;

    public function __construct(Render $render, currentUser $currentUser)
    {
        parent::__construct();
        $this->currentUser = $currentUser::getCurrentUser();
        $this->render = $render;
    }

    /**
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     */
    public function userViewSingle(Request $request, Response $response, $args){
        $rowWrapper = (new RowWrapper())->addClass('manageForm-wrapper');
        $rowWrapper->addComponent(dashboardTiles::userViewTile($args['id']));

        return System::asJson($response, $this->render::build([$rowWrapper]));
    }

    /**
     * @param Request $request
     * @param Response $response
     * @return Response
     */
    public function userSettingsPage(Request $request, Response $response){
        $form = (new FormComponent())
            ->src(Routes::getRoutePattern('user.display.forms.userSettings'))
            ->setID('register');
        return System::asJson($response, $this->render::build([$form]));
    }

    /**
     * @param Request $request
     * @param Response $response
     * @return Response
     * @throws \Exception
     */
    public function userSettingsForm(Request $request, Response $response){
        return $this->userEditForm($request, $response, ['id' => $this->currentUser->id]);
    }

    /**
     * Login page - @see userDisplayController in io-user
     * @param Request $request
     * @param Response $response
     * @return Response
     */
    public function loginPage(Request $request, Response $response){
        $this->render::setActive("noUI");
        $this->render::setMetaTitle("Login | ". ($this->config->app->defaultMetaTitle ?? "Io Dashboard"));
        return System::asJson($response, $this->render::build(
            [
                (new FormComponent())->src(Routes::getRoutePattern('user.display.forms.login'))
            ->title("Sign in with your email address.")
            ->addClass('login-form')->setID('login')
            ]));
    }

    /**
     * @param Request $request
     * @param Response $response
     * @return Response
     */
    public function loginForm(Request $request, Response $response){
        return System::asJson($response, (new loginForm())->loginForm());
    }

    /**
     * Forgotten Password Page
     * @param Request $request
     * @param Response $response
     * @return Response
     */
    public function forgotPasswordPage(Request $request, Response $response){
        $this->render::setActive("noUI");
        $this->render::setMetaTitle("Forgotten Password | ". ($this->config->app->defaultMetaTitle ?? "Io Dashboard"));
        $this->output::addResponse($request, $this->render::build([
            (new FormComponent())->src(Routes::getRoutePattern('user.display.forms.forgottenPassword'))
            ->title("Enter your email to reset your password.")
            ->addClass('forgotten-password-form')->setID('forgottenPassword')
        ]));
        return System::asJson($response);
    }

    /**
     * @param Request $request
     * @param Response $response
     * @return Response
     */
    public function forgotPasswordCheckEmail(Request $request, Response $response){
        $this->render::setActive("noUI");
        $this->render::setMetaTitle("Check Email | ". ($this->config->app->defaultMetaTitle ?? "Io Dashboard"));
        return System::asJson($response, $this->render::build([(new ContentTitle())
            ->addContent("Please check your email for further instructions.")
            ->addClass('login-form')->addClass('forgotten-password-form')
            ->setColWidth('md', 12)->setID('forgottenPassword'),
            (new SimpleButton())->addRoute(Routes::getRoutePattern('user.display.login'))->addText('Return to login')]));
    }

    /**
     * @param Request $request
     * @param Response $response
     * @return Response
     */
    public function forgotPasswordForm(Request $request, Response $response){
        return System::asJson($response, (new forgottenPasswordForm())->forgottenPasswordForm());
    }

    /**
     * @param Request $request
     * @param Response $response
     * @return Response
     */
    public function registerForm(Request $request, Response $response){
        $form = $this->primaryAccountForm(TRUE);
        return System::asJson($response, $form->getBuiltFormArray());
    }

    /**
     * @param Request $request
     * @param Response $response
     * @param array $args
     * @return Response
     * @throws \Exception
     */
    public function userEditForm(Request $request, Response $response, $args = []){
        $form = $this->primaryAccountForm();

        //set form field values
        $user = new userEntity();
        if (isset($args[$user->primaryKey])) {
            $user->load($args[$user->primaryKey]);
            $form->setMethod('PUT');
        }

        if($this->currentUser->isAllowed('ChangeUserActiveState')) {
            $activeField = (new checkboxField('active'))->setLabel('Active');
        } else {
            $activeField = (new inputField('active'))->setInputType('hidden');
        }

        if($id = $args[$user->primaryKey] ?? FALSE){
            $form->setActionUrl(Routes::getRoutePattern($user->getEntityName().'.put', ['id' => $id]));

            $form->addField($form->getFormField('first_name')->setValue($user->first_name));
            $form->addField($form->getFormField('last_name')->setValue($user->last_name));
            $form->addField($form->getFormField('email')->setValue($user->email));

            $activeField->setValue($user->active);
        }

        $form->addField($activeField);

        //allow other modules to add form fields.
        $form = Hook::processHook('displayFormRender', $form, $user, 'userEditForm');

        $roles = (new roleEntity())->loadMultiple();
        foreach ($roles as $role) {
            //for users that can't change roles, show labels for those that they do have.
            $showLabel = FALSE;
            if($this->currentUser->isAllowed('UpdateRoles') || $this->currentUser->isAllowed('AssignRole_'.$role->id)) {
                $roleField = (new checkboxField('role_' . $role->id))->setLabel($role->name);
            } else {
                $showLabel = TRUE;
                $roleField = (new inputField('role_' . $role->id));
                $roleField->setInputType('hidden');
            }
            if(($user->id ?? FALSE) && is_array($user->roles)) {
                if ($user && in_array($role->id, $user->roles)) {
                    //allows us to show the label only for those without the role to change.
                    //Only shows labels for the roles they have.
                    if ($showLabel) {
                        $roleField->setLabel($role->name);
                    }
                    $roleField->setValue($role->id);
                }
            }
            $form->addField($roleField, 'roles');
        }

        //set role values
        $this->output::addResponse($request, $form->getBuiltFormArray());
        return System::asJson($response);
    }

    /**
     * @param Request $request
     * @param Response $response
     * @return Response
     */
    public function userHome(Request $request, Response $response){

        $rowWrapper = (new RowWrapper())->setColWidth('sm', 12)->addClass('manageForm-wrapper');
        $rowWrapper->setColWidth('sm', 12)->addClass('manageForm-wrapper');
        $rowWrapper->addComponent(new userEntityDisplayTile());
        if($this->currentUser->isAllowed('ViewRoles')) {
            $rowWrapper->addComponent(new roleEntityDisplayTile());
        }

        $this->render::setPageTitle('User Dashboard');
        $this->output::addResponse($request, $this->render::build([$rowWrapper]));
        return System::asJson($response);
    }

    /**
     * @param Request $request
     * @param Response $response
     * @return Response
     * @throws \Exception
     */
    public function userList(Request $request, Response $response){

        $filterForm = new EntityFilteredDisplayTable((new userEntity()));
        $filterForm->addFilterForm('user.display.forms.filters.userFilter');

        //render
        $this->render::setPageTitle('User List');
        $this->output::addResponse($request, $this->render::build($filterForm));
        return System::asJson($response);
    }

    public function create(Request $request, Response $response, $args){
        $title = 'Create new user';
        if ($id = $args['id'] ?? FALSE) {
            $user = new userEntity();
            $user->load($id);
            $title = 'Update ' . $user->getNiceName();
        }

        $form = (new FormComponent())->setID('userCreate')->src(Routes::getRoutePattern('user.display.forms.createUpdate', $args));
        $this->render::setPageTitle($title);
        $this->output::addResponse($request, $this->render::build([genericComponents::dashboardBlockWrapper([$form])]));
        return System::asJson($response);
    }

    /**
     * @param Request $request
     * @param Response $response
     * @param array $args
     * @return Response
     * @throws \Exception
     */
    public function updateView(Request $request, Response $response, $args = []){
        $title = 'Create a user';
        if($id = $args['id'] ?? FALSE){
            $userEntity = new userEntity();
            $userEntity->load($id);
            $title = 'Edit ('.$id.') '. $userEntity->getNiceName();
        }

        $form = (new FormComponent())->setID('roleCreate')->src(Routes::getRoutePattern('user.display.forms.createUpdate', $args));
        $this->render::setPageTitle($title);
        $this->output::addResponse($request, $this->render::build($form));
        return System::asJson($response);
    }

    /**
     * @param bool $passwordRequired
     * @return formEntity
     */
    protected function primaryAccountForm($passwordRequired = FALSE): formEntity
    {
        $form = new formEntity();
        $form->setId('login');
        $form->setMethod('post');
        $form->setActionUrl(Routes::getRoutePattern('userEntity.post'));

        $form->addField((new inputField('first_name'))->setLabel('First Name')->isRequired());
        $form->addField((new inputField('last_name'))->setLabel('Surname')->isRequired());
        $form->addField((new inputField('email'))->setLabel('Email')->setInputType('email')->isRequired());

        $password = (new inputField('password'))->setLabel('Password')->setInputType('password')->addValidation("minLength", ["min" => 6]);
        $confirmPass = (new inputField('confirm_password'))->setLabel('Confirm Password')->setInputType('password')->addValidation("minLength", ["min" => 6]);
        if($passwordRequired) {
            $password->isRequired();
            $confirmPass->isRequired();
        }
        $form->addField($password);
        $form->addField($confirmPass);

        return $form;
    }

    /**
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     */
    public function userSummery(Request $request, Response $response, $args){
        $structure = [
            'tableHeader' => [
                'User ID',
                'Name',
                'Active',
                'Email',
                'Last Logged In',
                'Roles'
            ],
            'fields' => [
                'id',
                'first_name',
                'active',
                'email',
                'last_login',
                'roles'
            ],
            'callables' => [
                'field__last_login' => [tableRowAlters::class, 'userShortTable'],
                'field__first_name' => [tableRowAlters::class, 'userShortTable'],
                'field__roles' => [tableRowAlters::class, 'userShortTable']
            ]
        ];
        return $this->getEntityData($request, $response, $args, (new userEntity()), [], $structure);
    }
}