<?php

declare(strict_types=1);

namespace apexl\Io\modules\validation\Middleware;

use apexl\Io\modules\validation\Collections\ValidationRuleCollectionCollection;
use apexl\Io\modules\validation\Exceptions\ValidationException;
use apexl\Io\modules\validation\Factories\ValidatorFactory;
use apexl\Io\modules\validation\Interfaces\ValidatesDataInterface;
use apexl\Io\modules\validation\Services\ValidationRuleCollector;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\ServerRequestInterface;
use Psr\Http\Server\MiddlewareInterface;
use Psr\Http\Server\RequestHandlerInterface;

final readonly class ValidationMiddleware implements MiddlewareInterface
{
    public function __construct(
        private ValidatorFactory $validatorFactory,
        private ValidationRuleCollector $validationRuleCollector
    ) {}

    /**
     * @throws ValidationException
     */
    public function process(ServerRequestInterface $request, RequestHandlerInterface $handler): ResponseInterface
    {
        if ($request instanceof ValidatesDataInterface) {
            $rules = $this->rules($request);
            $data = $request->dataToValidate();
            $messages = $request->validationMessages();
            $lang = $request->validationLang();

            $validator = $this->validatorFactory->make($data, $rules, $messages, $lang);

            if (!$validator->isValid()) {
                throw new ValidationException($validator->getErrors());
            }
        }

        return $handler->handle($request);
    }

    private function rules(ValidatesDataInterface|ServerRequestInterface $request): array
    {
        $rules = $request->validationRules();

        if (is_array($rules)) {
            return $rules;
        }

        if ($rules instanceof ValidationRuleCollectionCollection) {
            return $rules->forValidator();
        }

        return $this->validationRuleCollector->collect($rules)->forValidator();
    }
}
