<?php

namespace apexl\Io\includes;

use apexl\Io\enums\HttpMethod;
use apexl\Io\services\HttpPaths;
use apexl\Io\utilityIncludes\Route;
use Psr\Http\Server\MiddlewareInterface;
use Slim\App;

final class RouteManager
{
    public function __construct(
        private readonly App $application,
        private readonly HttpPaths $httpPaths,
        private readonly HookManager $hookManager,
        protected string $baseRoute = '/',
        /** @var Route[] */
        public array $routes = [] {
            get {
                return $this->hookManager->processHook('routePreBuild', $this->routes);
            }
        },
    ) {}

    /**
     * @param MiddlewareInterface[] $middleware
     */
    public function post(string $name, string $pattern, $callable, array $middleware = []): void
    {
        $this->addRoute(HttpMethod::POST, $name, $pattern, $callable, $middleware);
    }

    public function addRoute(HttpMethod $method, $name, $pattern, $callable, $middleware = [], $args = []): void
    {
        $routes = $this->routes;
        $routes[$name] = new Route(
            $name,
            $pattern,
            $callable,
            $method,
            $args,
            $middleware,
        );
        $this->routes = $routes;
    }

    public function put(string $name, string $pattern, $callable, array $middleware = []): void
    {
        $this->addRoute(HttpMethod::PUT, $name, $pattern, $callable, $middleware);
    }

    public function delete(string $name, string $pattern, $callable, array $middleware = []): void
    {
        $this->addRoute(HttpMethod::DELETE, $name, $pattern, $callable, $middleware);
    }

    public function get(string $name, string $pattern, $callable, array $middleware = []): void
    {
        $this->addRoute(HttpMethod::GET, $name, $pattern, $callable, $middleware);
    }

    /**
     * Set a route object
     */
    public function setRoute(string $name, Route $route): void
    {
        $routes = $this->routes;
        $routes[$name] = $route;

        $this->routes = $routes;
    }

    /**
     * Function to build the set routes into a format Slim can understand.
     */
    public function buildRoutes(): void
    {
        foreach ($this->routes as $route) {
            /** @var \Slim\Routing\Route $slimRoute */
            $slimRoute = $this->application->{$route->getMethod()->forSlim()}(
                sprintf(
                    '%s/%s',
                    rtrim($this->baseRoute, '/'),
                    ltrim($route->getPattern(stripTags: false), '/'),
                ),
                $route->getCallable()
            );

            $slimRoute->setName($route->getName());

            foreach ($route->getArgs() as $argName => $argVal) {
                $slimRoute->setArgument($argName, $argVal);
            }

            foreach ($route->getMiddleware() as $middleware) {
                $slimRoute->addMiddleware($middleware);
            }
        }
    }

    /**
     * Add arg to already built route, store reference in routes static
     */
    public function addRouteArg(string $name, string $arg, mixed $value): void
    {
        $this->httpPaths->setCurrentRouteArgs($arg, $value);

        if ($route = $this->getRoute($name)) {
            $route->setArg($arg, $value);
        }
    }

    /**
     * Get a route object
     */
    public function getRoute($name): ?Route
    {
        return $this->routes[$name] ?? null;
    }

    public function getFullyQualifiedRoutePattern(
        string $name,
        array $replacements = [],
        bool $stripTags = true
    ): string|false {
        return $this->getRoute($name)?->getFullyQualifiedPattern($replacements, $stripTags);
    }

    /**
     * Get a route object
     */
    public function getRoutePattern(
        string $name,
        array $replacements = [],
        bool $stripTags = true
    ): ?string {
        return $this->getRoute($name)?->getPattern($replacements, $stripTags);
    }
}
