<?php

declare(strict_types=1);

namespace apexl\Io\modules\system\services;

use apexl\Io\includes\HookManager;
use apexl\Io\services\FilePaths;
use apexl\Io\WhoopsHandlers\JsonRecursionErrorHandler;
use DI\Container;
use Whoops\Handler\JsonResponseHandler;
use Whoops\Handler\PrettyPageHandler;
use function apexl\Io\config;

final readonly class WhoopsHandlerCollector
{
    final public const string HOOK__COLLECT = 'WhoopsHandlerCollector.collect';

    private const array EDITORS = [
        "io_sublime" => "subl://open?url=file://%file&line=%line",
        "io_textmate" => "txmt://open?url=file://%file&line=%line",
        "io_emacs" => "emacs://open?url=file://%file&line=%line",
        "io_macvim" => "mvim://open/?url=file://%file&line=%line",
        "io_phpstorm" => "phpstorm://open?file=%file&line=%line",
        "io_idea" => "idea://open?file=%file&line=%line",
        "io_vscode" => "vscode://file/%file:%line",
        "io_atom" => "atom://core/open/file?filename=%file&line=%line",
        "io_espresso" => "x-espresso://open?filepath=%file&lines=%line",
        "io_netbeans" => "netbeans://open/?f=%file:%line",
        "io_cursor" => "cursor://file/%file:%line",
    ];


    public function __construct(
        private Container $container,
        private FilePaths $filePaths,
        private HookManager $hookManager,
    ) {}

    public function collect(): array
    {

        return $this->hookManager->processHook(self::HOOK__COLLECT, [
            config('app.environment') === 'local' ?
                $this->prettyPageHandler() :
                $this->container->get(JsonResponseHandler::class),
            $this->container->get(JsonRecursionErrorHandler::class),
        ]);
    }

    private function prettyPageHandler(): PrettyPageHandler
    {
        $prettyPageHandler = $this->container->get(PrettyPageHandler::class);

        $editor = config('whoops.editor');
        if (!$editor) {
            return $prettyPageHandler;
        }


        if (isset(self::EDITORS[$editor])) {
            $editor = self::EDITORS[$editor];


            $prettyPageHandler->setEditor(function ($file, $line) use ($editor) {
                $translations = [];
                if ($localPath = config('whoops.local_base_path')) {
                    $translations[$this->filePaths->basePath()] = $localPath;
                }

                foreach ($translations as $from => $to) {
                    $from = preg_replace('#/+#', '/', $from . '/');
                    $to = preg_replace('#/+#', '/', $to . '/');
                    $file = preg_replace(
                        sprintf('#%s#', $from),
                        $to,
                        $file,
                        1
                    );
                }

                return str_replace(['%file', '%line'], [$file, $line], $editor);

            });

            return $prettyPageHandler;
        }


        $prettyPageHandler->setEditor($editor);

        return $prettyPageHandler;

    }


}