<?php

namespace apexl\Io\operators;

use apexl\entityCore\interfaces\EntityOperatorInterface;
use apexl\Io\includes\Hook;
use apexl\Vault\Vault;

class entityDatabaseOperator implements EntityOperatorInterface{
    /** @var Vault  */
    protected $vault;
    protected $dbTable;
    protected $primaryKey;
    protected $columns;

    protected $defaultOperator;
    protected $operatorVault;
    protected $accessControlType = 'default';

    /**
     * entityDatabaseOperator constructor.
     * @param $table
     * @param string $primaryKey
     * @param string $vault
     */
    public function __construct($table, $primaryKey = 'id', $vault = ''){
        $this->vault = Vault::getInstance();
        $this->dbTable = $table;
        $this->setPrimaryKey($primaryKey);
        $this->defaultOperator = $this->vault->getDefaultVault();
        $this->operatorVault = $vault;
    }

    public function setAccessControlType($type){
        $this->accessControlType = $type;
    }

    public function setOperatorVault($vaultName){
        $this->vault->setDefaultVault($vaultName);
        return $this;
    }

    public function useOperatorVault($reset = FALSE){
        if($this->operatorVault != ''){
            if($reset){
                $this->setOperatorVault($this->defaultOperator);
            } else {
                $this->setOperatorVault($this->operatorVault);
            }
        }
    }

    /**
     * Set the primary key field
     * @param $key
     * @return $this
     */
    public function setPrimaryKey($key){
        $this->primaryKey = $key;
        return $this;
    }

    /**
     * Query for loading entity data by id
     * @param $id
     * @param bool $skipAccessCheck
     * @return array
     */
    public function load($id, $skipAccessCheck = FALSE){
        $this->useOperatorVault();
        $query = $query = $this->genericQuery();
        $entity = $query->where($this->dbTable.'.'.$this->primaryKey, $id)->execute()->fetchAssoc();
        if(!$skipAccessCheck) {
            $entity = Hook::processHook('entityAccess', $entity, $this->dbTable);
        }

        $this->useOperatorVault(TRUE);
        return ['updateEntityData' => true, 'data' => $entity];
    }

    /**
     * Query to load all entity data
     * @param array $conditions
     * @param array $orderBy
     * @param false $limit
     * @param false $offset
     * @return mixed
     */
    public function loadMultiple($conditions = [], $orderBy = [], $limit = FALSE, $offset = FALSE){
        $this->useOperatorVault();
        $query = $this->genericQuery('entityLoadMultiple');
        // Do access control pass

        foreach($conditions as $condition){
            $operator = $condition[2] ?? '=';
            $type = $condition[3] ?? 'AND';
            $postCondition = $condition[4] ?? '';
            $query->where($condition[0], $condition[1], $operator, $type, $postCondition);
        }
        if(!empty($orderBy)){
            if(!is_array($orderBy)){
                //convert to an array
                $temp = $orderBy;
                $orderBy = [];
                $orderBy[0] = $temp;
            }
            $direction = $orderBy[1] ?? 'DESC';
            $query->orderBy($orderBy[0], $direction);
        }
        if($limit !== FALSE){
            $query->limit($limit, $offset);
        }

        $results = $query->execute()->fetchAll();
        $this->useOperatorVault(TRUE);
        return $results;
    }

    /**
     * Function to return the total number of entity records available
     * @param array $conditions
     * @return mixed
     */
    public function totalEntities($conditions = []){
        $this->useOperatorVault();
        $query = $this->vault->select($this->dbTable)->fields('count('.$this->dbTable.'.'.$this->primaryKey.') AS count');
        $query = Hook::processHook('accessControlJoins', $query, $this->accessControlType, $this->dbTable);
        $query = Hook::processHook('permissionRestrictViewEntities', $query, $this->dbTable, $this->getCurrentEntityName());
        $query = Hook::processHook('entityLoadMultipleQuery', $query, $this->dbTable);
        $query = Hook::processHook('accessControlClauses', $query, $this->accessControlType, $this->dbTable);
        if(!empty($conditions)){
            foreach($conditions as $condition) {
                $operator = $condition[2] ?? '=';
                $type = $condition[3] ?? 'AND';
                $postCondition = $condition[4] ?? '';
                $query->where($condition[0], $condition[1], $operator, $type, $postCondition);
            }
        }
        $result = $query->execute()->fetchAssoc();
        $this->useOperatorVault(TRUE);
        return $result['count'];
    }

    /**
     * Function to store entity data - Accounts for metadata storage on the entity
     * @param $data
     * @return array|bool
     * @throws \Exception
     */
    public function store($data){
        $this->useOperatorVault();
        $meta = NULL;
        //first, are we inserting or updating
        if(empty($data)){
            throw new \Exception("Cannot store empty entity data");
        }
        //store and strip metadata if its available. Allows entities to contain processing and maintain easy store methods.
        if(isset($data['_metadata'])){
            //we need to store this for update methods
            $meta = $data['_metadata'];
            unset($data['_metadata']);
        }
        //check for any extra fields that we don't have in the schema
        //make sure we know what the table supports
        $this->getTableColumns();
        $extraStore = [];
        foreach($data as $field => $value){
            if(!isset($this->columns[$field])){
                $extraStore[$field] = $value;
                //remove it from data for now.
                unset($data[$field]);
            }
        }

        $entity = [];
        if(isset($data[$this->primaryKey]) && !empty($data[$this->primaryKey])) {
            //we're storing a record, skip the access check to prevent data store issues.
            $entity = $this->load($data[$this->primaryKey], TRUE);
        }
        if (isset($entity['data']) && !empty($entity['data'])) {
            //not empty? we have a record, so update.
            $this->useOperatorVault();
            $this->vault->update($this->dbTable)->fields($data)->where($this->primaryKey, $data[$this->primaryKey])->execute();
            $this->useOperatorVault(TRUE);
            return TRUE;
        } else {
            $this->useOperatorVault();
            //we're inserting to force an entity update in case we have a new primary key.
            $this->vault->insert($this->dbTable)->fields($data)->execute();
            $lastStored = $this->getLastNewRecord(TRUE);
            $this->useOperatorVault(TRUE);
            //force the local entity to be updated with the new data (id etc)
            $updatedData = $this->load($lastStored[$this->primaryKey], TRUE)['data'];
            if($meta){
                $updatedData['_metadata'] = $meta;
            }
            //re-add any extra fields for use by other modules.
            if(!empty($extraStore)) {
                foreach ($extraStore as $field => $value) {
                    $data[$field] = $value;
                }
            }
            $this->useOperatorVault(TRUE);
            return ['updateEntityData' => true, 'data' => $updatedData];
        }
    }

    /**
     * Function to get the last created record for the given entity (by DB table)
     * @param bool $skipAccess
     * @return mixed
     */
    public function getLastNewRecord($skipAccess = FALSE){
        $this->useOperatorVault();
        $query = $this->genericQuery();
        $result = $query->orderBy($this->primaryKey, 'DESC')->execute()->fetchAssoc();
        if(!$skipAccess) {
            $result = Hook::processHook('entityAccess', $result, $this->dbTable);
        }
        $this->useOperatorVault(TRUE);
        return $result;
    }

    /**
     * Function to delete data by ID
     * @param $id
     * @return mixed
     */
    public function delete($id){
        $this->useOperatorVault();
        $deleted = $this->vault->delete($this->dbTable)->where($this->primaryKey, $id)->execute();
        $this->useOperatorVault(TRUE);
        return $deleted;
    }

    /**
     * Function to load all data available
     * @return mixed
     */
    public function loadAll(){
        $this->useOperatorVault();
        $query = $this->vault->select($this->dbTable);
        $result = $query->execute()->fetchAll();
        $result =  Hook::processHook('entityAccess', $result, $this->dbTable);
        $this->useOperatorVault(TRUE);
        return $result;
    }

    /**
     * Describe method to generate known  table columns for validation and safe data storage
     * @return mixed
     */
    public function getTableColumns(){
        $this->useOperatorVault();
        if(!$this->columns){
            $columns = $this->describe();
            //build the data column.
            foreach($columns as $col){
                $this->columns[$col->Field] = $col;
            }
        }
        $this->useOperatorVault(TRUE);
        return $this->columns;
    }

    protected function describe(){
        return $this->vault->describe($this->dbTable)->execute()->fetchAll();
    }

    protected function genericQuery($hook = 'entityLoad'){
        //clone here to prevent access checks interfering.
        $query = clone $this->vault->select($this->dbTable);
        $queryFields = Hook::processHook($hook.'Fields', [$this->dbTable => ['*']], $this->dbTable);
        $fields = [];
        foreach($queryFields as $table => $queryField){
            foreach($queryField as $tableColumn){
                $fields[] = strpos($tableColumn, '.') === FALSE ? $table.'.'.$tableColumn: $tableColumn;
            }
        }
        $query->fields($fields);
        $query = Hook::processHook('accessControlJoins', $query, $this->accessControlType, $this->dbTable);
        $query = Hook::processHook('permissionRestrictViewEntities', $query, $this->dbTable, $this->getCurrentEntityName());
        $query = Hook::processHook($hook.'Query', $query, $this->dbTable);
        $query = Hook::processHook('accessControlClauses', $query, $this->accessControlType, $this->dbTable);

        return $query;
    }

    protected function getCurrentEntityName()
    {
        $instantiatedClass = get_called_class();
        return str_ireplace("operator","", substr($instantiatedClass, strrpos($instantiatedClass, '\\') + 1));
    }
}