<?php

namespace apexl\Io\modules\coachingForms\controllers;

use apexl\Io\includes\Controller;
use apexl\Io\includes\System;
use apexl\Io\modules\coachingForms\entities\categoriesEntity;
use apexl\Io\modules\coachingForms\entities\coachingAnswersEntity;
use apexl\Io\modules\coachingForms\entities\coachingFormsEntity;
use apexl\Io\modules\coachingForms\entities\packagesEntity;
use apexl\Io\modules\coachingForms\entities\questionsEntity;
use apexl\Io\modules\user\entities\userEntity;
use apexl\Io\modules\user\services\currentUser;
use apexl\Io\services\Mailer;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;

/**
 * Controller to handle all form submissions and data.
 * Class coachingRecordsController
 * @package apexl\Io\modules\coachingForms\controllers
 */
class coachingRecordsController extends Controller
{

    protected $currentUser;

    public function __construct(currentUser $currentUser)
    {
        parent::__construct();
        $this->currentUser = $currentUser::getCurrentUser();
    }

    /**
     * Store a new coaching record. This will be for the active form.
     * @route PUT|POST /api/data/coaching-records[/{id}]
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     * @throws \Exception
     */
    public function storeNewCoachingRecord(Request $request, Response $response, $args){

        $body = $request->getParsedBody();

        $answers = new coachingAnswersEntity();
        if($id = $args['id'] ?? FALSE){
            $answers->load($id);
        }
        //basic validation
        if(!isset($body->staff_member) || empty($body->staff_member)){
            return System::asJson($response, System::output(['type' => 'failer', 'content' => 'Staff Member Missing'], "",'DoNothing'), 400);
        }

        $staffEntity = new userEntity();
        $staffEntity->load($body->staff_member);

        $answers->form_id = $body->form_id;
        $answers->transactionId = $body->transaction_id;
        $answers->team_leader_id = $this->currentUser->id; //Always store the user that did the review, not always the users tl.
        $answers->staff_id = $body->staff_member;
        $answers->agreed_actions = $body->agreed_actions ?? '';
        $answers->key_dev_areas = $body->key_dev_areas ?? '';
        $answers->key_strengths = $body->key_strengths ?? '';
        $answers->state = $body->state ?? 0;
        $answers->completed_date = time();
        $answers->store();
        //process the to level fields that dont change

        $selectedPackages = [];
        foreach ($body as $name => $fields){
            //we only take one of the question params so we dont duplicate the data,
            if(strpos($name, 'question_radios_') !== FALSE){
                $questionId = str_replace(['question_comments_', 'question_radios_'], '', $name);
                $radioValue = $body->{'question_radios_'.$questionId} ?? 3; //assume N/A if no value
                $commentValue = $body->{'question_comments_'.$questionId} ?? ''; //assume blank if no value
                $answers->storeQuestionAnswer($questionId, $answers->id, $radioValue, $commentValue);
            }

            if(strpos($name, 'package_') !== FALSE){
                $packageId = str_replace('package_', '', $name);
                $selectedPackages[$packageId] = $packageId;
            }
        }

        $answers->chosen_packages = serialize($selectedPackages);
        //store again with the selected packages (This is an update, not a create)
        $answers->store();

        //check state, if we're set to reviewed then send an email to the staff member with the report.
        if($answers->state === 2){
            $this->emailReport($answers->id);
        }

        return System::asJson($response, System::output(['type' => 'success', 'content' => 'Form Submitted'], "/reports/view-report/".$answers->id));
    }

    public function emailReport($id){
        //we reload so that we can make sure we have all information up to date.
        //build variables so we can pass them to a template file for emails.
        $completeForm = new coachingAnswersEntity();
        $completeForm->load($id);

        $staff = new userEntity();
        $staff->load($completeForm->staff_id);

        $manager = new userEntity();
        $manager->load($completeForm->team_leader_id);

        $emailFields = [];
        $emailFields['staff_name'] = $staff->first_name.' '.$staff->last_name;
        $emailFields['completed_date'] = date('d-m-Y H:i', $completeForm->completed_date);
        $emailFields['manager_name'] = $manager->first_name.' '.$manager->last_name;
        $emailFields['key_strengths'] = $completeForm->key_strengths;
        $emailFields['key_dev_areas'] = $completeForm->key_dev_areas;
        $emailFields['agreed_actions'] = $completeForm->agreed_actions;
        $emailFields['categories'] = [];

        //$packages = (new packagesEntity())->getFormPackages($completeForm->form_id);
        foreach($completeForm->categories as $category) {
            $categoryEmailFields = [];
            $categoryEmailFields['name'] = $category->name;
            $categoryEmailFields['score'] = $category->catScores->scoreText;
            $categoryEmailFields['questions'] = [];
            //grab category questions, and build the form row.
            foreach($category->questions as $question){
                $questionFields = [];
                $questionFields['name'] = $question->name;
                $questionFields['answer'] = questionsEntity::demonstratedToReadable($question->demonstrated_raw);
                $questionFields['comments'] = $question->comment;
                $questionFields['score'] = $question->score;
                $categoryEmailFields['questions'][] = $questionFields;
            }
            $emailFields['categories'][] = $categoryEmailFields;
        }

        $frontEndDomain = $this->config->app->site->frontend_domain ?? 'localhost';

        //move to config.
        $link = rtrim($frontEndDomain, '/').'/reports/view-report/'.$id;

        //build basic HTML
        $html = '<p>Hi '.$emailFields['staff_name'].'</p><br/>';
        $html .= '<p>Your report, completed by '.$emailFields['manager_name'].' on '.$emailFields['completed_date'].' is ready. The full report can be seen <a href="'.$link.'">here</a>, please find a summary below.</p><br/><br/>';

        foreach($emailFields['categories'] as $category){
            $html .= '<h3>'.$category['name'].' | Score: '.$category['score'].'</h3><br/><br/>';
            $html .= '<table>';
            foreach($category['questions'] as $question){
                $html .= '<tr><td>'.$question['name'].'</td><td>'.$question['answer'].'</td></tr>';
            }
            $html .= '</table><br/></br>';
        }

        $html .= '<h4>Key Strengths</h4><br/><p>'.$emailFields['key_strengths'].'</p></br></br>';
        $html .= '<h4>Key Development Areas</h4><br/><p>'.$emailFields['key_dev_areas'].'</p></br></br>';
        $html .= '<h4>Agreed Actions</h4><br/><p>'.$emailFields['agreed_actions'].'</p></br></br>';
        $html .= '<p><span>Thanks,</span><br><span>The Sky Coaching Team</span></p>';

        $from = $this->config->app->site->email_address ?? 'no-reply@localhost.com';
        $fromName = $this->config->app->site->name ?? 'localhost';

        $mailer = new Mailer();
        //send Email
        $mailer->setFrom($from, $fromName);
        $mailer->addAddress($staff->email);     // Add a recipient

        $mailer->Subject = 'Coaching Record Review :- '.$emailFields['completed_date'];
        $mailer->Body = $html;
        $mailer->IsHTML(true);
        $mailer->send();
    }

    /**
     * Get all available coaching records
     * @route GET /api/data/coaching-records
     * @param $request
     * @param $response
     * @param $args
     * @return Response
     */
    public function getRecords($request, $response, $args){
        $params = $request->getQueryParams();
        if(isset($params['from'])){
            $params['completed_date'] = ['completed_date', strtotime($params['from'].' 00:00:00'), '>='];
        }
        if(isset($params['to'])){
            $params['completed_date_to'] = ['completed_date', strtotime($params['to'].' 23:59:59'), '<='];
        }
        if(isset($params['new_starter'])){
            $params['users.created'] = ['users.created', strtotime('6 weeks ago'), '>='];
        }
        return $this->getEntityData($request, $response, $args, (new coachingAnswersEntity()), $params);
    }
}