<?php

namespace apexl\Io\modules\coachingForms\controllers;

use apexl\Io\includes\Controller;
use apexl\Io\includes\System;
use apexl\Io\modules\coachingForms\entities\categoriesEntity;
use apexl\Io\modules\coachingForms\entities\coachingFormsEntity;
use apexl\Io\modules\coachingForms\entities\questionsEntity;
use apexl\Io\modules\user\services\currentUser;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;

/**
 * Controller to handle all form administration
 * Class manageCoachingFormsController
 * @package apexl\Io\modules\coachingForms\controllers
 */
class manageCoachingFormsController extends Controller
{
    protected $currentUser;

    /**
     * manageCoachingFormsController constructor.
     * @param currentUser $currentUser
     */
    public function __construct(currentUser $currentUser)
    {
        parent::__construct();
        $this->currentUser = $currentUser::getCurrentUser();
    }

    /**
     * Get all available forms.
     * @route GET /api/data/coaching-forms[/{id}]
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     */
    public function getForms(Request $request, Response $response, $args){
        return $this->getEntityData($request, $response, $args, (new coachingFormsEntity()));
    }

    /**
     * Create/Update a coaching form.
     * @route POST|PUT /api/data/coaching-forms[/{id}]
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     * @throws \Exception
     */
    public function storeForm(Request $request, Response $response, $args){
        $body = $request->getParsedBody();

        //validate data.
        if(!isset($body->form_name) || empty($body->form_name ?? FALSE)){
            //return error.
            return System::asJson($response, System::output(['type' => 'error', 'content' => 'Form Name Field required'], "", false, 'doNothing'));
        }

        if(!isset($body->SkyCoachingForms) || empty($body->SkyCoachingForms ?? FALSE)){
            //return error.
            return System::asJson($response, System::output(['type' => 'error', 'content' => 'At least one category required.'], "", false, 'doNothing'));
        }

        //new form
        $coachingForm = new coachingFormsEntity();
        if($id = $args['id'] ?? FALSE){
            $coachingForm->load($id);
            $currentCategories = [];
            //if we're editing, get all know categories. and build an array of all current category ids.
            $savedCats = $coachingForm->getCategories();
            foreach($savedCats as $savedCat){
                $currentCategories[$savedCat->id] = $savedCat->id;
            }
        }

        $coachingForm->name = $body->form_name;

        //if we're setting this form to active, turn off any the other forms.
        if($body->active ?? FALSE){
            $coachingForm->turnOffForms();
            $coachingForm->active = $body->active;
            //Store the new active state.
            $coachingForm->store();
        } else {
            $coachingForm->active = 0;
            //store the form so we have an ID for the next sections.
            $coachingForm->store();
        }

        //loop over the repeating field and set data for storage.
        foreach($body->SkyCoachingForms as $category) {
            $categoryEntity = new categoriesEntity();
            if(isset($category->category_id)){
                //we still have the package so unset it, if its set.
                if(isset($currentCategories[$category->category_id])) {
                    unset($currentCategories[$category->category_id]);
                }
                $categoryEntity->load($category->category_id);
            }
            $categoryEntity->name = $category->name;
            $categoryEntity->parentId = $coachingForm->id;
            //Store before we continue so we have an ID to reference.
            $categoryEntity->store();
            //next, store the questions.
            if(isset($category->questions) && !empty($category->questions)) {
                foreach ($category->questions as $qid => $question) {
                    $questionEntity = new questionsEntity();
                    $questionEntity->load($qid); //try a load. if this returns nothing then we'll just create a new question.
                    $questionEntity->parentId = $categoryEntity->id;
                    $questionEntity->parentType = 'form_categories';
                    $questionEntity->name = $question;
                    $questionEntity->required = 0;
                    $questionEntity->score = is_array($category->scores) ? $category->scores[$qid] : $category->scores->$qid; //Store the question score - depending on scenario this could be object or array
                    $questionEntity->store();
                }
            }
        }

        //loop over any remaining packages and delete them.
        if(!empty($currentCategories)) {
            foreach ($currentCategories as $oldCatId) {
                $categoryEntity->delete($oldCatId);
            }
        }

        //store selected packages. We loop over body at this point so its easy to collect all selected packages.
        foreach ($body as $name => $fields){
            //only process packages
            if(strpos($name, 'package_') !== FALSE){
                //could use regex or strpos/strstr but replace is easier.
                $id = str_replace('package_', '', $name);
                //check we dont already have this package.
                $coachingForm->addPackage($id);
            }
        }

        return System::asJson($response, System::output(['type' => 'success', 'content' => $coachingForm->name.' created'], "/coaching/manage"));
    }

    /**
     * Delete a Coaching form.
     * @route DELETE /api/data/coaching-forms/{id}/delete
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     * @throws \Exception
     */
    public function deleteForm(Request $request, Response $response, $args){
        if($id = $args['id'] ?? FALSE) {
            $coachingForm = new coachingFormsEntity();
            $coachingForm->load($id);
            $coachingForm->deleted = 1;
            $coachingForm->store();
            return System::asJson($response, System::output(['type' => 'success', 'content' => 'Coaching form '. $coachingForm->name.' '. 'deleted'], "/coaching/manage"));
        }
        return System::asJson($response, System::output(['type' => 'fail', 'content' => 'Coaching form not found.'], "/coaching/manage"), 404);
    }

}