<?php

namespace apexl\Io\modules\coachingFormsDisplay\controllers;

use apexl\Io\includes\System;
use apexl\Io\modules\coachingForms\entities\categoriesEntity;
use apexl\Io\modules\coachingForms\entities\coachingFormsEntity;
use apexl\Io\modules\coachingForms\entities\packagesEntity;
use apexl\Io\modules\coachingFormsDisplay\components\content;
use apexl\Io\modules\coachingFormsDisplay\components\dashboardTiles;
use apexl\Io\modules\coachingFormsDisplay\components\dataTables;
use apexl\Io\modules\coachingFormsDisplay\components\forms;
use apexl\Io\modules\display\components\genericComponents;
use apexl\Io\modules\display\components\genericForms;
use apexl\Io\modules\formbuilder\entities\formEntity;
use apexl\Io\modules\formbuilder\includes\checkboxField;
use apexl\Io\modules\formbuilder\includes\inputField;
use apexl\Io\modules\formbuilder\includes\markupField;
use apexl\Io\modules\formbuilder\includes\repeatingFieldGroupField;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;

class manageCoachingFormsDisplayController extends coachingDisplayFormsController {

    /**
     * @route GET /coaching
     * @param Request $request
     * @param Response $response
     * @return Response
     */
    public function coachingParent(Request $request, Response $response){
        //should no child links be available, redirect to reports.
        $type = '';
        if($this->currentUser->isAllowed('ViewOwnReports')){
            $type = 'individual';
        }
        if($this->currentUser->isAllowed('ViewTeamReports')){
            $type = 'team';
        }
        if($this->currentUser->isAllowed('ViewCompanyReports')){
            $type = 'company';
        }
        if(empty($type)){
            return System::asJson($response, System::output([]), 403);
        }
        return System::asJson($response, System::output([], '/reports/'.$type.'/evaluations'));
    }

    /**
     * Manage Forms Dashboard page.
     * @route GET /coaching/manage
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     */
    public function manageCoachingForms(Request $request, Response $response, $args){
        $rowWrapper = genericComponents::rowWrapper('manageForm-wrapper col-sm-12');
        $rowWrapper->addComponent(dashboardTiles::manageCoachingFormList());
        $rowWrapper->addComponent(dashboardTiles::managePackageFormList());

        return System::asJson($response, $this->genericPage->wrapper([$rowWrapper]), 200);
    }

    /**
     * View all created coaching forms.
     * @route GET /coaching/view-all-forms
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     */
    public function viewAllForms(Request $request, Response $response, $args){
        $listTitle = genericComponents::ContentTitle('Created Coaching Forms', 'A list of all available coaching forms, Active form first.');
        $formTable = dataTables::manageCoachingFormsList();
        return System::asJson($response, $this->genericPage->wrapper([$listTitle, $formTable]));
    }

    /**
     * Create form page wrapper.
     * @route GET /coaching/manage/create-form
     * @route GET /coaching/manage/form/{id}/edit
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     */
    public function createCoachingForms(Request $request, Response $response, $args){
        return System::asJson($response, $this->genericPage->wrapper([
            content::coachingFormsCreateNewTitle($args),
            forms::createUpdateCoaching($args),
        ]));
    }

    /**
     * Display form to allow the creation of coaching forms.
     * @route GET /api/display/forms/coaching/manageForm
     * @route GET /api/display/forms/coaching/manageForm/{id}/edit
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     */
    public function manageCreateForm(Request $request, Response $response, $args){

        $coachingForm = null;
        if($id = $args['id'] ?? FALSE) {
            $coachingForm = new coachingFormsEntity();
            $coachingForm->load($args['id']);
        }

        $form = new formEntity();
        $form->setId('createCoachingForm');
        $form->setMethod('post');
        $form->setActionUrl('/api/data/coaching-forms'.($id ? '/'.$id : ''));

        //add sky section field
        $formName = (new inputField('form_name'))->setPlaceholder('Enter form name');
        //Repeating field for categories and questions.
        $repeating = (new repeatingFieldGroupField('SkyCoachingForms'))->setGroupPlaceholder('Category name');
        //markup description field.
        $packageDescription = new markupField('package_description', '<h3>Select Packages</h3><p>Select the packages to make available for this form.</p>');
        //active checkbox description markup field
        $activeDescription = new markupField('active_description', '<h3>Active status</h3><p>Please note that this will change the coaching form and available packages for Teams to fill in.</p>');
        //active option checkbox
        $activeOption = (new checkboxField('active'))->setLabel('Make this the active form?');

        //Are we editing? Build our values.
        if($coachingForm){
            $formName->setValue($coachingForm->name);
            //get all our questions and categories
            $categoriesEntity = new categoriesEntity();
            $repeatingValue = $this->buildQuestionSet($categoriesEntity->getFormCategorties($coachingForm->id),'category', $id);
            //build repeating field values, if we're editing.
            $repeating->setValue($repeatingValue);

            //get any selected packages for this form
            $selectedPackages = [];
            $selected = $coachingForm->getPackages();
            //build a searchable array
            foreach($selected as $selectedList){
                $selectedPackages[$selectedList->packageId] = $selectedList;
            }

            //are we the active form?
            $activeOption->setValue($coachingForm->active);
        }
        $form->addField($formName);
        $form->addField($repeating);
        $form->addField($packageDescription);

        //add checkboxes for all available packages
        $packageEntity = new packagesEntity();
        $availablePackages = $packageEntity->getActive();

        foreach($availablePackages as $availablePackage){
            $packageField = new checkboxField('package_'.$availablePackage->id);
            $packageField->setLabel($availablePackage->name);
            $packageField->classes = 'package-checkbox col-md-3 col-sm-6';
            if($coachingForm && isset($selectedPackages[$availablePackage->id])) {
                $packageField->setValue($availablePackage->id);
            }
            $form->addFieldToGroup($packageField, 'packages');
        }

        $form->addField($activeDescription);
        $form->addField($activeOption);

        return System::asJson($response, $form->getBuiltFormArray(), 200);
    }

    /**
     * Function to generate the Coaching Form Delete form.
     * GET /coaching/manage/forms/{id}/delete
     * @param Request $request
     * @param Response $response
     * @param array $args
     * @return Response
     */
    public function manageFormsDeleteView(Request $request, Response $response, $args = []){
        if($id = $args['id'] ?? FALSE) {
            $coachingForm = new coachingFormsEntity();
            $coachingForm->load($args['id']);

            //don't boolean check this, we want id 0 to fail (anon) so it can't be deleted.
            if($coachingForm->id) {
                $form = genericComponents::formTile('forms', '/api/display/forms/coaching/forms/'.$args['id'].'/delete', 'col-md-12');
                $form->addProperty('header', 3);
                $form->addProperty('title', "Are you sure you want to delete the ".$coachingForm->name." form?");
                return System::asJson($response, $this->genericPage->wrapper([$form], 'withBackground'), 200);
            }
            return System::asJson($response, [],404);
        }
        return System::asJson($response, [],400);
    }

    /**
     * function to delete coaching forms. Should only be used if a form has no records
     * @route GET /api/display/forms/coaching/forms/{id}/delete
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     */
    public function formsDeleteForm(Request $request, Response $response, $args){
        $form = genericForms::entityActionButtonForm('/api/data/coaching-forms/'.$args['id'].'/delete');
        return System::asJson($response, $form->getBuiltFormArray());
    }
}