<?php

namespace apexl\Io\modules\coachingFormsDisplay\controllers;

use apexl\Io\includes\System;
use apexl\Io\modules\coachingForms\entities\categoriesEntity;
use apexl\Io\modules\coachingForms\entities\coachingAnswersEntity;
use apexl\Io\modules\coachingForms\entities\coachingFormsEntity;
use apexl\Io\modules\coachingForms\entities\packagesEntity;
use apexl\Io\modules\coachingForms\entities\questionsEntity;
use apexl\Io\modules\coachingFormsDisplay\components\content;
use apexl\Io\modules\coachingFormsDisplay\components\dashboardTiles;
use apexl\Io\modules\coachingFormsDisplay\components\dataTables;
use apexl\Io\modules\coachingFormsDisplay\components\forms;
use apexl\Io\modules\component\entities\componentEntity;
use apexl\Io\modules\display\components\genericComponents;
use apexl\Io\modules\formbuilder\entities\formEntity;
use apexl\Io\modules\formbuilder\includes\checkboxField;
use apexl\Io\modules\formbuilder\includes\inputField;
use apexl\Io\modules\formbuilder\includes\markupField;
use apexl\Io\modules\formbuilder\includes\selectField;
use apexl\Io\modules\formbuilder\includes\textareaField;
use apexl\Io\modules\user\entities\userEntity;
use apexl\Io\modules\user\services\currentUser;
use apexl\Io\services\pathUtility;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Http\Message\ServerRequestInterface as Request;

class manageCoachingRecordsDisplayController extends coachingDisplayFormsController {

    /**
     * Function to return the main Add new records page.
     * @route GET /coaching/add-new
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     */
    public function addNewCoachingRecord(Request $request, Response $response, $args){
        //Into text
        $subtitle = new componentEntity();
        $subtitle->name = 'ContentTitle';
        $subtitle->addProperty('title', 'Create new');
        $subtitle->addProperty('content', 'This document is designed to provide consistent levels of support for all of our field sales team. It aims to measure key elements of interaction expected to see when a BDE speaks to a prospect and provides specific development areas to encourage a high performing team.');

        return System::asJson($response, $this->genericPage->wrapper([$subtitle, forms::addNewCoachingForm($args)]), 200);
    }

    /**
     * Function to return a paged list of all coaching form submissions.
     * @route GET /reports
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     */
    public function viewAllCoachingRecords(Request $request, Response $response, $args){
        $type = $args['type'] ?? 'company';
        //set the type based on user permissions
        //check our permissions and build our dynamic responses.
        $currentUser = currentUser::getCurrentUser();
        switch($type){
            case 'company':
                if(!$this->currentUser->isAllowed('ViewCompanyReports')){
                    return System::asJson($response, System::output([

                    ], '/reports/team'));
                }
                $title = 'Company reports';
                break;
            case 'team':
                if(!$this->currentUser->isAllowed('ViewTeamReports')){
                    return System::asJson($response, System::output([
                    ], '/reports/individual'));
                }
                $title = 'My team reports';
                break;
            case 'individual':
                if(!$this->currentUser->isAllowed('ViewOwnReports')){
                    return System::asJson($response, System::output([
                    ], '/'));
                }
                $title = 'My Reports';
                break;
            default:
                //no matches? 404.
                return System::asJson($response, System::output([
                    'type' => 'fail',
                    'content' => '404 - not found.'
                ], '/'), 404);
                break;
        }

        $listTitle = genericComponents::ContentTitle('Reporting', $title);
        $filterForm = genericComponents::formTile('reportsFilter', '/api/display/callbacks/reports-filter', 'col-md-12');
        $filterForm->addProperty('buildSubmitOnly', true);

        //enforce params based on display.
        $path = pathUtility::getInstance();
        $url = $path->getPath();

        $filters = '';
        switch ($url){
            case 'reports/team':
            case 'reports/team/components':
                //if sales manager
                if($currentUser->isAllowed('ViewSalesTeamCoachingStaff')){
                    $filters = '?sales_manager_id='.$currentUser->id;
                } else {
                    $filters = '?team_leader_id='.$currentUser->id;
                }
                break;
            case 'reports/individual':
            case 'reports/individual/components':
                //limit to own.
                $filters = '?staff_id='.$currentUser->id;
                break;
            default:
                break;
        }

        return System::asJson($response, $this->genericPage->wrapper([], '', [$listTitle, $filterForm], "/reports/".$type."/components".$filters));
    }

    public function viewAllEvaluations(Request $request, Response $response, $args){
        $listTitle = genericComponents::ContentTitle('Reporting', 'View all evaluation reports');

        $filterForm = genericComponents::formTile('reportsFilter', '/api/display/callbacks/reports-filter', 'col-md-12');
        $filterForm->addProperty('buildSubmitOnly', true);

        $params = $request->getQueryParams();
        $args = $this->buildEvaluationArgs($params);

        //convert args to string for now @todo - do this differently.
        $cleanArgs = [];
        if(!empty($args)){
            foreach ($args as $field => $options){
                $cleanArgs[] = $options[0].'='.$options[1];
            }
        }
        $args = implode(', ', $cleanArgs);
        $formTable = dataTables::manageCoachingRecordsList('?orderBy[]=id&orderBy[]=DESC&'.$args);
        $formTable->addProperty('basePath', '/reports/view-report');

        return System::asJson($response, $this->genericPage->wrapper([$listTitle, $formTable], '', [$filterForm]));
    }

    public function reportComponents(Request $request, Response $response, $args){
        $type = $args['type'];
        //get the main display tiles
        $params = $request->getQueryParams();
        $args = $this->buildEvaluationArgs($params, $type);

        $evaluation = dashboardTiles::dashboardCompletedEvaluations($args);
        $breakdown = dashboardTiles::dashboardBreakdownbycoach($args);
        return System::asJson($response, ["components" => [
            json_decode((string) $evaluation), json_decode((string) $breakdown)
        ]]);
    }

    /**
     * Function to return the view display for any given coaching form
     * @route GET /coaching/{id}
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     */
    public function viewCoachingRecord(Request $request, Response $response, $args){
        $rowWrapper = genericComponents::rowWrapper('evaluation-wrapper col-sm-12');
        $rowWrapper->addComponent(content::coachingRecordDisplay($args['id']));
        return System::asJson($response, $this->genericPage->wrapper([$rowWrapper]));
    }

    /**
     * Function to build and return the current active coaching form.
     * @route /
     * @param Request $request
     * @param Response $response
     * @param $args
     * @return Response
     * @throws \Exception
     */
    public function coachingRecordForm(Request $request, Response $response, $args){

        $filledCoachingForm = NULL;
        if($id = $args['id'] ?? FALSE){
            $filledCoachingForm = new coachingAnswersEntity();
            $filledCoachingForm->load($args['id']);
        }

        $form = new formEntity();
        $form->setId("coachingEvaluations");
        $form->setActionUrl('/api/data/coaching-records'.($id ? '/'.$id : ''));
        $form->removeSubmitButton();

        $state = (new inputField('state'))->setInputType('hidden')->setValue(($filledCoachingForm->state ?? 0));
        $transactionId = (new inputField('transaction_id'))->setLabel('Transaction ID')->isRequired();

        //use markup fields for labels here so we can more easily control the display.
        $staffSelectLabel = (new markupField('staff_member_label', 'Staff Member:'))->addClass('col-md-2 col-sm-6 bmd-form-group bmd-form-group__label');

        //First get the current user's available staff.
        //If we're updating a form, dont allow the staff member to be changed.
        if(!$id){
            $staffMembers = (new userEntity())->loadMultiple([], ['last_name', 'ASC']);
            $options = [];
            foreach($staffMembers as $staff){
                $options[$staff->id] = $staff->last_name .', '. $staff->first_name;
            }
            $selectField = (new selectField('staff_member', $options))->addClass('col-md-4 col-sm-6')->isRequired();
        } else {
            $userEntity = (new userEntity())->load($filledCoachingForm->staff_id);
            $selectField = (new markupField('staff_member_selected', $userEntity->last_name .','. $userEntity->first_name))->addClass('col-md-3');
            $selectHidden = (new inputField('staff_member',  'hidden'))->setValue($userEntity->id);
            $form->addField($selectHidden);
            $transactionId->setValue($filledCoachingForm->transactionId);
        }

        $coachSelectLabel = (new markupField('coach_member_label', 'Manager:'))->addClass('col-md-2 col-sm-6 bmd-form-group bmd-form-group__label');
        $staffSelectText = (new markupField('coach_name', '<i class="fal fa-fw fa-user-circle"></i>'.$this->currentUser->last_name.', '.$this->currentUser->first_name))->addClass('col-md-4 col-sm-6 bmd-form-group');

        $form->addField($state);
        $form->addField($transactionId);

        $form->addFieldToGroup($staffSelectLabel,'staff_select_group');
        $form->addFieldToGroup($selectField,'staff_select_group');
        $form->addFieldToGroup($coachSelectLabel,'staff_select_group');
        $form->addFieldToGroup($staffSelectText,'staff_select_group');

        //Add in the form key description
        $formKey = new markupField(
            'coach_name',
            '<ul class="key-list">
<li><span class="key-title red">(R) Red</span><span class="key-text"></span> = Demonstrates minimal or no competencies</li>
<li><span class="key-title amber">(A) Amber</span><span class="key-text"></span> = Demonstrates some competencies</li>
<li><span class="key-title green">(G) Green</span><span class="key-text"></span> = Demonstrates all competencies</li>
</ul>');

        $form->addField($formKey);

        //Title markup fields
        $form->addField(new markupField('form_title_1', '<div class="form-sub-title"><strong>Behaviours</strong></div>'));
        $form->addField(new markupField('form_title_2', '<div class="row fieldset-sub-title"><div class="col-md-9"><strong>Category</strong></div><div class="col-md-3"><strong>Status</strong></div></div>'));

        //Next, load the active form -- NOTE for simplicity this is split into multiple loads per section.
        // Could probably do the same with a single query and some joins but the implementation would be more complicated
        $coachingForm = (new coachingFormsEntity())->getActiveForm();
        $categories = (new categoriesEntity())->getFormCategorties($coachingForm['id']);
        $packages = (new packagesEntity())->getFormPackages($coachingForm['id']);

        //Build the fields
        $formId = (new inputField('form_id', 'hidden'))->setValue($coachingForm['id']);
        $form->addField($formId);

        foreach($categories as $category){
            //grab category questions, and build the form row.
            $questions = (new questionsEntity())->getCategoryQuestions($category->id);
            //add the "table" headers
            $headerFields = $this->categoryHeaderFields($category->id);
            foreach($headerFields as $headerField) {
                $form->addFieldToGroup($headerField, 'category_header_group_' . $category->id, $category->name);
            }

            //Set the fieldset config.
            $form->setFieldsetSetting($category->name, 'fieldset_legend', $category->name);
            $form->setFieldsetSetting($category->name, 'fieldset_classes', 'row fieldset-category-'.$category->id);
            $form->setFieldsetSetting($category->name, 'is_collapsable', true);
            $form->setFieldsetSetting($category->name, 'is_collapsed', true);
            $form->setFieldsetSetting($category->name, 'collapsed_link', true);
            $form->setFieldsetSetting($category->name, 'collapsed_lb_text', '[Edit]');
            $form->setFieldsetSetting($category->name, 'overview', true);
            $form->setFieldsetSetting($category->name, 'overview_callback', '');

            foreach($questions as $question){
                $questionRow = $this->questionRowFields($question, $filledCoachingForm);
                foreach($questionRow as $questionField){
                    $form->addFieldToGroup($questionField,'question_group_'.$question->id, $category->name);
                }
            }
        }

        //add package fields
        $packageFields = [];
        $chosenPackages = [];
        $callbackRoute = '/api/display/callbacks/package/';
        if($filledCoachingForm){
            $chosenPackages = unserialize($filledCoachingForm->chosen_packages);
        }
        foreach($packages as $availablePackage){
            $packageEntity = new packagesEntity();
            $packageEntity->load($availablePackage->packageId);
            if(isset($packageEntity->id)) {
                $packageField = (new checkboxField('package_' . $packageEntity->id))->setLabel($packageEntity->name)->addClass('package-checkbox col-md-3 col-sm-6');
                $packageField->setCallback($callbackRoute.$packageEntity->id);
                if (isset($chosenPackages[$packageEntity->id])) {
                    $packageField->setValue($packageEntity->id);
                }
                $form->addFieldToGroup($packageField, 'packages');

                //create a blank field for easy callback targeting, the field type is not important here as the callback provides a new form object.
                //need to know which package fields are checked.
                $packageFields[] = new inputField('fieldset_package_' . $packageEntity->id . '_target', 'holding');
            }
        }
        //next we need target objects to add package questions to,
        // to add blanks to the form array for easy finding
        // We add them here to ensure they appear below the package options.
        foreach ($packageFields as $packageHolder){
            $form->addField($packageHolder);
        }

        //add the next fields
        $keyStrengths = (new textareaField('key_strengths'))->setLabel('Key Strengths')->enableWysiwyg();
        $keyDevAreas = (new textareaField('key_dev_areas'))->setLabel('Key Development Areas')->enableWysiwyg();
        $agreedActions = (new textareaField('agreed_actions'))->setLabel('Agreed Actions')->enableWysiwyg();

        if($filledCoachingForm){
            $keyStrengths->setValue($filledCoachingForm->key_strengths);
            $keyDevAreas->setValue($filledCoachingForm->key_dev_areas);
            $agreedActions->setValue($filledCoachingForm->agreed_actions);
        }
        $form->addField($keyStrengths);
        $form->addField($keyDevAreas);
        $form->addField($agreedActions);
        //return the form.
        return System::asJson($response, $form->getBuiltFormArray());
    }

    /**
     * @param $params
     * @param string $type
     * @return array
     */
    protected function buildEvaluationArgs($params, $type = ''){
        $args = [];
        foreach($params as $field => $param){
            switch ($field){
                case 'from':
                    $args['from'] = [
                        "complete_form.completed_date",
                        strtotime($param.' 00:00:00'),
                        '>='
                    ];
                    break;
                case 'to':
                    $args['to'] = [
                        "complete_form.completed_date",
                        strtotime($param.' 23:59:59'),
                        '<='
                    ];
                    break;
                case 'team_leader_id':
                    $args['team_leader_id'] = [
                        "complete_form.team_leader_id",
                        $param,
                        '='
                    ];
                    break;
                case 'staff_id':
                    $args['staff_id'] = [
                        "complete_form.staff_id",
                        $param,
                        '='
                    ];
                    break;
                case 'new_starter':
                    $args['new_starter'] = [
                        "users.created",
                        strtotime('24 hours ago'),
                        '>='
                    ];
                    break;
                case 'sales_manager_id':
                    $args['sales_manager_id'] = [
                        "( ctt.team_leader_id",
                        $param,
                        '='
                    ];
                    $args['sales_manager_id_null'] = [
                        "ctt.team_leader_id",
                        'null',
                        'IS NULL',
                        'OR',
                        ')'
                    ];
                    break;
            }
        }
        $currentUser = currentUser::getCurrentUser();
        if(!empty($type)){
            switch($type){
                case 'individual':
                    $args['staff_id'] = [
                        "complete_form.staff_id",
                        $currentUser->id,
                        '='
                    ];
                    break;
                case 'team':
                    if(!$currentUser->isAllowed('ViewAllCoachingStaff') && !$currentUser->isAllowed('ViewSalesTeamCoachingStaff')){
                        $args['team_leader_id'] = [
                            "complete_form.team_leader_id",
                            $currentUser->id,
                            '='
                        ];
                    }
                    break;
                case 'company':
                    break;
            }
        }
        return $args;
    }
}