<?php

namespace apexl\Vault\operators;

use apexl\Vault\exceptions\ExecutionException;
use PDO as PdoConnection;
use PDOStatement;
use SebastianBergmann\Diff\Exception;
use stdClass;
use Throwable;

use apexl\Vault\interfaces\operators;

abstract class pdo implements operators
{
    protected string $database;
    protected string $host;
    protected string $user;
    protected string $pass;
    protected string $schema;

    protected string $sql = '';
    /* @var string[] */
    protected array $queryFields = [];
    protected string $table;

    protected stdClass $connectionDetails;
    protected array $options;
    protected PDOStatement $statement;
    protected int $loggedInUserId;
    public array $statementError;

    protected ?PdoConnection $connection;

    public function __construct(stdClass $connectionDetails)
    {

        $this->setHost($connectionDetails->dbhost ?? '')
            ->setDatabase($connectionDetails->dbname ?? '')
            ->setUser($connectionDetails->dbuser ?? '')
            ->setPass($connectionDetails->dbpass ?? '')
            ->setSchema($connectionDetails->driver ?? '')
            ->setOptions($connectionDetails->options ?? []);
    }

    public function __destruct()
    {
        $this->connection = null;
    }

    public function setHost(string $host): pdo
    {
        $this->host = $host;

        return $this;
    }

    /**
     * Function to set the name of the database we're connecting to
     */
    public function setDatabase(string $name = ''): pdo
    {
        $this->database = $name;

        return $this;
    }

    /**
     * Function to set the username to connect with

     */
    public function setUser(string $user = ''): pdo
    {
        $this->user = $user;

        return $this;
    }

    public function setPass(string $pass = ''): pdo
    {
        $this->pass = $pass;

        return $this;
    }

    public function setSchema(string $schema = ''): pdo
    {
        $this->schema = $schema;

        return $this;
    }

    public function setOptions(array $options = []): pdo
    {
        $this->options = $options;

        return $this;
    }

    public function getDatabase(): string
    {
        return $this->database;
    }

    public function getUser(): string
    {
        return $this->user;
    }

    public function getPass(): string
    {
        return $this->pass;
    }

    public function getTable(): ?string
    {
        return $this->table ?? null;
    }

    private function makeConnection(): PdoConnection
    {
        $dsn = sprintf('%s:host=%s', $this->schema, $this->host);

        //allow us to connect to a mysql server, but without a database if one isn't specified.
        if ($this->database) {
            $dsn = sprintf('%s;dbname=%s', $dsn, $this->database);
        }

        return new PdoConnection($dsn, $this->user, $this->pass, $this->options);
    }

    protected function connection(): pdo
    {
        $this->connection = $this->makeConnection();

        return $this;
    }

    public function testConnection(): bool
    {
        try {
            $this->makeConnection();

            return true;
        } catch (Throwable $throwable) {
            return false;
        }
    }

    /**
     * @throws ExecutionException
     */
    public function execute(): pdo
    {
        //Allow us to connect as late as possible.
        if (empty($this->connection)) {
            $this->connection();
        }

        $prepare = $this->connection->prepare($this->sql);

        if (!$prepare) {
            $this->statementError = $this->connection->errorInfo();

            return $this;
        }

        $this->statement = $prepare;
        logger('query')->debug($this->sql, [
            'params' => $this->queryFields,
        ]);

        try {
            $this->statement->execute($this->queryFields);

        } catch (Throwable $e) {
            $e->getMessage();
        }
        $this->statementError = $this->statement->errorInfo();


        if ($this->statementError[0] !== '00000') {
            logger('query')->error($this->sql, [
                'params' => $this->queryFields,
                'error' => $this->statementError,
            ]);

            throw new ExecutionException(implode(', ', $this->statementError));
        }

        return $this;
    }

    // This should be used only with migrations,  otherwise for safety execute should be used

    /**
     * @throws ExecutionException
     */
    public function executeRaw(string $sql): array
    {
        $this->sql = $sql;
        $this->queryFields = [];
        $this->execute();

        logger('query')->debug($this->sql, [
            'params' => $this->queryFields,
            'error' => $this->statementError,
        ]);

        if ($this->statementError[0] == '00000') {
            return [true, null];
        return $this->connection(true);
        }

        return [false, implode(', ', $this->statementError)];
    }

    /**
     * @return array|false
     */
    public function fetchAll()
    {
        return $this->statement->fetchAll(PdoConnection::FETCH_OBJ);
    }

    /**
     * @return array|false
     */
    public function fetchAssoc()
    {
        return $this->statement->fetch(PdoConnection::FETCH_ASSOC);
    }

    public function setLoggedInUserId(int $userId)
    {
        $this->loggedInUserId = $userId;

        return true;
    }

    public function isCount(): bool
    {
        return stripos((string) $this->sql, "count(") !== false;
    }
}
