<?php

declare(strict_types=1);

namespace apexl\ConvertKitHelper;

use apexl\ConvertKit;
use n1ghteyes\apicore\structure\response;
use stdClass;

class ConvertKitHelper
{
    private const META__SUBSCRIBER_ID = 'convert-kit-subscriber-id';

    private static function api(): ConvertKit
    {
        $api = new ConvertKit();
        $api->setApiSecret(defined('CONVERT_KIT__API_SECRET') ? CONVERT_KIT__API_SECRET : '');

        return $api;
    }

    public static function allTags(): array
    {
        $api = self::api();

        return $api->GET->tags()->data->tags;
    }

    /**
     * @throws NoConvertKitIdForUserException
     */
    public static function getSubscriberIdForUser(int $wpUserId): string
    {

        $user = get_userdata($wpUserId);
        if ($subscriberId = self::fetchSubscriberIdByEmail($user->user_email)) {
            error_log(sprintf('Saved new subscriber ID data for user %d', $wpUserId), E_USER_NOTICE);
            update_user_meta($wpUserId, self::META__SUBSCRIBER_ID, $subscriberId);

            return $subscriberId;
        }

        return '';
    }

    public static function tagsForUser(int $wpUserId): array
    {
        $api = self::api();

        try {
            $subscriberId = self::getSubscriberIdForUser($wpUserId);
        } catch (NoConvertKitIdForUserException $exception) {
            trigger_error(
                $exception->getMessage(),
                E_USER_WARNING
            );

            return [];
        }

        return $api->GET->subscribers->{$subscriberId}->tags()->data->tags ?? [];
    }

    public static function interests(): array
    {
        $tags = self::allTags();

        $interests = array_filter($tags, [self::class, 'isInterest']);

        return self::mapInterestsAsNameToId($interests);
    }

    public static function updateUserData(string $subscriberId, array $convertKitUserData)
    {
        $api = self::api();

        $api->PUT->subscribers->{$subscriberId}($convertKitUserData);
    }

    public static function removeTagFromSubscriber(string $subscriberId, string $tagId): void
    {
        $api = self::api();

        $api->DELETE->subscribers->{$subscriberId}->tags->{$tagId}();
    }

    /**
     * @throws NoConvertKitIdForUserException
     */
    public static function updateInterestsForUser(int $wpUserId, $submittedInterests)
    {
        $subscriberId = self::getSubscriberIdForUser($wpUserId);
        $currentInterests = self::interestsForUser($wpUserId);
        foreach ($currentInterests as $interest => $tagId) {
            if (!in_array($interest, $submittedInterests)) {
                self::removeTagFromSubscriber($subscriberId, (string) $tagId);
            }
        }

        $user = get_userdata($wpUserId);
        foreach ($submittedInterests as $interest) {
            if (!in_array($interest, $currentInterests)) {
                self::subscribeUserToTag(
                    $user->user_email,
                    self::tag(sprintf('interest_%s', $interest))
                );
            }
        }
    }

    public static function updateMachinesForUser(int $wpUserId, $submittedMachines)
    {
        $subscriberId = self::getSubscriberIdForUser($wpUserId);
        $currentMachines = self::machinesForUser($wpUserId);
        foreach ($currentMachines as $machine => $tagId) {
            if (!in_array($machine, $submittedMachines)) {
                self::removeTagFromSubscriber($subscriberId, (string) $tagId);
            }
        }

        $user = get_userdata($wpUserId);

        foreach ($submittedMachines as $machine) {
            error_log('machine: ' . $machine);
            if (!in_array($machine, $currentMachines)) {
                self::subscribeUserToTag(
                    $user->user_email,
                    self::tag(sprintf('machine_%s', $machine))
                );
            }
        }
    }

    private static function isInterest(stdClass $tag): bool
    {
        return str_starts_with($tag->name, 'interest_');
    }

    private static function isMachine(stdClass $tag): bool
    {
        return str_starts_with($tag->name, 'machine_');
    }

    private static function mapInterestsAsNameToId(array $interests): array
    {
        $output = [];

        foreach ($interests as $interest) {
            $output[substr($interest->name, strlen('interest_'))] = $interest->id;
        }

        return $output;
    }

    private static function mapMachinesAsNameToId(array $machines): array
    {
        $output = [];

        foreach ($machines as $machine) {
            $output[substr($machine->name, strlen('machine_'))] = $machine->id;
        }

        return $output;
    }

    public static function interestsForUser(int $wpUserId): array
    {
        $tags = self::tagsForUser($wpUserId);

        $interests = array_filter($tags, [self::class, 'isInterest']);

        return self::mapInterestsAsNameToId($interests);
    }

    public static function machinesForUser(int $wpUserId): array
    {
        $tags = self::tagsForUser($wpUserId);

        $machines = array_filter($tags, [self::class, 'isMachine']);

        return self::mapMachinesAsNameToId($machines);
    }

    public static function subscribe(string $formId, string $name, string $email, array $tags = []): response
    {
        $api = self::api();

        return $api->POST->forms->{$formId}->subscribe([
            'first_name' => $name,
            'email' => $email,
            'tags' => $tags,
        ]);
    }

    public static function userHasTag(int $wpUserId, string $tagName): bool
    {
        $tags = self::tagsForUser($wpUserId);

        foreach ($tags as $tag) {
            if ($tag->name === $tagName) {
                return true;
            }
        }

        return false;
    }

    public static function tag(string $tagName): ?int
    {
        foreach (self::allTags() as $tag) {
            if ($tag->name === $tagName) {
                return $tag->id;
            }
        }

        return null;
    }

    public static function subscribeUserToTag(string $email, int $tag)
    {
        $api = self::api();

        $api->POST->tags->{$tag}->subscribe([
            'email' => $email,
        ]);
    }

    public static function machineTagForUser(int $wpUserId): ?string
    {
        foreach (self::tagsForUser($wpUserId) as $tag) {
            if (str_starts_with($tag->name, 'cutting_machine_')) {
                return substr($tag->name, strlen('cutting_machine_'));
            }
        }

        return null;
    }

    private static function fetchSubscriberIdByEmail(string $user_email): ?string
    {
        $api = self::api();

        /** @var response $response */
        $response = $api->GET->subscribers(['email_address' => $user_email]);

        if ($response->data->total_subscribers === 1) {
            return (string) $response->data->subscribers[0]->id;
        }

        return null;
    }

    // unsubscribe user
    public static function unsubscribeUser(string $user_email)
    {
        $api = self::api();
        $api->PUT->unsubscribe([
            'email' => $user_email,
        ]);

        //delete user meta
        $user = get_user_by('email', $user_email);
        delete_user_meta($user->ID, self::META__SUBSCRIBER_ID);
    }
}
